package com.digiwin.mobile.mobileuibot.config;

import com.digiwin.mobile.mobileuibot.config.request.CommonAppInterceptor;
import com.digiwin.mobile.mobileuibot.config.request.RequestContextInterceptor;
import com.digiwin.mobile.mobileuibot.config.request.TokenInterceptor;
import com.digiwin.mobile.mobileuibot.config.request.log.PerformanceLogInterceptor;
import com.digiwin.mobile.mobileuibot.config.request.log.RequestResponseLogInterceptor;
import com.digiwin.mobile.mobileuibot.config.request.mock.MockInterceptor;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.ResourceHandlerRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurationSupport;

import java.nio.charset.StandardCharsets;
import java.util.List;

/**
 * <p>功能描述：后端工程请求拦截器配置类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: InterceptorConfig
 * @Author: Zaregoto
 * @Date: 2021/5/27 10:14
 */
@Configuration
public class InterceptorConfig extends WebMvcConfigurationSupport {

    @Autowired
    private RequestResponseLogInterceptor requestResponseLogInterceptor;

    @Autowired
    private PerformanceLogInterceptor performanceLogInterceptor;

    @Autowired
    private RequestContextInterceptor requestContextInterceptor;

    @Autowired
    private CommonAppInterceptor commonAppInterceptor;

    @Autowired
    private ObjectMapper objectMapper;

    @Autowired
    private MockInterceptor mockInterceptor;

    @Autowired
    private TokenInterceptor tokenInterceptor;

    /**
     * 静态资源访问
     *
     * @param registry
     * @author yanfeng
     */
    @Override
    public void addResourceHandlers(ResourceHandlerRegistry registry) {
        registry.addResourceHandler("/**").addResourceLocations("classpath:/META-INF/resources/")
                .addResourceLocations("classpath:/resources/").addResourceLocations("classpath:/static/")
                .addResourceLocations("classpath:/public/");
        super.addResourceHandlers(registry);
    }


    @Override
    protected void addInterceptors(InterceptorRegistry registry) {
        //排除此路径，其他都拦截
        //        registry.addInterceptor(baseInterceptor).excludePathPatterns("/base/test");
        //指定路径，只拦截此路径
        //        registry.addInterceptor(baseInterceptor).addPathPatterns("/base/test");
        registry.addInterceptor(requestResponseLogInterceptor);
        registry.addInterceptor(performanceLogInterceptor);
        registry.addInterceptor(commonAppInterceptor);
        registry.addInterceptor(requestContextInterceptor);
        registry.addInterceptor(mockInterceptor);
        registry.addInterceptor(tokenInterceptor).addPathPatterns(TokenInterceptor.PATH_PATTERN)
                .excludePathPatterns(TokenInterceptor.EXCLUDE_PATH_PATTERN);
        //如果两个同时存在，排除生效，指定的路径会拦截两次
        //添加拦截器
        super.addInterceptors(registry);
    }

    @Override
    protected void extendMessageConverters(List<HttpMessageConverter<?>> converters) {
        /**
         * 继承这个类后，无论是改配置文件，自定义ObjectMapper，还是自己定义MappingJackson2HttpMessageConverter,
         * jackson的配置均失效了。
         * 解决办法就是重写本方法。
         *
         * 参考链接：https://www.huangchaoyu.com/2020/02/16/springboot%E9%85%8D%E7%BD%AEMappingJackson2HttpMessageConverter%E6%9C%80%E4%BD%B3%E5%AE%9E%E8%B7%B5/
         */

        //将StringHttpMessageConverter设成UTF8格式
        converters.stream()
                .filter(c -> c instanceof StringHttpMessageConverter)
                .map(c -> (StringHttpMessageConverter) c)
                .forEach(c -> c.setDefaultCharset(StandardCharsets.UTF_8));

        //将MappingJackson2HttpMessageConverter设为自己的ObjectMapper
        converters.stream()
                .filter(c -> c instanceof MappingJackson2HttpMessageConverter)
                .map(c -> (MappingJackson2HttpMessageConverter) c)
                .forEach(c -> c.setObjectMapper(this.objectMapper));
    }
}
