package com.digiwin.mobile.mobileuibot.config;


import com.digiwin.mobile.mobileuibot.common.converter.DateToTimestampConverter;
import com.digiwin.mobile.mobileuibot.common.converter.TimestampToDateConverter;
import org.springframework.boot.autoconfigure.mongo.MongoProperties;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.core.convert.converter.Converter;
import org.springframework.data.mongodb.MongoDatabaseFactory;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.SimpleMongoClientDatabaseFactory;
import org.springframework.data.mongodb.core.convert.MappingMongoConverter;
import org.springframework.data.mongodb.core.convert.MongoCustomConversions;

import java.util.ArrayList;
import java.util.List;


/**
 * mongoDB 多数据源配置
 *
 * @author yanfeng
 * @date 2024/3/15
 */
@Configuration
public class MultipleMongoConfig {

    /**
     * 第一数据源配置信息
     *
     * @return
     */
    @Bean
    @Primary
    @ConfigurationProperties(prefix = "spring.data.mongodb.primary")
    public MongoProperties primaryMongoProperties() {
        return new MongoProperties();
    }

    /**
     * 第二数据源配置信息
     *
     * @return
     */
    @Bean("secondMongoProperties")
    @ConfigurationProperties(prefix = "spring.data.mongodb.second")
    public MongoProperties secondMongoProperties() {
        return new MongoProperties();
    }

    /**
     * 默认 MongoTemplate
     * 使用方式：
     *
     * @return
     * @Autowired private MongoTemplate mongoTemplate;
     */
    @Bean
    @Primary
    public MongoTemplate mongoTemplate() {
        MongoTemplate mongoTemplate = new MongoTemplate(primaryMongoFactory(primaryMongoProperties()));
        MappingMongoConverter mongoMapping = (MappingMongoConverter) mongoTemplate.getConverter();
        mongoMapping.afterPropertiesSet();
        return mongoTemplate;
    }

    /**
     * 使用方式：
     *
     * @return
     * @Autowired
     * @Qualifier("secondMongoTemplate") private MongoTemplate secondMongoTemplate;
     */
    @Bean(name = "secondMongoTemplate")
    public MongoTemplate secondMongoTemplate() {
        MongoProperties mongoProperties = secondMongoProperties();
        MongoDatabaseFactory mongoDatabaseFactory = secondMongoFactory(mongoProperties);
        MongoTemplate mongoTemplate = new MongoTemplate(mongoDatabaseFactory);
        MappingMongoConverter mongoMapping = (MappingMongoConverter) mongoTemplate.getConverter();
        mongoMapping.afterPropertiesSet();
        return mongoTemplate;
    }

    /**
     * 默认 MongoDatabaseFactory
     *
     * @param mongoProperties
     * @return
     */
    @Bean
    @Primary
    public MongoDatabaseFactory primaryMongoFactory(MongoProperties mongoProperties) {
        return new SimpleMongoClientDatabaseFactory(mongoProperties.getUri());
    }

    @Bean
    public MongoDatabaseFactory secondMongoFactory(MongoProperties mongoProperties) {
        return new SimpleMongoClientDatabaseFactory(mongoProperties.getUri());
    }

    @Bean
    public MongoCustomConversions customConversions() {
        List<Converter<?, ?>> converters = new ArrayList<>();
        converters.add(new DateToTimestampConverter());
        converters.add(new TimestampToDateConverter());
        return new MongoCustomConversions(converters);
    }

}

