package com.digiwin.mobile.mobileuibot.config;

import com.digiwin.mobile.mobileuibot.common.http.DigiwinHttpReqMsgInterceptor;
import com.digiwin.mobile.mobileuibot.common.http.DigiwinHttpReqTimeInterceptor;
import com.digiwin.mobile.mobileuibot.common.http.DigiwinHttpResponseErrorHandler;
import com.digiwin.mobile.mobileuibot.common.http.constant.TimeOutEnum;
import org.apache.http.HeaderElement;
import org.apache.http.HeaderElementIterator;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicHeaderElementIterator;
import org.apache.http.protocol.HTTP;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.DependsOn;
import org.springframework.context.annotation.Primary;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.client.OkHttp3ClientHttpRequestFactory;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.util.NumberUtils;
import org.springframework.web.client.RestTemplate;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.SocketAddress;
import java.net.URI;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.Properties;

/**
 * <p>功能描述：配置系统统一的RestTemplate</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: RestTemplateConfig
 * @Author: Zaregoto
 * @Date: 2021/4/21 1:32
 */
@DependsOn("baseConfig")
@Configuration
public class RestTemplateConfig {

    @Bean
    public DigiwinHttpReqMsgInterceptor digiwinHttpReqMsgInterceptor() {
        return new DigiwinHttpReqMsgInterceptor();
    }

    @Bean
    @Primary
    public RestTemplate restTemplate(RestTemplateBuilder builder) {
        RestTemplateBuilder restTemplateBuilder = builder
                .setConnectTimeout(this.getConnectTimeoutDuration())
                .setReadTimeout(this.getReadTimeoutDuration());
        return this.restTemplateBuilder(restTemplateBuilder);
    }

    @Bean
    public RestTemplate asaRestTemplate (ClientHttpRequestFactory factory) {
        return new RestTemplate(factory);
    }

    @Bean
    public ClientHttpRequestFactory httpRequestFactory () {
        HttpComponentsClientHttpRequestFactory requestFactory = new HttpComponentsClientHttpRequestFactory(httpClient ()) {
            @Override
            protected HttpUriRequest createHttpUriRequest(HttpMethod httpMethod, URI uri) {
                if (httpMethod == HttpMethod.GET) {
                    return new HttpGetRequestWithEntity(uri);
                }
                return super.createHttpUriRequest(httpMethod, uri);
            }
        };
        return requestFactory;
    }

    public HttpClient httpClient () {
        Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create().register("http",
                PlainConnectionSocketFactory.getSocketFactory()).register("https",
                SSLConnectionSocketFactory.getSocketFactory()).build();

        PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager(registry);
        // 设置最大连接池的数量
        connectionManager.setMaxTotal(1000);
        // 每个主机的最大并发量，route是指域名。--对MaxTotal的细化
        connectionManager.setDefaultMaxPerRoute(300);

        RequestConfig requestConfig = RequestConfig.custom()
                // 数据返回超时时间
                .setSocketTimeout(60 * 1000)
                // 连接超时时间
                .setConnectTimeout(2 * 1000)
                // 从连接池中获取连接的超时时间
                .setConnectionRequestTimeout(200).build();

        return HttpClientBuilder.create()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionManager(connectionManager)
                .setRetryHandler(new DefaultHttpRequestRetryHandler(3, true))
                .setKeepAliveStrategy((response, context) -> {
                    HeaderElementIterator iterator = new BasicHeaderElementIterator(
                            response.headerIterator(HTTP.CONN_KEEP_ALIVE));
                    while (iterator.hasNext()) {
                        HeaderElement headerElement = iterator.nextElement();
                        String param = headerElement.getName();
                        String value = headerElement.getValue();
                        if (null != value && param.equalsIgnoreCase("timeout")) {
                            return Long.parseLong(value) * 1000;
                        }
                    }
                    return 60;
                })
                .build();

    }

    private static final class HttpGetRequestWithEntity extends HttpEntityEnclosingRequestBase {
        public HttpGetRequestWithEntity(final URI uri) {
            super.setURI(uri);
        }
        @Override
        public String getMethod() {
            return HttpMethod.GET.name();

        }
    }

//    @Bean(name = "digiwinApiUrlRestTemplate")
//    public RestTemplate digiwinApiUrlRestTemplate(RestTemplateBuilder builder) {
//        RestTemplateBuilder restTemplateBuilder = builder
//                .setConnectTimeout(Duration.ofMillis(3000L))
//                .setReadTimeout(Duration.ofMillis(3000L));
//        return this.restTemplateBuilder(restTemplateBuilder);
//    }

    private RestTemplate restTemplateBuilder(RestTemplateBuilder restTemplateBuilder) {
        return restTemplateBuilder
                .requestFactory(() -> {
                    Properties props = System.getProperties();
                    if ("true".equals(props.getProperty("http.proxySet"))) {
                        String host = props.getProperty("http.proxyHost");
                        String port = props.getProperty("http.proxyPort");
                        SimpleClientHttpRequestFactory simpleClientHttpRequestFactory =
                                new SimpleClientHttpRequestFactory();
                        SocketAddress address =
                                new InetSocketAddress(host, NumberUtils.parseNumber(port, Integer.class));
                        Proxy proxy = new Proxy(Proxy.Type.HTTP, address);
                        simpleClientHttpRequestFactory.setProxy(proxy);

                        return simpleClientHttpRequestFactory;
                    } else if (props.containsKey("https.proxySet")) {
                        String host = props.getProperty("https.proxyHost");
                        String port = props.getProperty("https.proxyPort");
                        SSLContext sc = null;
                        try {
                            sc = SSLContext.getInstance("SSL");
                            // 指定信任https
                            sc.init(null, new TrustManager[]{new X509TrustManager() {
                                @Override
                                public void checkClientTrusted(X509Certificate[] x509Certificates,
                                                               String s) throws CertificateException {
                                }

                                @Override
                                public void checkServerTrusted(X509Certificate[] x509Certificates,
                                                               String s) throws CertificateException {
                                }

                                @Override
                                public X509Certificate[] getAcceptedIssuers() {
                                    return new X509Certificate[0];
                                }
                            }}, new java.security.SecureRandom());
                        } catch (NoSuchAlgorithmException | KeyManagementException e) {
                            e.printStackTrace();
                        }

                        SimpleClientHttpRequestFactory simpleClientHttpRequestFactory =
                                new SimpleClientHttpRequestFactory();
                        SocketAddress address =
                                new InetSocketAddress(host, NumberUtils.parseNumber(port, Integer.class));
                        Proxy proxy = new Proxy(Proxy.Type.HTTP, address);
                        simpleClientHttpRequestFactory.setProxy(proxy);
                        //                    simpleClientHttpRequestFactory.set

                        return simpleClientHttpRequestFactory;

                    } else {
                        return new OkHttp3ClientHttpRequestFactory();
                    }
                })
//                .rootUri("http://api1.example.com/")
                .interceptors(this.digiwinHttpReqMsgInterceptor(), new DigiwinHttpReqTimeInterceptor())
                .errorHandler(new DigiwinHttpResponseErrorHandler())
                .defaultHeader("Content-Type", MediaType.APPLICATION_JSON.toString()).build();
    }

    private Duration getConnectTimeoutDuration() {
        return TimeOutEnum.getDuration();
    }

    private Duration getReadTimeoutDuration() {
        return TimeOutEnum.getDuration();
    }
}
