package com.digiwin.mobile.mobileuibot.config.filter;

import com.digiwin.mobile.mobileuibot.config.SysEnvConfig;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.context.support.SpringBeanAutowiringSupport;

import javax.servlet.*;
import javax.servlet.annotation.WebFilter;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;

/**
 * <p>功能描述：跨域Filter</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: DigiwinCorsFilter
 * @Author: xujinc
 * @Date: 2023/7/13 10:24
 */
@Component
@WebFilter(urlPatterns = "/*", filterName = "corsFilter", asyncSupported = true)
public class CorsFilter implements Filter {

    @Autowired
    private SysEnvConfig config;

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain)
            throws IOException, ServletException {
        String allowOrigin = config.getAllowedOrigin();
        String requestOrigin = ((HttpServletRequest) servletRequest).getHeader("Origin");

        boolean always = "*".equals(allowOrigin) || StringUtils.isEmpty(requestOrigin);
        //请求域名匹配配置文件允许的域名时，方可跨域访问
        if (always || (StringUtils.isNotEmpty(allowOrigin) && StringUtils.isNotEmpty(requestOrigin))) {
            List<String> allowOriginList = Arrays.asList(StringUtils.split(allowOrigin, ","));
            if (always || allowOriginList.contains(requestOrigin)) {
                HttpServletResponse response = (HttpServletResponse) servletResponse;

                // 允许访问的域名指定请求的同源域名
                response.setHeader("Access-Control-Allow-Origin", always ? "*" : requestOrigin);
                response.setHeader("Access-Control-Allow-Credentials", "true");
                //用来指定本次预检请求的有效期，单位为秒，在此期间不用发出另一条预检请求
                response.setHeader("Access-Control-Max-Age", "3600");
                response.setHeader("Access-Control-Allow-Methods", "*");
                response.setHeader("Access-Control-Allow-Headers", "*");
            }
        }
        filterChain.doFilter(servletRequest, servletResponse);
    }

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        // 在Filter初始化时，手动注入Spring管理的Bean,因为Tomcat启动是先执行Filter 后执行@PostConstruct
        //那这样的话sysEnvConfig还没初始化尼,在此类第35楼则会报错~
        SpringBeanAutowiringSupport.processInjectionBasedOnServletContext(this,
                filterConfig.getServletContext());
        Filter.super.init(filterConfig);
    }

    @Override
    public void destroy() {
        Filter.super.destroy();
    }
}