package com.digiwin.mobile.mobileuibot.config.request;

import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.map.MapUtil;
import com.digiwin.mobile.mobileuibot.common.request.RequestParameterUtil;
import com.digiwin.mobile.mobileuibot.common.url.UrlUtil;
import com.digiwin.mobile.mobileuibot.mock.model.MockData;
import com.digiwin.mobile.mobileuibot.mock.model.MockDataRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.HandlerInterceptor;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * <p>功能描述：</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: RequestDataHandleInterceptor
 * @Author: Zaregoto
 * @Date: 2021/11/29 20:07
 */
@Component
public class RequestContextInterceptor implements HandlerInterceptor {

    private static final Logger logger = LoggerFactory.getLogger(RequestContextInterceptor.class);

    @Autowired
    private MockDataRepository mockDataRepository;

    private static final String MOCK_URI = "/mobile/v1/mock/data";
    // 排除框架内置错误请求【org.springframework.boot.autoconfigure.web.servlet.error.BasicErrorController#error(HttpServletRequest)】
    private static final String ERROR_URI = "/error";

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response,
                             Object handler) throws Exception {
        String requestUri = AppRequestContext.getContextEntity().getRequestUri();
        if (MOCK_URI.equalsIgnoreCase(requestUri) || ERROR_URI.equalsIgnoreCase(requestUri)) {
            return true;
        }
        if (request.getContentType() != null && request.getContentType().contains("multipart/form-data;")) {
            return true;
        }

        String pageId = "";
        String dataId = "";
        String locale = "";
        String requestId = "";
        String userId = "";
        String tenantId = "";
        String iamUserToken = "";
        Map<String, Object> rawData = new HashMap<>();
        if (HttpMethod.POST.name().equalsIgnoreCase(request.getMethod())) {
            Map<String, Object> params = RequestParameterUtil.getPostDataMap(request, false, "UTF-8");
            if (params.containsKey("sysParam")) {
                Map<String, String> sysParam = (Map<String, String>) params.get("sysParam");
                pageId = sysParam.get("pageId");
                dataId = sysParam.get("dataId");
                locale = sysParam.get("locale");
                requestId = sysParam.get("requestId");
                userId = sysParam.get("userId");
                tenantId = sysParam.get("tenantId");
                iamUserToken = sysParam.get("iamUserToken");
            } else {
                pageId = (String) params.get("pageId");
                dataId = (String) params.get("dataId");
                locale = (String) params.get("locale");
                requestId = (String) params.get("requestId");
                userId = (String) params.get("userId");
                tenantId = (String) params.get("tenantId");
                iamUserToken = (String) params.get("iamUserToken");
                rawData = (Map<String, Object>) params.get("rawData");
            }
        }
        if (!StringUtils.hasLength(dataId)) {
            //从url传参获取dataId
            String url = requestUri + "?" + request.getQueryString();
            Map<String, String> params = UrlUtil.parseUrl(url);
            if (params.containsKey("dataId")) {
                dataId = params.get("dataId");
            }
            if (params.containsKey("userId")) {
                userId = params.get("userId");
            }
            if (params.containsKey("tenantId")) {
                tenantId = params.get("tenantId");
            }
            if (params.containsKey("iamUserToken")) {
                iamUserToken = params.get("iamUserToken");
            }
        }

        // 获取代理proxyToken，主要作用于line场景下任务卡获取应用数据
        String proxyToken = request.getHeader("proxyToken");
        if (!CollectionUtils.isEmpty(rawData)) {
            AppRequestContext.getContextEntity().setIsTeamTask((boolean) MapUtil.getOrDefault(rawData, "isTeamTask", false));
            AppRequestContext.getContextEntity().setPinning((boolean) MapUtil.getOrDefault(rawData, "pinning", false));
            if (rawData.containsKey("digi-proxy-token")
                    && StringUtils.hasLength((String) rawData.get("digi-proxy-token"))) {
                proxyToken = (String) rawData.get("digi-proxy-token");
            }
        }
        if (!"null".equalsIgnoreCase(proxyToken)) {
            AppRequestContext.getContextEntity().setProxyToken(proxyToken);
        }

        AppRequestContext.getContextEntity().setPageId(pageId);
        AppRequestContext.getContextEntity().setDataId(dataId);
        String modelType = request.getHeader(AppRequestContext.REQUEST_CONTEXT_KEY_MODELTYPE);
        /**
         * 为大陆安卓应用商店可成功上架做特殊处理
         * 针对登录的账号是shidw@digiwin.com的情况，做如下处理
         * 1. 默认请求头的modelType=show，视为体验模式，并设置体验角色为采购经理PURCHASING_MANAGER
         * 2. 底导航请求：强制使用Demo体验数据，使用两个的底导航。uri为/mobile/v1/navigation/bottom
         * 需要判断token不为空，防止进入游客模式 抛出异常
         */
        String FIXED_USER_ID = "lucxa@digiwin.com";
        final String MOCK_IAMUSERTOKEN = "dabc7c4a-0912-4183-b629-382884a9e6f1";
        if (StringUtils.hasLength(iamUserToken) && (FIXED_USER_ID.equals(userId) || MOCK_IAMUSERTOKEN.equals(iamUserToken))) {
            modelType = AppRequestContext.REQUEST_CONTEXT_VALUE_SHOW;
            this.buildAppRequestContextEntity(request, true,
                    AppRequestContext.REQUEST_CONTEXT_VALUE_EXPERIENCE_MODE_ROLECODE_PURCHASING_MANAGER);
        } else {
            // 无特殊处理的情况
            this.buildAppRequestContextEntity(request, false, request.getHeader(AppRequestContext.REQUEST_CONTEXT_KEY_EXPERIENCE_MODE_ROLE));
        }
//        this.buildAppRequestContextEntity(request, false, request.getHeader(AppRequestContext.REQUEST_CONTEXT_KEY_EXPERIENCE_MODE_ROLE));

        if (AppRequestContext.REQUEST_CONTEXT_VALUE_SHOW.equalsIgnoreCase(modelType)) {
            //查找mock数据
            MockData mockData = mockDataRepository.getMockData(requestUri, pageId, dataId, locale, requestId);
            //检查是否有mock数据，有mock数据则重定向
            if (mockData != null) {
                //把查到的模拟数据传下来
                request.setAttribute("mockData", mockData);
                // 把实际的URI传递下去，作用于 体验模式：V1版本DSL转V2版本DSL
                request.setAttribute("experience_real_url", requestUri);
                request.getRequestDispatcher(MOCK_URI + "/" + request.getMethod().toLowerCase()).forward(request, response);
                return false;
            }
        }


        return HandlerInterceptor.super.preHandle(request, response, handler);
    }

    /**
     * 构建 本应用运行时收到请求的上下文 实体
     *
     * @param request
     * @param forceExperienceMode  是否强制标识为Demo体验模式
     * @param experienceModeRoleId 若forceExperienceMode=true，则需填入Demo体验模式的角色ID
     */
    private void buildAppRequestContextEntity(HttpServletRequest request, boolean forceExperienceMode, String experienceModeRoleId) throws IOException {
        final AppRequestContext.ContextEntity context = AppRequestContext.getContextEntity();

        // 获取渲染DSL版本
        Integer renderVersion = request.getIntHeader(AppRequestContext.REQUEST_HEADER_KEY_RENDERVERSION);
        context.setRenderVersion(renderVersion);

        String modelType = request.getHeader(AppRequestContext.REQUEST_CONTEXT_KEY_MODELTYPE);
        if (forceExperienceMode) {
            modelType = AppRequestContext.REQUEST_CONTEXT_VALUE_SHOW;
        }
        context.setExperienceRoleId(experienceModeRoleId);
        //todo 后续会把这个判断去掉
//        if (null != modelType && Objects.equals(AppRequestContext.REQUEST_CONTEXT_VALUE_SHOW, modelType)) {
        context.setModelType(modelType);
//        }
        if (HttpMethod.GET.name().equals(request.getMethod())) {
            doGet(request, context);
        } else {
            doPost(request, context);
        }
        // 如果前端没有传递tenantId，则从请求头中获取routerKey赋值给tenantId
        if (!StringUtils.hasLength(context.getTenantId())) {
            context.setTenantId(request.getHeader("routerKey"));
        }
        // locale，则获取请求头中的locale
        if (!StringUtils.hasLength(context.getLocale())) {
            context.setLocale(request.getHeader("locale"));
        }
        // 若token为空，则获取请求头中的token
        if (!StringUtils.hasLength(context.getIamUserToken())) {
            context.setIamUserToken(this.getHttpHeaderToken(request));
        }
        // 清理 URL 字符串中的 CRLF 字符
        String requestURI = request.getRequestURI();
        String sanitizedUrl = requestURI.replace("\r", "").replace("\n", "");
        logger.debug("request uri:{}, ip:{}, context:{}", sanitizedUrl, request.getRemoteAddr(), JsonUtil.javaObjectToJsonString(context));
    }

    /**
     * 获取请求头中的token
     * token > iamusertoken > digi-middleware-auth-user
     *
     * @param request 请求
     * @return  token
     */
    private String getHttpHeaderToken(HttpServletRequest request) {
        String token = request.getHeader("token");
        if (StringUtils.hasLength(token)) {
            return token;
        }
        token = request.getHeader("iamusertoken");
        if (StringUtils.hasLength(token)) {
            return token;
        }
        token = request.getHeader("digi-middleware-auth-user");
        if (StringUtils.hasLength(token)) {
            return token;
        }
        return "";
    }

    /**
     * POST 请求 获取参数封装
     *
     * @param request
     * @param context
     * @return
     */
    private void doPost(HttpServletRequest request, AppRequestContext.ContextEntity context)
            throws IOException {
        String tenantId = request.getParameter("tenantId");
        String locale = request.getParameter("locale");
        String userId = request.getParameter("userId");
        String iamUserToken = request.getParameter("iamUserToken");
        context.setTenantId(tenantId);
        context.setLocale(locale);
        context.setUserId(userId);
        context.setIamUserToken(iamUserToken);
        if (StringUtils.hasLength(tenantId)
                && StringUtils.hasLength(locale)
                && StringUtils.hasLength(userId)) {
            return;
        }
        Map<String, Object> params = RequestParameterUtil.getPostDataMap(request, false, request.getCharacterEncoding());
        ApiRequest apiRequest = JsonUtil.objectToJavaObject(params, ApiRequest.class);
        context.setTenantId(apiRequest.getTenantId());
        context.setLocale(apiRequest.getLocale());
        context.setUserId(apiRequest.getUserId());
        context.setIamUserToken(apiRequest.getIamUserToken());
    }

    /**
     * Get 请求 获取参数封装
     *
     * @param request
     * @param context
     * @return
     */
    private void doGet(HttpServletRequest request, AppRequestContext.ContextEntity context) {
        context.setTenantId(request.getParameter("tenantId"));
        context.setLocale(request.getParameter("locale"));
        context.setUserId(request.getParameter("userId"));
        context.setIamUserToken(request.getParameter("iamUserToken"));
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response,
                                Object handler, Exception ex) throws Exception {
        AppRequestContext.remove();
        HandlerInterceptor.super.afterCompletion(request, response, handler, ex);
    }
}
