package com.digiwin.mobile.mobileuibot.config.request;

import com.digiwin.mobile.mobileuibot.common.CommonErrorController;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.proxy.iam.model.DigiwinIamAnalyzedToken;
import com.digiwin.mobile.mobileuibot.proxy.iam.service.DigiwinIamProxyService;
import com.digiwin.mobile.mobileuibot.versionManage.VersionManageLoginService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.HandlerInterceptor;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * <p>功能描述：Token拦截器</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: TokenInterceptor.java
 * @Author: wangjwc
 * @Date: created at 2025/8/13 16:20
 */
@Slf4j
@Component
public class TokenInterceptor implements HandlerInterceptor {
    /**
     * 路由标识
     */
    public static final String PATH_PREFIX = "/mobile/v2/version/manage";
    /**
     * 路由匹配模式
     */
    public static final String PATH_PATTERN = PATH_PREFIX + "/**";
    /**
     * 排除的路由匹配模式
     */
    public static final String EXCLUDE_PATH_PATTERN = PATH_PREFIX + "/login/**";

    @Autowired
    private DigiwinIamProxyService digiwinIamProxyService;

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response,
                             Object handler) throws Exception {
        String requestUri = AppRequestContext.getContextEntity().getRequestUri();
        String iamUserToken = AppRequestContext.getContextEntity().getIamUserToken();
        if (StringUtils.isBlank(iamUserToken)) {
            log.error("request uri: {} be intercepted, iamUserToken is empty", requestUri);
            sendTokenError(request, response);
            return false;
        }
        // 解析token
        DigiwinIamAnalyzedToken analyzedToken = this.tokenAnalyze(iamUserToken);
        if (analyzedToken == null || analyzedToken.getId() == null) {
            log.error("request uri: {} be intercepted, iamUserToken is invalid", requestUri);
            sendTokenError(request, response);
            return false;
        }
        // 判断用户是否在白名单
        if (!this.checkUserInWhiteList(analyzedToken.getId())) {
            log.error("request uri: {} be intercepted, user is not in white list", requestUri);
            sendWhiteListError(request, response);
            return false;
        }
        // 设置用户id
        AppRequestContext.getContextEntity().setUserId(analyzedToken.getId());
        return true;
    }

    /**
     * 解析token
     *
     * @param iamUserToken 用户token
     * @return DigiwinIamAnalyzedToken
     */
    private DigiwinIamAnalyzedToken tokenAnalyze(String iamUserToken) {
        try {
            return digiwinIamProxyService.tokenAnalyze(iamUserToken);
        } catch (Exception e) {
            log.error("tokenAnalyze failure, throwable: {}", e.getMessage(), e);
        }
        return null;
    }

    /**
     * 判断用户是否在白名单中
     *
     * @param userId 用户id
     * @return true:在白名单中，false:不在白名单中
     */
    private boolean checkUserInWhiteList(String userId) {
        return VersionManageLoginService.WHITELIST.contains(userId);
    }

    /**
     * 跳转“用户token失效，请重新登录或联系客服专员”控制器
     *
     * @param request  HttpServletRequest
     * @param response HttpServletResponse
     */
    private void sendTokenError(HttpServletRequest request, HttpServletResponse response) {
        try {
            request.getRequestDispatcher(request.getContextPath() + CommonErrorController.TOKEN_ERROR).forward(request, response);
        } catch (Exception e) {
            log.error("sendTokenError failure, throwable: {}", e.getMessage(), e);
        }
    }

    /**
     * 跳转“用户不在白名单中，请联系客服专员”控制器
     *
     * @param request  HttpServletRequest
     * @param response HttpServletResponse
     */
    private void sendWhiteListError(HttpServletRequest request, HttpServletResponse response) {
        try {
            request.getRequestDispatcher(request.getContextPath() + CommonErrorController.WHITE_LIST_ERROR).forward(request, response);
        } catch (Exception e) {
            log.error("sendWhiteListError failure, throwable: {}", e.getMessage(), e);
        }
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response,
                                Object handler, Exception ex) throws Exception {
        HandlerInterceptor.super.afterCompletion(request, response, handler, ex);
    }
}
