package com.digiwin.mobile.mobileuibot.config.request.log;

import com.alibaba.ttl.TransmittableThreadLocal;
import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.common.calculate.UUIDUtil;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.log.TraceIdUtil;
import com.digiwin.mobile.mobileuibot.common.logrecord.OuterRequestRecordManage;
import com.digiwin.mobile.mobileuibot.common.logrecord.RequestSourceEnum;
import com.digiwin.mobile.mobileuibot.common.request.RequestParameterUtil;
import com.digiwin.mobile.mobileuibot.config.request.RequestWrapper;
import com.digiwin.mobile.mobileuibot.model.db1.OuterRequestRecord;
import com.fasterxml.jackson.core.type.TypeReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.sql.Timestamp;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

/**
 * <p>功能描述：本工程请求响应数据日志拦截器</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: RequestResponseLogInterceptor
 * @Author: Zaregoto
 * @Date: 2021/5/27 10:08
 */
@Component
public class RequestResponseLogInterceptor implements HandlerInterceptor {

    private static final Logger logger = LoggerFactory.getLogger(RequestResponseLogInterceptor.class);

    public static final String RESPONSE_BODY = "response_body";

    /**
     * 存储请求链中 内部请求的相关信息
     */
    private static final TransmittableThreadLocal<OuterRequestRecord> OUTER_REQUEST_RECORD = new TransmittableThreadLocal<>();

    @Autowired
    private OuterRequestRecordManage outerRequestRecordManage;

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response,
                             Object handler) throws Exception {
        try {
            // 防止请求时多个斜杠
            String requestUri = request.getRequestURI().replaceAll("//", "/");
            AppRequestContext.getContextEntity().setRequestUri(requestUri);

            // 获取请求参数
            RequestWrapper requestWrapper = new RequestWrapper(request);

            // 封装请求参数
            this.buildOuterRequestRecordOfRequest(requestWrapper);
            // 清理 URL 字符串中的 CRLF 字符
            String requestURI = request.getRequestURI();
            String sanitizedUrl = requestURI.replace("\r", "").replace("\n", "");
            logger.debug("----------------内部HTTP请求----------------\n"
                            + "CurrentTime: {}\n"
                            + "RequestURI: {}\n"
                            + "Method: {}\n"
                            + "RequestHeaders: {}\n"
                            + "QueryString: {}\n"
                            + "RequestBody: {}\n"
                            + "Controller & Method: {}\n"
                            + "-------------------------End-------------------------",
                    DateTimeUtil.getTodayTimeUseDefaultPattern(),
                    sanitizedUrl,
                    request.getMethod(),
                    JsonUtil.javaObjectToJsonString(this.getRequestHeaderNamesAndValues(request)),
                    request.getQueryString(),
                    requestWrapper.getBodyString(),
                    handler);
        } catch (Exception e) {
            logger.error("MVC业务处理-拦截器异常-preHandle：", e);
        }
        return true;
    }

    /**
     * 封装日志跟踪对象
     *
     * @param requestWrapper
     * @throws IOException
     */
    private void buildOuterRequestRecordOfRequest(RequestWrapper requestWrapper) throws IOException {
        OUTER_REQUEST_RECORD.set(new OuterRequestRecord()
                .setRequestId(UUIDUtil.getUuid())
//                .setTraceId(TraceIdUtil.get())
                .setMethod(requestWrapper.getMethod())
                .setSource(RequestSourceEnum.INNER.getSource())
                .setUrl(requestWrapper.getRequestURL().toString())
                .setRequestObj(this.buildRequest(requestWrapper))
                .setRequestTime(new Timestamp(System.currentTimeMillis())));
    }

    /**
     * 构建请求数据
     *
     * @param request
     * @return
     * @throws IOException
     */
    private OuterRequestRecord.Request buildRequest(RequestWrapper request) throws IOException {
        OuterRequestRecord.Request reqRecord = new OuterRequestRecord.Request();
        reqRecord.setHeader(this.getRequestHeaderNamesAndValues(request))
                .setQueryString(request.getQueryString());
        if (HttpMethod.POST.name().equals(request.getMethod())) {
            Map<String, Object> body = RequestParameterUtil
                    .getPostDataMap(request, false, request.getCharacterEncoding());
            reqRecord.setBody(body);
        }
        return reqRecord;
    }

    @Override
    public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler,
                           ModelAndView modelAndView) throws Exception {
        //响应完成，记录响应处理总时间,移除traceId
        if (StringUtils.hasLength(TraceIdUtil.get())) {
            long startTime = Long.parseLong(TraceIdUtil.getTraceStartTime());
            long endTime = System.currentTimeMillis();
            logger.debug("========================HTTP请求处理完成==========================" +
                            "\nresponseTime：{}" +
                            "\nrunningTime：{}ms",
                    DateTimeUtil.getTodayTimeUseDefaultPattern(), endTime - startTime);
        }
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response,
                                Object handler, Exception ex) throws Exception {
        try {
            // 封装响应参数
            this.buildOuterRequestRecordOfResponse(request, response);
        } catch (Exception e) {
            logger.error("MVC业务处理-拦截器异常-afterCompletion：", e);
        }
        OUTER_REQUEST_RECORD.remove();
    }

    /**
     * 封装响应数据
     *
     * @param request
     * @param response
     */
    private void buildOuterRequestRecordOfResponse(HttpServletRequest request, HttpServletResponse response) {
        Object responseBody = request.getAttribute(RESPONSE_BODY);
        OuterRequestRecord requestRecord = OUTER_REQUEST_RECORD.get();
        if (null == requestRecord) {
            return;
        }
        Timestamp responseTime = new Timestamp(System.currentTimeMillis());
        requestRecord.setResponseObj(this.buildResponse(responseBody, response))
                // 移动内部请求在响应后记录traceId，因为请求中的traceId是移动自定义的，非平台的traceId
                .setTraceId(this.buildTraceId(responseBody))
                .setResponseTime(responseTime)
                .setCode(responseBody == null ? 500 : Integer.parseInt(((ApiResponse) responseBody).getCode()))
                .setMillis(responseTime.getTime() - requestRecord.getRequestTime().getTime());

        // 队列存储
        outerRequestRecordManage.add(requestRecord);
    }

    private String buildTraceId(Object responseBody) {
        String traceId = TraceIdUtil.get();
        try {
            if (responseBody instanceof ApiResponse) {
                String s = ((ApiResponse) responseBody).getTraceId();
                if (StringUtils.hasLength(s)) {
                    return s;
                }
            }
        } catch (Exception ignored) {
        }
        return traceId;
    }

    private OuterRequestRecord.Response buildResponse(Object responseBody, HttpServletResponse response) {
        Map<String, Object> body = new HashMap<>();
        if (null != responseBody) {
            ApiResponse apiResponse = (ApiResponse) responseBody;
            body = JsonUtil.objectToJavaObject(apiResponse, new TypeReference<Map<String, Object>>() {
            });
        }
        OuterRequestRecord.Response respRecord = new OuterRequestRecord.Response();
        respRecord.setHeader(this.getResponseHeaderNamesAndValues(response)).setBody(body);
        return respRecord;
    }

    /**
     * 根据请求头返回完整的请求头数据字符串，用于日志打印
     *
     * @param request
     * @return
     */
    private Map<String, Object> getRequestHeaderNamesAndValues(HttpServletRequest request) {
        Map<String, Object> headers = new HashMap<>();
        Enumeration<String> headerNames = request.getHeaderNames();
        while (headerNames.hasMoreElements()) {
            String headerName = headerNames.nextElement();
            String headerValue = request.getHeader(headerName);
            headers.put(headerName, headerValue);
        }
        return headers;
    }

    /**
     * 根据响应头返回完整的响应头数据
     *
     * @param response
     * @return
     */
    private Map<String, Object> getResponseHeaderNamesAndValues(HttpServletResponse response) {
        Map<String, Object> headers = new HashMap<>();
        response.getHeaderNames().forEach(headerName -> headers.put(headerName, response.getHeader(headerName)));
        return headers;
    }
}
