package com.digiwin.mobile.mobileuibot.config.request.mock;

import com.alibaba.ttl.TransmittableThreadLocal;
import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.common.request.RequestParameterUtil;
import com.digiwin.mobile.mobileuibot.common.string.StringUtil;
import com.digiwin.mobile.mobileuibot.common.url.UrlUtil;
import com.digiwin.mobile.mobileuibot.core.component.MobileRenderDataUtil;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSettingIdPresetEnum;
import com.digiwin.mobile.mobileuibot.experience.model.ExperienceRoleIdEnum;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.mock.MockService;
import com.digiwin.mobile.mobileuibot.mock.enums.EnableEnum;
import com.digiwin.mobile.mobileuibot.model.db1.Mock;
import com.digiwin.mobile.mobileuibot.navigation.BottomNavigation;
import com.digiwin.mobile.mobileuibot.navigation.Navigation;
import com.digiwin.mobile.mobileuibot.navigation.NavigationService;
import com.digiwin.mobile.mobileuibot.proxy.ProxyController;
import com.digiwin.mobile.mobileuibot.proxy.experience.service.DigiwinExperienceProxyService;
import com.digiwin.mobile.mobileuibot.proxy.iam.model.DigiwinIamAuthoredUser;
import com.digiwin.mobile.mobileuibot.proxy.model.ProxyRefreshTenant;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.fasterxml.jackson.core.type.TypeReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.HandlerInterceptor;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.LinkedBlockingDeque;

/**
 * <p>功能描述：Mock拦截器</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: MockInterceptor.java
 * @Author: wangjwc
 * @Date: created at 2024/11/8 15:39
 */
@Component
public class MockInterceptor implements HandlerInterceptor {

    private static final Logger logger = LoggerFactory.getLogger(MockInterceptor.class);

    public static final String RECORD_MOCK_ID = "record-mock-id";

    public static final String LOGIN_URI = "/mobile/v1/proxy/loginInDigiwhale";
    private static final String VERIFICATION_CODE_URI = "/mobile/v1/proxy/getVerificationCode";
    private static final String BOTTOM_URI = "/mobile/v1/navigation/bottom";
    private static final String IM_USER_INFO_URI = "/mobile/v1/im/user/info";
    private static final String REFRESHTENANT_URI = "/mobile/v1/proxy/refreshtenant";

    private static final String RENDER_PAGE_URI = "/mobile/v1/uibot/model";

    /**
     * 存储请求链中 内部请求的相关信息
     */
    public static final TransmittableThreadLocal<OuterMockData> OUTER_MOCK_DATA = new TransmittableThreadLocal<>();

    /**
     * 缓存内容，默认容量 1000
     */
    public static BlockingDeque<OuterMockData> BLOCKING_DEQUE = new LinkedBlockingDeque<>(1000);

    @Autowired
    private DigiwinExperienceProxyService digiwinExperienceProxyService;
    @Autowired
    private LocaleService localeService;

    @Autowired
    private MockService mockService;
    @Autowired
    private NavigationService navigationService;
    @Autowired
    private ProxyController proxyController;

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response,
                             Object handler) throws Exception {
        String mockId = request.getHeader(RECORD_MOCK_ID);
        String modelType = request.getHeader(AppRequestContext.REQUEST_CONTEXT_KEY_MODELTYPE);

        if (!StringUtils.hasLength(mockId)) {
            // 既不是录制也不是体验
            return true;
        }
        if (AppRequestContext.REQUEST_CONTEXT_VALUE_MOCK.equalsIgnoreCase(modelType)) {
            // 体验模式：获取验证码接口走真实场景
            if (VERIFICATION_CODE_URI.equals(AppRequestContext.getContextEntity().getRequestUri())) {
                return true;
            }
            if (LOGIN_URI.equals(AppRequestContext.getContextEntity().getRequestUri())
                    && ExperienceRoleIdEnum.ATHENA_MOCK.getValue().equals(request.getParameter("expRoleId"))) {
                // 体验模式：开启娜娜功能下，手机号注册接口走真实场景
                return true;
            }
            if (IM_USER_INFO_URI.equals(AppRequestContext.getContextEntity().getRequestUri())) {
                // 体验模式：开启娜娜功能下，获取云信账密接口走真实场景 并替换租户ID
                Mock mock = mockService.selectByMockId(mockId, null);
                if (mock != null && EnableEnum.YES.getValue().equals(mock.getEnableNana())) {
                    // 替换租户ID --- 使用配置娜娜的租户来获取云信账密
                    request.setAttribute("nanaTenantId", mock.getNanaTenantId());
                }
                return true;
            }
            // 体验模式
            this.getMockData(mockId, request, response);
            return false;
        }
        // 录制模式
        this.recordMock(mockId, request);
        return true;
    }

    private void recordMock(String mockId, HttpServletRequest request) throws IOException {
        String requestUri = AppRequestContext.getContextEntity().getRequestUri();
        Map<String, Object> payload = this.buildPayload(request);
        OUTER_MOCK_DATA.set(new OuterMockData().setUrl(requestUri).setMethod(request.getMethod())
                .setPayload(JsonUtil.javaObjectToJsonString(payload))
                .setLocale(StringUtil.valueOf(payload.get("locale")))
                .setMockId(mockId));
    }

    private Map<String, Object> buildPayload(HttpServletRequest request) throws IOException {
        String pageId = "";
        String dataId = "";
        String locale = "";
        String requestId = "";
        String tenantId = "";
        // 项目中控台使用
        String projectId = null;
        // 我的任务列表和行事历区分
        Integer calenderType = null;
        if (HttpMethod.POST.name().equalsIgnoreCase(request.getMethod())) {
            Map<String, Object> params = RequestParameterUtil.getPostDataMap(request, false, "UTF-8");
            if (params.containsKey("sysParam")) {
                Map<String, String> sysParam = (Map<String, String>) params.get("sysParam");
                pageId = sysParam.get("pageId");
                dataId = sysParam.get("dataId");
                locale = sysParam.get("locale");
                requestId = sysParam.get("requestId");
                tenantId = sysParam.get("tenantId");
            } else {
                pageId = (String) params.get("pageId");
                dataId = (String) params.get("dataId");
                locale = (String) params.get("locale");
                requestId = (String) params.get("requestId");
                tenantId = (String) params.get("tenantId");
                calenderType = (Integer) params.get("calenderType");
            }
        }
        if (!StringUtils.hasLength(dataId)) {
            String requestUri = AppRequestContext.getContextEntity().getRequestUri();
            //从url传参获取dataId
            String url = requestUri + "?" + request.getQueryString();
            Map<String, String> params = UrlUtil.parseUrl(url);
            if (params.containsKey("dataId")) {
                dataId = params.get("dataId");
            }
            if (params.containsKey("tenantId")) {
                tenantId = params.get("tenantId");
            }
            if (params.containsKey("projectId")) {
                projectId = params.get("projectId");
            }
        }
        Map<String, Object> payload = new HashMap<>();
        payload.put("pageId", pageId);
        payload.put("dataId", dataId);
        payload.put("locale", locale);
        payload.put("requestId", requestId);
        payload.put("tenantId", tenantId);
        payload.put("projectId", projectId);
        payload.put("calenderType", calenderType);
        return payload;
    }

    /**
     * 获取mock体验下的数据
     *
     * @param mockId
     * @param request
     * @param response
     */
    private void getMockData(String mockId, HttpServletRequest request, HttpServletResponse response)
            throws IOException {
        String requestUri = AppRequestContext.getContextEntity().getRequestUri();
        Map<String, Object> payload = buildPayload(request);

        // 查询mock数据
        String responseBody = digiwinExperienceProxyService.gainMockDataOne(new OuterMockData()
                .setMockId(mockId).setUrl(requestUri)
                .setPayload(JsonUtil.javaObjectToJsonString(payload)).setMethod(request.getMethod()));
        String locale = StringUtil.valueOf(payload.get("locale"));
        locale = StringUtils.hasLength(locale) ? locale : "zh_CN";
        this.returnResponseData(mockId, requestUri, locale, responseBody, response);
    }

    private void returnResponseData(String mockId, String requestUri, String locale, String responseBody, HttpServletResponse response) {
        response.setCharacterEncoding("UTF-8");
        response.setContentType("application/json; charset=utf-8");
        try (PrintWriter writer = response.getWriter()) {
            if (StringUtils.hasLength(responseBody)) {
                // 处理返回数据
                responseBody = this.handleResponseBody(mockId, requestUri, locale, responseBody);
                writer.print(MobileRenderDataUtil.upgradeToV2(requestUri, responseBody));
                return;
            }
            if (RENDER_PAGE_URI.equals(requestUri)) {
                // 创建空组件
                UiBotModel uiBotModel = UiBotModel.createEmptyComponent(
                        localeService.getLanguageValue(locale, "此场景暂未录制"),
                        false,
                        "mock_not_yet_record");
                ApiResponse apiResponse = ApiResponse.buildOK().setData(uiBotModel);
                writer.print(JsonUtil.javaObjectToJsonString(MobileRenderDataUtil.upgradeToV2(requestUri, apiResponse)));
                return;
            }
            // 失败 数据不存在
            ApiResponse<Object> result = ApiResponse.buildMockError(localeService.getLanguageValue(locale, "此场景暂未录制"));
            writer.print(JsonUtil.javaObjectToJsonString(MobileRenderDataUtil.upgradeToV2(requestUri, result)));
        } catch (IOException e) {
            logger.error("接口响应异常：", e);
        }
    }

    private String handleResponseBody(String mockId, String requestUri, String locale, String responseBody) {
        if (BOTTOM_URI.equals(requestUri)) {
            // 体验模式：开启娜娜下，底部导航增加娜娜入口
            ApiResponse<List<BottomNavigation>> apiResponse = JsonUtil.jsonStringToObject(responseBody, new TypeReference<ApiResponse<List<BottomNavigation>>>() {
            });
            List<BottomNavigation> data = apiResponse.getData();
            if (!CollectionUtils.isEmpty(data)) {
                if (data.stream().noneMatch(bottomNavigation -> PageSettingIdPresetEnum.GPT_AI.name().equals(bottomNavigation.getPageId()))) {
                    // 录制场景下底部导航中没有GPT_AI，开启娜娜下，则增加娜娜入口
                    Mock mock = mockService.selectByMockId(mockId, null);
                    if (mock != null && EnableEnum.YES.getValue().equals(mock.getEnableNana())) {
                        Navigation navigation = navigationService.findFirst();
                        if (navigation != null && !CollectionUtils.isEmpty(navigation.getBottomNavigations())) {
                            navigation.getBottomNavigations().stream().filter(item -> PageSettingIdPresetEnum.GPT_AI.name().equals(item.getPageId())).findFirst().ifPresent(item -> {
                                item.setPageTitle(LocaleUtil.getMobileTextByKey(locale, item.getPageId()));
                                item.setName(LocaleUtil.getMobileTextByKey(locale, item.getPageId()));
                                data.add(2, item);
                            });
                        }
                    }
                }
            }
            responseBody = JsonUtil.javaObjectToJsonString(apiResponse);
        }
        if (REFRESHTENANT_URI.equals(requestUri)) {
            // 体验模式：开启娜娜功能下，刷新token接口返回注册用户的信息(不使用mock的数据)并租户替换为录制人切换的租户  --- 为了防止切换租户后，相关信息变成了mock的数据，使娜娜使用存在问题
            ApiResponse<ProxyRefreshTenant> apiResponse = JsonUtil.jsonStringToObject(responseBody, new TypeReference<ApiResponse<ProxyRefreshTenant>>() {
            });
            if (apiResponse.isOK()) {
                Mock mock = mockService.selectByMockId(mockId, null);
                if (mock != null && EnableEnum.YES.getValue().equals(mock.getEnableNana())) {
                    ProxyRefreshTenant mockBody = apiResponse.getData();
                    if (mockBody != null) {
                        // 获取注册用户的信息 ---tenantSid=null,tenantId=注册用户的租户ID,token=注册用户的token(前端传递的token)
                        ApiResponse<ProxyRefreshTenant> refreshTenant = proxyController.refreshTenant(
                                null,
                                mock.getNanaTenantId(),
                                AppRequestContext.getContextEntity().getIamUserToken());
                        if (refreshTenant.isOK()) {
                            ProxyRefreshTenant body = refreshTenant.getData();
                            // 租户替换为录制人切换的租户
                            if (body != null) {
                                body.setTenantId(mockBody.getTenantId());
                                body.setTenantSid(mockBody.getTenantSid());
                                body.setTenantName(mockBody.getTenantName());

                                DigiwinIamAuthoredUser iamAuth = body.getAuthoredUser();
                                if (iamAuth != null) {
                                    iamAuth.setTenantId(mockBody.getTenantId());
                                    iamAuth.setTenantSid(mockBody.getTenantSid());
                                    iamAuth.setTenantName(mockBody.getTenantName());
                                }
                                responseBody = JsonUtil.javaObjectToJsonString(refreshTenant);
                            }
                        }
                    }
                }
            }
        }
        return responseBody;
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response,
                                Object handler, Exception ex) throws Exception {
        this.buildResponse();
        OUTER_MOCK_DATA.remove();
    }

    private void buildResponse() {
        OuterMockData outerMockData = OUTER_MOCK_DATA.get();
        if (null == outerMockData) {
            return;
        }
        // 队列存储
        BLOCKING_DEQUE.offer(outerMockData);
    }
}
