package com.digiwin.mobile.mobileuibot.core.component;

import com.digiwin.mobile.mobileuibot.common.calculate.UUIDUtil;
import com.digiwin.mobile.mobileuibot.common.context.SpringContextHolder;
import com.digiwin.mobile.mobileuibot.core.strategy.modelbuild.designer.multipledatasource.UiBotMultipleDataSourceService;
import com.digiwin.mobile.mobileuibot.designer.uibot.enums.DWCallWorkEnum;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotExecuteContext;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotQueryInfo;
import com.digiwin.mobile.mobileuibot.sso.enums.SsoSourceTypeEnum;
import com.digiwin.mobile.mobileuibot.task.common.TaskStringUtil;
import lombok.Data;
import org.apache.commons.collections4.MapUtils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * <p>功能描述：移动端所有组件基类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: BaseMobileComponent
 * @Author: Zaregoto
 * @Date: 2021/6/16 22:39
 */
@Data
public abstract class BaseMobileComponent implements MobileComponent, Serializable {
    private static final long serialVersionUID = -3463956265364418603L;

    /**
     * 组件运行时UUID
     */
    protected String cmptUuid;

    /**
     * 组件版本号
     * 若该字段没有值，或版本号不为2，则前端app会使用旧渲染引擎与相关配套组件（无BaseComponent版本）
     * 若该字段有值，且版本号为2，则前端app会使用新渲染引擎与相关配套组件（有BaseComponent版本）
     * 20230208 add by mowj
     */
    protected Integer cmptVersion;

    /**
     * V2版DSL，组件中需要填充schema值，提供给前端，作用于规则校验
     */
    protected String schema;
    /**
     * schema全路径
     * 在与开发平台设计器集成后引入该字段，用于标识schema的读取数据路径,如：task_info.op_name -- V2版设计器新增属性
     */
    protected String mobilePath;

    /**
     * 组件id
     */
    protected String componentId;

    /**
     * 高级属性
     */
    protected Map<String, Object> hooks;


    protected BaseMobileComponent() {
        this.cmptUuid = UUIDUtil.getUuid();
    }

    public void initSetting(String locale) {
    }

    /**
     * 强制设置组件版本为2
     */
    public void forceSetCmptVersion2() {
        this.cmptVersion = 2;
    }

    public String getCmptUuid() {
        return StringUtils.hasLength(this.getComponentId()) ? this.getComponentId() : this.cmptUuid;
    }

    @Override
    public void applyEditable(Boolean enable) {
        MobileComponent.super.applyEditable(enable);
    }

    @Override
    public void applyHidden(Boolean hidden) {
        MobileComponent.super.applyHidden(hidden);
    }

    public void resetCmptMultipleDataSourceData(ComponentContext cmptContext, PcUiBotExecuteContext executeContext, DataSource dataSource, Map<String, Object> data) {
        // 处理多数据源配置
        Map<String, Object> multipleDataSourceData = SpringContextHolder.getBean(UiBotMultipleDataSourceService.class)
                .buildCmptMultipleDataSourceData(cmptContext, executeContext, dataSource, data);
        if (multipleDataSourceData != null) {
            // 存在多数据源配置，替换原数据
            data.clear();
            data.putAll(multipleDataSourceData);
        }
    }

    public void resetCmptMultipleDataSourceDataAndQueryInfo(ComponentContext cmptContext, PcUiBotExecuteContext executeContext, DataSource dataSource, Map<String, Object> data, PcUiBotQueryInfo queryInfo) {
        Map<String, Object> multipleDataSourceData = null;
        if (queryInfo == null || queryInfo.getPageInfo() == null || queryInfo.getPageInfo().getPageSize() == null) {
            // 不分页
            // 处理多数据源配置
            multipleDataSourceData = SpringContextHolder.getBean(UiBotMultipleDataSourceService.class)
                    .buildCmptMultipleDataSourceData(cmptContext, executeContext, dataSource, data);
        } else {
            // 分页
            // 处理多数据源配置
            multipleDataSourceData = SpringContextHolder.getBean(UiBotMultipleDataSourceService.class)
                    .resetCmptMultipleDataSourceDataAndQueryInfo(cmptContext, executeContext, dataSource, data, queryInfo);
        }
        if (multipleDataSourceData != null) {
            // 存在多数据源配置，替换原数据
            data.clear();
            data.putAll(multipleDataSourceData);
        }
    }

    @Override
    public void handleComponentParam(ComponentContext cmptContext, String mobilePath, String schema, Map<String, Object> data,
                                     PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        MobileComponent.super.handleComponentParam(cmptContext, mobilePath, schema, data, executeContext, mobilePageRawData);

        if (this.judgeNotEditable(cmptContext, mobilePath, schema, data, executeContext, mobilePageRawData)) {
            // 对应的组件执行对应类中的方法，组件设置不可编辑
            this.applyEditable(false);
        }
        // 高级属性-组件级
        if (!CollectionUtils.isEmpty(cmptContext.getHooks())) {
            this.hooks = cmptContext.getHooks().stream()
                    .filter(e -> Objects.equals(this.getComponentId(), MapUtils.getString(e, "eventSource")))
                    .findFirst().orElse(null);
        }
    }

    private boolean judgeNotEditable(ComponentContext cmptContext, String mobilePath, String schema, Map<String, Object> data, PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        if (DWCallWorkEnum.EDIT.getValue().equals(cmptContext.getRawData().getDesignParameter().getDWCallWork())) {
            // 数据录入--业务主键在编辑时栏位不可编辑
            List<String> dataKeys = mobilePageRawData.getDataKeys();
            if (!CollectionUtils.isEmpty(dataKeys) && dataKeys.contains(schema)) {
                return true;
            }
        }
        if (TaskStringUtil.isBizSmartAdjust(executeContext.getTmActivityId())
                && TaskStringUtil.isBizApproval(executeContext.getPattern(), executeContext.getCategory())) {
            // 随心控(人工型：code+_task_manual;审核型：code+_task_approve) 审核型：code+_task_approve，设置不可编辑
            return true;
        }
        if (SsoSourceTypeEnum.NANA_MESSAGE.getSource().equals(cmptContext.getSourceType()) && cmptContext.getReadOnly()) {
            // 任务已完成、分享等场景不可编辑
            return true;
        }
        return false;
    }
}
