package com.digiwin.mobile.mobileuibot.core.component;

import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.context.SpringContextHolder;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.reflection.ReflectionUtils;
import com.digiwin.mobile.mobileuibot.core.component.basic.AutoFill;
import com.digiwin.mobile.mobileuibot.core.component.button.*;
import com.digiwin.mobile.mobileuibot.core.component.card.card.Card;
import com.digiwin.mobile.mobileuibot.core.component.card.cardgroup.CardGroup;
import com.digiwin.mobile.mobileuibot.core.component.chart.*;
import com.digiwin.mobile.mobileuibot.core.component.chart.chartcoordinate.ChartCoordinate;
import com.digiwin.mobile.mobileuibot.core.component.chart.chartcoordinate.ChartCoordinateJsonDeserializer;
import com.digiwin.mobile.mobileuibot.core.component.group.CustomGroup;
import com.digiwin.mobile.mobileuibot.core.component.group.CustomGroupContent;
import com.digiwin.mobile.mobileuibot.core.component.input.attachment.Attachment;
import com.digiwin.mobile.mobileuibot.core.component.input.attachment.AttachmentRawDataDigiwinAthena;
import com.digiwin.mobile.mobileuibot.core.component.input.numeric.InputNumeric;
import com.digiwin.mobile.mobileuibot.core.component.input.numeric.InputNumericTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.layout.singlelistcontainer.SingleListContainer;
import com.digiwin.mobile.mobileuibot.core.component.list.editcardlist.EditCardList;
import com.digiwin.mobile.mobileuibot.core.component.list.projectcc.PhaseList;
import com.digiwin.mobile.mobileuibot.core.component.list.projectcc.TaskList;
import com.digiwin.mobile.mobileuibot.core.component.list.ztbviewlist.ZtbViewList;
import com.digiwin.mobile.mobileuibot.core.component.processnode.MobileProcess;
import com.digiwin.mobile.mobileuibot.core.component.processnode.ProcessNode;
import com.digiwin.mobile.mobileuibot.core.component.progress.TaskProgress;
import com.digiwin.mobile.mobileuibot.core.component.progress.TaskProgressList;
import com.digiwin.mobile.mobileuibot.core.component.tab.customtab.CustomTab;
import com.digiwin.mobile.mobileuibot.core.component.tab.customtab.CustomTabContentContainer;
import com.digiwin.mobile.mobileuibot.core.component.tab.customtab.CustomTabs;
import com.digiwin.mobile.mobileuibot.core.component.tab.customtab.CustomTabsWrapper;
import com.digiwin.mobile.mobileuibot.core.component.tab.customtabcontroller.CustomTabContentArea;
import com.digiwin.mobile.mobileuibot.core.component.tab.customtabcontroller.CustomTabController;
import com.digiwin.mobile.mobileuibot.core.component.tab.customtabcontroller.CustomTabControllerItem;
import com.digiwin.mobile.mobileuibot.core.component.tab.tabs.TabItem;
import com.digiwin.mobile.mobileuibot.core.component.tab.tabs.TabItemCard;
import com.digiwin.mobile.mobileuibot.core.component.tab.tabs.TabItemCardDetailPage;
import com.digiwin.mobile.mobileuibot.core.component.tab.tabs.TabsComponent;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSettingIdPresetEnum;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.locale.service.impl.LocaleServiceImpl;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotRenderData;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.module.SimpleModule;
import lombok.SneakyThrows;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>功能描述：移动端渲染数据处理工具类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: MobileRenderDataUtil
 * @Author: zaregoto
 * @Date: 2023/2/9 16:31
 */
public class MobileRenderDataUtil {

    private static final Logger logger = LoggerFactory.getLogger(MobileRenderDataUtil.class);

    public static final Map<String, Class<? extends BaseMobileComponent>> allMobileComponentsMap = new HashMap<>();
    // 定制的列表组件Map，如：类似于TabsComponent组件，继承了MobileComponent，无法解析到
    public static final Map<String, Class<? extends MobileComponent>> customizeListComponentsMap = new HashMap<>();

    static {
        customizeListComponentsMap.put(TabsComponent.COMPONENT_TYPE, TabsComponent.class);
        customizeListComponentsMap.put(CustomTabs.COMPONENT_TYPE, CustomTabs.class);
        customizeListComponentsMap.put(PhaseList.COMPONENT_TYPE, PhaseList.class);
        customizeListComponentsMap.put(TaskList.COMPONENT_TYPE, TaskList.class);
    }

    static {
        //Reflections方法遇到需要加载依赖jar里的对象时，在部署启动Tomcat服务的时候就会遇到无法到子类的问题
//        Reflections reflections = new Reflections("com.digiwin.mobile.mobileuibot.core.component");
//        Set<Class<? extends BaseMobileComponent>> allMobileComponents = reflections.getSubTypesOf(BaseMobileComponent.class);
        Set<Class<? extends BaseMobileComponent>> allMobileComponents = ReflectionUtils.getSubTypesOf(BaseMobileComponent.class);
        // 转为map，方便查询所有组件类
        for (Class<? extends BaseMobileComponent> clazz : allMobileComponents) {
            String reflectedComponentType = null;
            try {
                if (Modifier.isAbstract(clazz.getModifiers())) {
                    continue;
                }
                if (clazz.equals(CustomTabsWrapper.class)) {
                    continue;
                }
                reflectedComponentType = String.valueOf(clazz.getField("COMPONENT_TYPE").get(null));
                if (StringUtils.hasLength(reflectedComponentType)) {
                    allMobileComponentsMap.put(reflectedComponentType, clazz);
                    //V2组件统一加了前缀"DW_"
                    allMobileComponentsMap.put("DW_" + reflectedComponentType, clazz);
                } else {
                    String notQualifiedComponentType = "[NotQualified]" + clazz.getName();
                    allMobileComponentsMap.put(notQualifiedComponentType, clazz);
                }
            } catch (IllegalAccessException | NoSuchFieldException e) {
                throw new ServiceException(e.getLocalizedMessage());
            }
        }
    }

    /**
     * 升级UiBotModel渲染数据格式
     * v1是放在layout+pageData中的，升级到v2后放在renderData内
     *
     * @param bodyContainerType 页面渲染DSL中body最外层要套的容器类型。0-不使用容器，1-使用单层容器，2-使用多层容器。
     * @return
     */
    @SneakyThrows
    public static <T extends UiBotModel> T upgradeToV2RenderData(T uiBotModelV1, int bodyContainerType) {
        AppRequestContext.getContextEntity().setIsUpgrade(true);
        uiBotModelV1.setRenderVersion(MobileRenderMetaData.RENDER_VERSION_2);
        if (ObjectUtils.isEmpty(uiBotModelV1.getRenderData())) {
            uiBotModelV1.setRenderData(UiBotRenderData.createEmptyRenderData());
        }
        List<BaseMobileComponentWrapper<BaseMobileComponent>> headerContainerWrapperedList = new ArrayList<>(uiBotModelV1.getLayout().size());
        List<BaseMobileComponentWrapper<BaseMobileComponent>> bodyContainerWrapperedList = new ArrayList<>(uiBotModelV1.getLayout().size());
        List<BaseMobileComponentWrapper<BaseMobileComponent>> bottomContainerWrapperedList = new ArrayList<>(uiBotModelV1.getLayout().size());

        for (UiBotLayout layoutToProcess : uiBotModelV1.getLayout()) {
            Object rawComponentData = uiBotModelV1.getPageData().get(layoutToProcess.getSchema());
            if (null == rawComponentData) {
                continue;
            }
            String type = layoutToProcess.getType();
            Class<? extends BaseMobileComponent> clazz = allMobileComponentsMap.getOrDefault(type, null);
            Class<? extends MobileComponent> customizeClazz = customizeListComponentsMap.getOrDefault(type, null);
            if (null == clazz && null == customizeClazz) {
                continue;
            }
            boolean isHeader = false;
            boolean isBody = false;
            boolean isBottom = false;
            BaseMobileComponent baseMobileComponent;
            if (null != clazz) {
                // FIXME 底部按钮格式旧版有设计缺陷，无法直接从数组数据转成某个组件对象数据。
                // 通过重新设计ButtonGroup来统一成对象数据
                if (clazz.equals(BottomButtonDigiwinAthena.class)) {
                    List<BottomButtonDigiwinAthena> bottomButtons = JsonUtil.objectToJavaObject(rawComponentData,
                            new TypeReference<List<BottomButtonDigiwinAthena>>() {
                            });
                    ButtonGroup buttonGroup = new ButtonGroup();
                    buttonGroup.addAllComponentsToGroup(bottomButtons, "DW_" + SingleBottomButton.COMPONENT_TYPE);
                    baseMobileComponent = buttonGroup;
                    isBottom = true;
                } else if (clazz.equals(Attachment.class)) {
                    baseMobileComponent = JsonUtil.objectToJavaObject(rawComponentData, new TypeReference<Attachment<AttachmentRawDataDigiwinAthena>>() {
                    });
                } else if (clazz.equals(AutoFill.class)) {
                    AutoFill autoFill = JsonUtil.objectToJavaObject(rawComponentData, new TypeReference<AutoFill>() {
                    });
                    //一键报工组件的schema写死为AUTO_FILL，和MongoDB 集合：mobileUiBotJsRule里面配置保持一致
                    layoutToProcess.setSchema(AutoFill.COMPONENT_TYPE);
                    autoFill.setSchema(AutoFill.COMPONENT_TYPE);
                    baseMobileComponent = autoFill;
                } else if (clazz.equals(InputNumeric.class)) {
                    InputNumeric inputNumeric = JsonUtil.objectToJavaObject(rawComponentData, new TypeReference<InputNumeric>() {
                    });
                    //给前端的百分比类型数据(0-1)，所以这里需要除以100
                    if (InputNumericTypeEnum.PERCENTAGE.getValue().equals(inputNumeric.getType())) {
                        String text = StringUtils.hasLength(inputNumeric.getText()) ? inputNumeric.getText() : "0";
                        text = (new BigDecimal(text).divide(new BigDecimal(100))).stripTrailingZeros().toPlainString();
                        inputNumeric.setText(text);
                    }
                    baseMobileComponent = inputNumeric;
                } else if (clazz.equals(TaskProgress.class)) {
                    // 因组件类型TASK_PROGRESS 表示为TaskProgressList或TaskProgress
                    if (rawComponentData instanceof List) {
                        List<TaskProgress> taskProgressList = JsonUtil.objectToJavaObject(rawComponentData, new TypeReference<List<TaskProgress>>() {
                        });
                        TaskProgressList customList = new TaskProgressList();
                        customList.setTaskProgressList(taskProgressList);
                        baseMobileComponent = customList;
                    } else {
                        baseMobileComponent = JsonUtil.objectToJavaObject(rawComponentData, TaskProgressList.class);
                    }
                } else if (clazz.getSuperclass().equals(Chart.class)) {
                    Map<String, Object> componentDataMap = (HashMap) rawComponentData;
                    String chartType = String.valueOf(componentDataMap.get("type"));
                    ChartTypeEnum typeEnum = ChartTypeEnum.getEnumByValue(chartType);

                    // FIXME 这里没法复用现有JsonUtil或spring的jackson配置
                    ObjectMapper objectMapper = new ObjectMapper();
                    objectMapper.disable(SerializationFeature.FAIL_ON_EMPTY_BEANS);
                    objectMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
                    objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);

                    SimpleModule simpleModule = new SimpleModule();
                    simpleModule.addDeserializer(ChartCoordinate.class, new ChartCoordinateJsonDeserializer());
                    objectMapper.registerModule(simpleModule);

                    switch (typeEnum) {
                        case LINE:
                            String tempLine = objectMapper.writeValueAsString(rawComponentData);
                            baseMobileComponent = objectMapper.readValue(tempLine, LineChart.class);
//                            baseMobileComponent = JsonUtil.objectToJavaObject(rawComponentData, LineChart.class);
                            break;
                        case BAR:
                            String tempBar = objectMapper.writeValueAsString(rawComponentData);
                            baseMobileComponent = objectMapper.readValue(tempBar, BarChart.class);
//                            baseMobileComponent = JsonUtil.objectToJavaObject(rawComponentData, BarChart.class);
                            break;
                        case LINE_BAR:
                            String tempLineBar = objectMapper.writeValueAsString(rawComponentData);
                            baseMobileComponent = objectMapper.readValue(tempLineBar, LineBarChart.class);
//                            baseMobileComponent = JsonUtil.objectToJavaObject(rawComponentData, BarChart.class);
                            break;
                        case PIE:
                            String tempPie = objectMapper.writeValueAsString(rawComponentData);
                            baseMobileComponent = objectMapper.readValue(tempPie, PieChart.class);
//                            baseMobileComponent = JsonUtil.objectToJavaObject(rawComponentData, PieChart.class);
                            break;
                        case MIXED:
                            String tempMixed = objectMapper.writeValueAsString(rawComponentData);
                            baseMobileComponent = objectMapper.readValue(tempMixed, MixedChart.class);
//                            baseMobileComponent = JsonUtil.objectToJavaObject(rawComponentData, PieChart.class);
                            break;
                        case SCATTER:
                            String tempScatter = objectMapper.writeValueAsString(rawComponentData);
                            baseMobileComponent = objectMapper.readValue(tempScatter, ScatterChart.class);
                        case BUBBLE:
                            String tempSBubble = objectMapper.writeValueAsString(rawComponentData);
                            baseMobileComponent = objectMapper.readValue(tempSBubble, BubbleChart.class);
                        default:
                            baseMobileComponent = JsonUtil.objectToJavaObject(rawComponentData, LineChart.class);
                            break;
                    }
                } else {
                    baseMobileComponent = JsonUtil.objectToJavaObject(rawComponentData, clazz);
                    // 类似CustomGroup支持嵌套的容器组件，且使用v1格式的，需要做升级
                    if (clazz.equals(CustomGroup.class) && baseMobileComponent instanceof CustomGroup) {
                        CustomGroup customGroup = (CustomGroup) baseMobileComponent;
//                        customGroup.setContent();
                        UiBotModel tU1 = upgradeToV2RenderData(customGroup.getContent(), 0);
                        customGroup.setContentGroup(tU1.getRenderData().getBody());
                        // 清空原有结构数据
                        customGroup.setContent(null);
                    } else if (clazz.equals(ProcessNode.class) && baseMobileComponent instanceof ProcessNode) {
                        ProcessNode processNode = (ProcessNode) baseMobileComponent;
                        if (CollectionUtils.isEmpty(processNode.getProcessList())) {
                            continue;
                        }
                        List<MobileProcess> processList = new ArrayList<>();
                        for (MobileProcess mobileProcess : processNode.getProcessList()) {
                            UiBotModel uiBotModel = upgradeToV2RenderData(mobileProcess.getData(), 0);
                            mobileProcess.setData(uiBotModel);
                            processList.add(mobileProcess);
                        }
                        // 清空原有结构数据
                        processNode.setProcessList(null);
                        processNode.setProcessList(processList);
                    }
                }
            } else {
                if (customizeClazz.equals(CustomTabs.class)) {
                    baseMobileComponent = JsonUtil.objectToJavaObject(rawComponentData, CustomTabsWrapper.class);

                    ((CustomTabsWrapper) baseMobileComponent).getData().forEach(e -> {
                        e.getContent().forEach(i -> {
                            Class<? extends BaseMobileComponent> baseMobileComponentType = allMobileComponentsMap.getOrDefault(i.getContentCmptType(), null);
                            if (Objects.nonNull(baseMobileComponentType) && baseMobileComponentType.equals(ProcessNode.class)) {
                                ProcessNode processNode = JsonUtil.objectToJavaObject(i.getData(), ProcessNode.class);
                                if (!CollectionUtils.isEmpty(processNode.getProcessList())) {
                                    List<MobileProcess> processList = new ArrayList<>();
                                    for (MobileProcess mobileProcess : processNode.getProcessList()) {
                                        UiBotModel uiBotModel = upgradeToV2RenderData(mobileProcess.getData(), 0);
                                        mobileProcess.setData(uiBotModel);
                                        processList.add(mobileProcess);
                                    }
                                    // 清空原有结构数据
                                    processNode.setProcessList(null);
                                    processNode.setProcessList(processList);
                                }
                                i.setData(processNode);
                            }
                        });
                    });

                } else {
                    // 定制的列表组件Map
                    MobileComponent mobileComponent = JsonUtil.objectToJavaObject(rawComponentData, customizeClazz);
                    CustomList<List> customList = new CustomList<>();
                    customList.setContent((List) mobileComponent);
                    customList.setComponentType(type);
                    baseMobileComponent = customList;
                }

//                if (customizeClazz.equals(TabsComponent.class)) {
//                    TabsComponent tabsComponent = JsonUtil.objectToJavaObject(rawComponentData, TabsComponent.class);
//                    CustomList<TabsComponent> customList = new CustomList<>();
//                    customList.setContent(tabsComponent);
//                    customList.setComponentType(type);
//                    baseMobileComponent = customList;
//                } else if (customizeClazz.equals(CustomTabs.class)) {
//                    CustomTabs customTabs = JsonUtil.objectToJavaObject(rawComponentData, CustomTabs.class);
//                    CustomList<CustomTabs> customList = new CustomList<>();
//                    customList.setContent(customTabs);
//                    customList.setComponentType(type);
//                    baseMobileComponent = customList;
//                } else {
//                    continue;
//                }
            }
            baseMobileComponent.forceSetCmptVersion2();
            BaseMobileComponentWrapper<BaseMobileComponent> componentWrapper =
                    new BaseMobileComponentWrapper<>(baseMobileComponent, "DW_" + baseMobileComponent.returnComponentType(), layoutToProcess.getSchema());
            if (isBottom) {
                bottomContainerWrapperedList.add(componentWrapper);
            } else if (isHeader) {
                headerContainerWrapperedList.add(componentWrapper);
            } else {
                bodyContainerWrapperedList.add(componentWrapper);
            }
        }
        if (bodyContainerType == 1) {
            SingleListContainer<BaseMobileComponent> singleListContainer = new SingleListContainer<>();
            singleListContainer.addAll(bodyContainerWrapperedList);
            uiBotModelV1.getRenderData().addWrapperedComponentToBody(
                    new BaseMobileComponentWrapper<>(singleListContainer, "DW_" + singleListContainer.returnComponentType()));
        } else if (bodyContainerType == 2) {

        } else {
            uiBotModelV1.getRenderData().addWrapperedComponentListToBody(bodyContainerWrapperedList);
        }

        uiBotModelV1.getRenderData().addWrapperedComponentListToHeader(headerContainerWrapperedList);
        uiBotModelV1.getRenderData().addWrapperedComponentListToBottom(bottomContainerWrapperedList);

        uiBotModelV1.getLayout().clear();
        uiBotModelV1.getPageData().clear();

        return uiBotModelV1;
    }


    /**
     * v2团队任务和项目的页面渲染的一些处理，主要是操作按钮不展示，编辑栏位不可操作
     *
     * @param uiBotModel
     * @param allowDisplayButtonNameList
     * @return
     */
    public static UiBotModel handleV2TeamWork(UiBotModel uiBotModel, List<String> allowDisplayButtonNameList) {
        UiBotRenderData uiBotRenderData = uiBotModel.getRenderData();
        if (Objects.isNull(uiBotRenderData)) {
            return null;
        }
        List<BaseMobileComponentWrapper<BaseMobileComponent>> header = uiBotRenderData.getHeader();
        header.forEach(e -> e.setData((MobileRenderDataUtil.handleV2TeamWorkComponent(e.getData(), e.getComponentType(), allowDisplayButtonNameList))));

        List<BaseMobileComponentWrapper<BaseMobileComponent>> bodyList = uiBotRenderData.getBody();
        bodyList.forEach(e -> e.setData((MobileRenderDataUtil.handleV2TeamWorkComponent(e.getData(), e.getComponentType(), allowDisplayButtonNameList))));

        List<BaseMobileComponentWrapper<BaseMobileComponent>> bottomList = uiBotRenderData.getBottom();
        bottomList.forEach(e -> e.setData(MobileRenderDataUtil.handleV2TeamWorkComponent(e.getData(), e.getComponentType(), allowDisplayButtonNameList)));
        return uiBotModel;
    }

    /**
     * v2团队任务和项目的组件处理，主要是操作按钮不展示，编辑栏位不可操作
     *
     * @param componentPageData
     * @param type
     * @param allowDisplayButtonNameList
     * @return
     */
    private static BaseMobileComponent handleV2TeamWorkComponent(Object componentPageData, String type, List<String> allowDisplayButtonNameList) {
        if (Objects.isNull(componentPageData)) {
            return null;
        }
        Class<? extends BaseMobileComponent> allMobileClazz = allMobileComponentsMap.getOrDefault(type, null);
        if (null == allMobileClazz) {
            return null;
        }

        if (Button.class.equals(allMobileClazz)) {
            Button button = JsonUtil.objectToJavaObject(componentPageData, Button.class);
            String buttonName = button.getName();
            if (allowDisplayButtonNameList.stream().allMatch(c -> StringUtils.hasLength(buttonName) && buttonName.indexOf(c) < 0)) {
                return null;
            }
            Object rawData = button.getAction().getRawData();
            if (rawData instanceof Map) {
                ((Map<Object, Object>) rawData).put("isTeamTask", AppRequestContext.getContextEntity().getIsTeamTask());
            }
            return button;
        } else if (ButtonGroup.class.equals(allMobileClazz)) {
            ButtonGroup buttonGroup = JsonUtil.objectToJavaObject(componentPageData, ButtonGroup.class);
            List<BaseMobileComponentWrapper<BottomButtonDigiwinAthena>> buttonList = buttonGroup.getContentGroup();
            if (CollectionUtils.isEmpty(buttonList)) {
                return null;
            }
            buttonList = buttonList.stream().filter(e -> {
                BaseMobileComponent button = handleV2TeamWorkComponent(e.getData(), e.getComponentType(), allowDisplayButtonNameList);
                if (Objects.isNull(button)) {
                    return false;
                }
                BottomButtonDigiwinAthena bottomButtonDigiwinAthena = JsonUtil.objectToJavaObject(button, BottomButtonDigiwinAthena.class);
                e.setData(bottomButtonDigiwinAthena);
                return true;
            }).collect(Collectors.toList());
            buttonGroup.setContentGroup(buttonList);
            return buttonGroup;
        } else if (CustomGroup.class.equals(allMobileClazz)) {
            CustomGroup customGroup = JsonUtil.objectToJavaObject(componentPageData, CustomGroup.class);
            List<BaseMobileComponentWrapper<BaseMobileComponent>> contentGroupList = customGroup.getContentGroup();
            contentGroupList.forEach(e -> e.setData(handleV2TeamWorkComponent(e.getData(), e.getComponentType(), allowDisplayButtonNameList)));
            return customGroup;
        } else if (CardGroup.class.equals(allMobileClazz)) {
            CardGroup cardGroup = JsonUtil.objectToJavaObject(componentPageData, CardGroup.class);
            List<Card> contentList = cardGroup.getContent().stream().map(e -> (Card) handleV2TeamWorkComponent(e, e.returnComponentType(), allowDisplayButtonNameList)).collect(Collectors.toList());
            cardGroup.setContent(contentList);
            return cardGroup;
        } else if (SingleListContainer.class.equals(allMobileClazz)) {
            SingleListContainer singleListContainer = JsonUtil.objectToJavaObject(componentPageData, SingleListContainer.class);
            List<BaseMobileComponentWrapper> contentList = Optional.ofNullable(singleListContainer.getContentList()).orElse(Collections.emptyList());
            contentList.forEach(e -> e.setData(handleV2TeamWorkComponent(e.getData(), e.getComponentType(), allowDisplayButtonNameList)));
            return singleListContainer;
        } else if (Card.class.equals(allMobileClazz)) {
            Card card = JsonUtil.objectToJavaObject(componentPageData, Card.class);

            Optional detailPageRawData = Optional.ofNullable(card.getDetailPage()).map(TabItemCardDetailPage::getRawData);
            if (detailPageRawData.isPresent()) {
                Map<String, Object> detailPageRawDataMap = (Map<String, Object>) detailPageRawData.get();
                detailPageRawDataMap.put("isTeamTask", AppRequestContext.getContextEntity().getIsTeamTask());
            }

            if (!CollectionUtils.isEmpty(card.getButtonList())) {
                List<Button> buttonList = card.getButtonList().stream().map(e ->
                        (Button) handleV2TeamWorkComponent(e, e.returnComponentType(), allowDisplayButtonNameList)
                ).filter(e -> !Objects.isNull(e)).collect(Collectors.toList());
                card.setButtonList(buttonList);
            }

            if (!CollectionUtils.isEmpty(card.getCardButtons())) {
                List<BaseMobileComponentWrapper<BaseMobileComponent>> cardButtonList = card.getCardButtons().stream().filter(e -> {
                    BaseMobileComponent button = handleV2TeamWorkComponent(e.getData(), e.getComponentType(), allowDisplayButtonNameList);
                    if (Objects.isNull(button)) {
                        return false;
                    }
                    e.setData(button);
                    return true;
                }).collect(Collectors.toList());
                card.setCardButtons(cardButtonList);
            }
            return card;
        } else if (ZtbViewList.class.equals(allMobileClazz)) {
            ZtbViewList ztbViewList = JsonUtil.objectToJavaObject(componentPageData, ZtbViewList.class);
            List<Card> cardList = ztbViewList.getData();
            if (!CollectionUtils.isEmpty(cardList)) {
                cardList = cardList.stream().map(e -> (Card) handleV2TeamWorkComponent(e, e.returnComponentType(), allowDisplayButtonNameList)).collect(Collectors.toList());
                ztbViewList.setData(cardList);
            }
            return ztbViewList;
        } else if (CustomTabController.class.equals(allMobileClazz)) {
            CustomTabController customTabController = JsonUtil.objectToJavaObject(componentPageData, CustomTabController.class);
            List<CustomTabControllerItem> customTabControllerItemList = customTabController.getTabItems();
            if (CollectionUtils.isEmpty(customTabControllerItemList)) {
                return null;
            }
            for (CustomTabControllerItem customTabControllerItem : customTabControllerItemList) {
                List<CustomTabContentContainer> bodyAreaList = Optional.ofNullable(customTabControllerItem.getContentArea())
                        .map(CustomTabContentArea::getBodyArea).orElse(Collections.emptyList());
                bodyAreaList.forEach(e -> e.setData(handleV2TeamWorkComponent(e.getData(), e.getContentCmptType(), allowDisplayButtonNameList)));

                List<CustomTabContentContainer> bottomAreaList = Optional.ofNullable(customTabControllerItem.getContentArea())
                        .map(CustomTabContentArea::getBottomArea).orElse(Collections.emptyList());
                bottomAreaList.forEach(e -> e.setData(handleV2TeamWorkComponent(e.getData(), e.getContentCmptType(), allowDisplayButtonNameList)));
            }
            return customTabController;
        } else if (Attachment.class.equals(allMobileClazz)) {
            Attachment attachment = JsonUtil.objectToJavaObject(componentPageData, new TypeReference<Attachment<AttachmentRawDataDigiwinAthena>>() {
            });
            attachment.setEnable(false);
            return attachment;
        } else {
            if (isBaseInputMobileComponent(allMobileClazz)) {
                BaseMobileComponent baseMobileComponent = JsonUtil.objectToJavaObject(componentPageData, allMobileClazz);
                BaseInputMobileComponent baseInputMobileComponent = (BaseInputMobileComponent) baseMobileComponent;
                baseInputMobileComponent.setEnable(false);
                baseInputMobileComponent.setRequired(false);
                return baseInputMobileComponent;
            }
        }
        return JsonUtil.objectToJavaObject(componentPageData, allMobileClazz);
    }


    /**
     * v1团队任务和项目页面渲染的一些处理，主要是操作按钮不展示，编辑栏位不可操作
     * v1有些渲染的组件是使用UiBotModel结构，如CustomGroup的content,递归时需要区分判断
     *
     * @param data
     * @param type
     * @param allowDisplayButtonNameList
     * @return
     */
    public static Object handleV1TeamWork(Object data, String type, List<String> allowDisplayButtonNameList) {
        if (data instanceof UiBotModel) {
            UiBotModel uiBotModel = (UiBotModel) data;
            for (UiBotLayout layout : uiBotModel.getLayout()) {
                Object handleResult = handleV1TeamWorkComponent(uiBotModel.getPageData().get(layout.getSchema()), layout.getType(), allowDisplayButtonNameList);
                uiBotModel.getPageData().put(layout.getSchema(), handleResult);
            }
            return uiBotModel;
        } else {
            return handleV1TeamWorkComponent(data, type, allowDisplayButtonNameList);
        }
    }

    /**
     * v1团队任务和项目的组件处理，主要是操作按钮不展示，编辑栏位不可操作
     *
     * @param componentPageData
     * @param type
     * @param allowDisplayButtonNameList
     * @return
     */
    private static Object handleV1TeamWorkComponent(Object componentPageData, String type, List<String> allowDisplayButtonNameList) {
        if (Objects.isNull(componentPageData)) {
            return null;
        }
        Class<? extends BaseMobileComponent> allMobileClazz = allMobileComponentsMap.getOrDefault(type, null);
        Class<? extends MobileComponent> customizeClazz = customizeListComponentsMap.getOrDefault(type, null);

        if (null != allMobileClazz) {
            if (BottomButtonDigiwinAthena.class.equals(allMobileClazz)) {
                List<BottomButtonDigiwinAthena> bottomButtons = JsonUtil.objectToJavaObject(componentPageData,
                        new TypeReference<List<BottomButtonDigiwinAthena>>() {
                        });
                if (!CollectionUtils.isEmpty(bottomButtons)) {
                    bottomButtons = handleV1ButtonList(bottomButtons, allowDisplayButtonNameList);
                }
                return bottomButtons;
            } else if (CustomGroup.class.equals(allMobileClazz)) {
                CustomGroup customGroup = JsonUtil.objectToJavaObject(componentPageData, CustomGroup.class);
                customGroup.setContent((CustomGroupContent) handleV1TeamWork(customGroup.getContent(), null, allowDisplayButtonNameList));
                return customGroup;
            } else if (CardGroup.class.equals(allMobileClazz)) {
                CardGroup cardGroup = JsonUtil.objectToJavaObject(componentPageData, CardGroup.class);
                List<Card> contentList = cardGroup.getContent().stream().map(e -> (Card) handleV2TeamWorkComponent(e, e.returnComponentType(), allowDisplayButtonNameList)).collect(Collectors.toList());
                cardGroup.setContent(contentList);
                return cardGroup;
            } else if (Card.class.equals(allMobileClazz)) {
                Card card = JsonUtil.objectToJavaObject(componentPageData, Card.class);
                Optional detailPageRawData = Optional.ofNullable(card.getDetailPage()).map(TabItemCardDetailPage::getRawData);
                if (detailPageRawData.isPresent()) {
                    card.getDetailPage().getRawData().put("isTeamTask", AppRequestContext.getContextEntity().getIsTeamTask());
                }
                List<Button> buttonList = card.getButtonList();
                if (!CollectionUtils.isEmpty(buttonList)) {
                    buttonList = handleV1ButtonList(buttonList, allowDisplayButtonNameList);
                    card.setButtonList(buttonList);
                }
                return card;
            } else if (ZtbViewList.class.equals(allMobileClazz)) {
                ZtbViewList ztbViewList = JsonUtil.objectToJavaObject(componentPageData, ZtbViewList.class);
                List<Card> cardList = ztbViewList.getData();
                if (!CollectionUtils.isEmpty(cardList)) {
                    cardList = cardList.stream().map(e -> (Card) handleV1TeamWorkComponent(e, e.returnComponentType(), allowDisplayButtonNameList)).collect(Collectors.toList());
                    ztbViewList.setData(cardList);
                }
                return ztbViewList;
            } else if (ProcessNode.class.equals(allMobileClazz)) {
                ProcessNode processNode = JsonUtil.objectToJavaObject(componentPageData, ProcessNode.class);
                List<MobileProcess> mobileProcessList = processNode.getProcessList();
                if (!CollectionUtils.isEmpty(mobileProcessList)) {
                    for (MobileProcess mobileProcess : mobileProcessList) {
                        UiBotModel mobileProcessUiBotModel = mobileProcess.getData();
                        mobileProcess.setData((UiBotModel) handleV1TeamWork(mobileProcessUiBotModel, null, allowDisplayButtonNameList));

                        List<BottomButtonDigiwinAthena> bottomButtons = mobileProcess.getBottomButtonList();
                        if (!CollectionUtils.isEmpty(bottomButtons)) {
                            bottomButtons = handleV1ButtonList(bottomButtons, allowDisplayButtonNameList);
                            BottomButtonList bottomButtonList = new BottomButtonList();
                            bottomButtonList.addAll(bottomButtons);
                            mobileProcess.setBottomButtonList(CollectionUtils.isEmpty(bottomButtonList) ? null : bottomButtonList);
                        }
                    }
                }
                return processNode;
            } else if (Attachment.class.equals(allMobileClazz)) {
                Attachment attachment = JsonUtil.objectToJavaObject(componentPageData, new TypeReference<Attachment<AttachmentRawDataDigiwinAthena>>() {
                });
                attachment.setEnable(false);
                return attachment;
            } else {
                if (isBaseInputMobileComponent(allMobileClazz)) {
                    BaseMobileComponent baseMobileComponent = JsonUtil.objectToJavaObject(componentPageData, allMobileClazz);
                    BaseInputMobileComponent baseInputMobileComponent = (BaseInputMobileComponent) baseMobileComponent;
                    baseInputMobileComponent.setEnable(false);
                    baseInputMobileComponent.setRequired(false);
                    return baseInputMobileComponent;
                }
            }
        }

        if (null != customizeClazz) {
            if (CustomTabs.class.equals(customizeClazz)) {
                CustomTabsWrapper customTabsWrapper = JsonUtil.objectToJavaObject(componentPageData, CustomTabsWrapper.class);
                List<CustomTab> customTabs = customTabsWrapper.getData();
                for (CustomTab customTab : customTabs) {
                    List<CustomTabContentContainer> customTabContentContainerList = customTab.getContent();
                    for (CustomTabContentContainer customTabContentContainer : customTabContentContainerList) {
                        Object customTabContentContainerData = handleV1TeamWork(customTabContentContainer.getData(), customTabContentContainer.getContentCmptType(), allowDisplayButtonNameList);
                        customTabContentContainer.setData(customTabContentContainerData);
                        Optional detailPageRawData = Optional.ofNullable(customTabContentContainer.getDetailPage()).map(TabItemCardDetailPage::getRawData);
                        if (detailPageRawData.isPresent()) {
                            customTabContentContainer.getDetailPage().getRawData().put("isTeamTask", true);
                        }
                    }
                }
                return customTabsWrapper;
            }
            if (TabsComponent.class.equals(customizeClazz)) {
                TabsComponent tabsComponent = JsonUtil.objectToJavaObject(componentPageData, TabsComponent.class);
                if (!CollectionUtils.isEmpty(tabsComponent)) {
                    for (TabItem tabItem : tabsComponent) {
                        List<TabItemCard> tabItemCardList = tabItem.getContent();
                        if (CollectionUtils.isEmpty(tabItemCardList)) {
                            continue;
                        }
                        tabItemCardList.forEach(e -> {
                            e.setButtonLayout((UiBotModel) handleV1TeamWork(e.getButtonLayout(), null, allowDisplayButtonNameList));
                            Optional detailPageRawData = Optional.ofNullable(e.getDetailPage()).map(TabItemCardDetailPage::getRawData);
                            if (detailPageRawData.isPresent()) {
                                e.getDetailPage().getRawData().put("isTeamTask", true);
                            }
                        });
                    }
                }
                return tabsComponent;
            }
        }
        return componentPageData;
    }

    /**
     * 获取局部规则对应的组件
     */
    public static void getUseLocalRulesBaseMobileComponent(Object componentPageData, String type, List<BaseMobileComponent> baseMobileComponentList) {
        if (Objects.isNull(componentPageData)) {
            return;
        }
        Class<? extends BaseMobileComponent> allMobileClazz = allMobileComponentsMap.getOrDefault(type, null);
        if (null == allMobileClazz) {
            return;
        }
        if (CustomGroup.class.equals(allMobileClazz)) {
            CustomGroup customGroup = JsonUtil.objectToJavaObject(componentPageData, CustomGroup.class);
            List<BaseMobileComponentWrapper<BaseMobileComponent>> contentGroupList = customGroup.getContentGroup();
            contentGroupList.forEach(e -> getUseLocalRulesBaseMobileComponent(e.getData(), e.getComponentType(), baseMobileComponentList));
        } else if (SingleListContainer.class.equals(allMobileClazz)) {
            SingleListContainer singleListContainer = JsonUtil.objectToJavaObject(componentPageData, SingleListContainer.class);
            List<BaseMobileComponentWrapper> contentList = Optional.ofNullable(singleListContainer.getContentList()).orElse(Collections.emptyList());
            contentList.forEach(e -> getUseLocalRulesBaseMobileComponent(e.getData(), e.getComponentType(), baseMobileComponentList));
        } else if (CustomTabController.class.equals(allMobileClazz)) {
            CustomTabController customTabController = JsonUtil.objectToJavaObject(componentPageData, CustomTabController.class);
            List<CustomTabControllerItem> customTabControllerItemList = customTabController.getTabItems();
            if (CollectionUtils.isEmpty(customTabControllerItemList)) {
                return;
            }
            for (CustomTabControllerItem customTabControllerItem : customTabControllerItemList) {
                List<CustomTabContentContainer> bodyAreaList = Optional.ofNullable(customTabControllerItem.getContentArea())
                        .map(CustomTabContentArea::getBodyArea).orElse(Collections.emptyList());
                bodyAreaList.forEach(e -> getUseLocalRulesBaseMobileComponent(e.getData(), e.getContentCmptType(), baseMobileComponentList));
            }
        } else if (ZtbViewList.class.equals(allMobileClazz)) {
            ZtbViewList ztbViewList = JsonUtil.objectToJavaObject(componentPageData, ZtbViewList.class);
            baseMobileComponentList.add(ztbViewList);
        } else if (EditCardList.class.equals(allMobileClazz)) {
            EditCardList editCardList = JsonUtil.objectToJavaObject(componentPageData, EditCardList.class);
            baseMobileComponentList.add(editCardList);
        }
    }

    /**
     * 判断是否是录入型组件
     *
     * @param allMobileClazz
     * @return boolean
     */
    private static boolean isBaseInputMobileComponent(Class<?> allMobileClazz) {
        if (BaseMobileComponent.class.equals(allMobileClazz)) {
            return false;
        }

        Optional isBaseInputMobileComponent = Optional.ofNullable(allMobileClazz.getSuperclass()).filter(e -> BaseInputMobileComponent.class.equals(e));
        if (isBaseInputMobileComponent.isPresent()) {
            return true;
        }
        return isBaseInputMobileComponent(allMobileClazz.getSuperclass());
    }

    private static <T extends Button> List<T> handleV1ButtonList(List<T> buttonList, List<String> allowDisplayButtonNameList) {
        String locale = AppRequestContext.getContextEntity().getLocale();
        LocaleService localeService = SpringContextHolder.getBean(LocaleServiceImpl.class);
        String cancelName = localeService.getLanguageValue(locale, "取消");
        buttonList = buttonList.stream().filter(e -> allowDisplayButtonNameList.stream()
                .anyMatch(c -> {
                    if (StringUtils.hasLength(e.getName())) {
                        if (e.getName().startsWith(cancelName)) {
                            e.setIsCheckButton(false);
                            return e.getName().equalsIgnoreCase(cancelName);
                        }
                        return e.getName().indexOf(c) > -1;
                    }
                    return false;
                })
        ).collect(Collectors.toList());
        buttonList.forEach(e -> {
            Object rawData = e.getAction().getRawData();
            if (rawData instanceof Map) {
                ((Map<Object, Object>) rawData).put("isTeamTask", AppRequestContext.getContextEntity().getIsTeamTask());
            }
        });
        return buttonList;
    }

    /**
     * 多语言处理
     *
     * @param result
     * @param locale
     */
    public static void multilingualProcessing(Map<String, Object> result, String locale) {
        if (ObjectUtils.isEmpty(result)) {
            return;
        }
        if (!ObjectUtils.isEmpty(result.get("lang"))) {
            Map<String, Map<String, String>> langMap = (Map<String, Map<String, String>>) result.get("lang");
            langMap.keySet().forEach(key -> Optional.ofNullable(langMap.get(key)).map(m -> m.get(locale)).ifPresent(v -> result.put(key, v)));
        }
        result.forEach((key, value) -> {
            if (value instanceof Map) {
                multilingualProcessing((Map<String, Object>) value, locale);
            } else if (value instanceof List) {
                List<Object> list = (List) value;
                if (!CollectionUtils.isEmpty(list)) {
                    for (Object o : list) {
                        if (o instanceof Map) {
                            multilingualProcessing((Map<String, Object>) o, locale);
                        }
                    }
                }
            }
        });
    }

    /**
     * 通过V1版本的DSL 升级为V2版本的DSL
     *
     * @param requestUri   请求的URI
     * @param responseBody 构建完成后的V1版本DSL
     * @return V2版本的DSL
     */
    public static String upgradeToV2(String requestUri, String responseBody) {
        if (!MobileRenderMetaData.RENDER_VERSION_2.equals(AppRequestContext.getContextEntity().getRenderVersion())) {
            // renderVersion!=2 ，则直接返回，不升级
            return responseBody;
        }
        if (!StringUtils.hasLength(responseBody)) {
            return responseBody;
        }
        ApiResponse apiResponse =
                JsonUtil.jsonStringToObject(responseBody, new TypeReference<ApiResponse>() {
                });
        return JsonUtil.javaObjectToJsonString(upgradeToV2(requestUri, apiResponse));
    }

    /**
     * 根据条件判断：V1版本的DSL 是否 升级为V2版本的DSL
     * <p>
     * 前端请求传递renderVersion版本，与UiBotModel返回的renderVersion版本：
     * 若一样，均是v1，或均是v2，不升级；
     * 若请求v1，返回v2，不升级；
     * 若请求v2，返回v1，要升级
     *
     * @param requestUri  请求的URI
     * @param apiResponse V1版本DSL
     * @return V1版本的DSL或V2版本的DSL
     */
    public static ApiResponse upgradeToV2(String requestUri, ApiResponse apiResponse) {
        if (!MobileRenderMetaData.RENDER_VERSION_2.equals(AppRequestContext.getContextEntity().getRenderVersion())) {
            // renderVersion!=2 ，则直接返回，不升级
            return apiResponse;
        }
        if (apiResponse.isError()) {
            return apiResponse;
        }
        switch (requestUri) {
            case "/mobile/v1/uibot/model":
            case "/mobile/v2/uibot/model":
            case "/mobile/v1/proxy/get/breadCrumb/data":
                try {
                    UiBotModel uiBotModel = JsonUtil.objectToJavaObject(apiResponse.getData(), new TypeReference<UiBotModel>() {
                    });
                    if (MobileRenderMetaData.RENDER_VERSION_2.equals(uiBotModel.getRenderVersion())) {
                        // uibotModel 已经返回renderVersion==2 ，则直接返回，不升级
                        return apiResponse;
                    }
                    String pageId = AppRequestContext.getContextEntity().getPageId();
                    int bodyContainerType = 1;
                    if (PageSettingIdPresetEnum.INDEX.name().equals(pageId)
                            || PageSettingIdPresetEnum.MOBILE_ATHENA_PROJECT.name().equals(pageId)
                            || PageSettingIdPresetEnum.MOBILE_MY_TODO.name().equals(pageId)
                            || PageSettingIdPresetEnum.MOBILE_ATHENA_RECONCILIATION_PROGRESS_NOTICE_DETAIL.name().equals(pageId)
                            || PageSettingIdPresetEnum.MOBILE_ATHENA_SYSTEM_CLOUD_HOUSE_KEEPER_PROJECT_DETAIL.name().equals(pageId)
                            || PageSettingIdPresetEnum.MOBILE_ATHENA_INSPECTION_RESULTS_REGISTRATION_SUB_ITEM_CUSTOMIZE_DETAIL.name().equals(pageId)
                            || PageSettingIdPresetEnum.MOBILE_ATHENA_TBDS_PRODUCTION_PROCESS_RECORD.name().equals(pageId)
                            || PageSettingIdPresetEnum.MOBILE_ATHENA_TBDS_PRODUCTION_PROCESS_RECORD_NEXT_STEP.name().equals(pageId)
                            || PageSettingIdPresetEnum.MOBILE_ATHENA_PROJECT_CC_PROJECT_DETAIL.name().equals(pageId)
                            || PageSettingIdPresetEnum.MOBILE_ATHENA_SHARE_OTHERS.name().equals(pageId)
                            || isNotUseSingleContainerByUrl(requestUri)
                    ) {
                        // 不使用单层容器
                        bodyContainerType = 0;
                    }
                    upgradeToV2RenderData(uiBotModel, bodyContainerType);
                    apiResponse.setData(uiBotModel);
                } catch (Exception e) {
                    logger.error("V1 DSL upgrade To V2 DSL error:", e);
                }
                break;
            default:
                break;
        }
        return apiResponse;
    }

    private static boolean isNotUseSingleContainerByUrl(String requestUrl) {
        return "/mobile/v1/proxy/get/breadCrumb/data".equals(requestUrl);
    }
}