package com.digiwin.mobile.mobileuibot.core.component.basic;

import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponentPerson;
import com.digiwin.mobile.mobileuibot.core.component.ComponentContext;
import com.digiwin.mobile.mobileuibot.core.component.MobilePageRawData;
import com.digiwin.mobile.mobileuibot.core.component.action.Action;
import com.digiwin.mobile.mobileuibot.core.component.card.card.Card;
import com.digiwin.mobile.mobileuibot.designer.uibot.service.UiBotDesignerRenderService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotExecuteContext;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumn;
import com.digiwin.mobile.mobileuibot.proxy.uibot.parser.ParseFieldData;
import lombok.Data;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.util.StringUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import static com.digiwin.mobile.mobileuibot.core.layout.doublepattern.util.ModuleUtils.localeService;

/**
 * <p>功能描述：基础的字段label和value展示组件</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: Field
 * @Author: Zaregoto
 * @Date: 2021/5/20 14:45
 */
@Data
public class Field extends BaseMobileComponent {
    private static final long serialVersionUID = -8888100200580189673L;
    public static final String COMPONENT_TYPE = "FIELD";

    /**
     * 展示文字标签
     */
    protected String label;

    /**
     * 是否展示文字标签，false不展示，默认true展示
     */
    protected Boolean labelVisible;

    /**
     * eg. 表示字段展示时使用的类型,同pc端uibot的column.type，非dataType。如AMOUNT、LABEL
     * CURRENT_ACCOUNT表示value展示当前登陆人用户名
     */
    protected String type;
    /**
     * 展示的字段值
     */
    protected String value;
    /**
     * 组件对应的字段
     */
    protected String schema;
    /**
     * 隐藏值
     */
    private String hideValue;

    /**
     * 提交时的内容
     */
    protected String text;

    /**
     * 对齐方式
     *
     * @see FieldAlignmentTypeEnum
     */
    protected Integer alignmentType = FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_RIGHT.getValue();

    /**
     * label样式类型
     * <p>
     * 1：黑色，普通型文本。2：灰色，弱化型文本
     *
     * @see FieldLabelStyleTypeEnum
     */
    protected Integer labelStyleType;

    /**
     * value样式类型
     * <p>
     * 1：黑色，普通型文本。2：灰色，弱化型文本。3：大标题样式。4：红色
     *
     * @see FieldValueStyleTypeEnum
     */
    protected Integer valueStyleType;

    /**
     * 仅alignmentType为3时有效，显示有右侧
     */
    private BaseMobileComponentPerson person;
    /**
     * 是否变动
     */
    private Boolean changeFlag = false;
    /**
     * 是否隐藏
     */
    protected Boolean hidden;
    protected Boolean enable;
    /**
     * 行事历的执行人
     */
    protected Executor executor;

    /**
     * 点击事件
     * 当action不为空且action.type不为空时，value右侧展示可点击按钮
     */
    protected Action action;

    /**
     * 内容排列
     * 默认1 1水平  2垂直
     */
    private Integer align = FieldAlignEnum.HORIZONTAL.getValue();

    public Field() {
    }

    protected Field(String label, String type, String value) {
        this.label = label;
        /**
         * 类型会收到的有以下几个：
         * LABEL、AMOUNT、SELECT
         */
        this.type = null == type ? "LABEL" : type;
        this.value = value;
    }

    public static Field create(String label, String value) {
        return Field.create(label, "LABEL", StringUtils.hasLength(value) ? value : "-");
    }

    public static Field createLabelValueLeftFieldNoNewLineWith18px(String label, String value) {
        Field field = Field.create(label, "LABEL", StringUtils.hasLength(value) ? value : "-");
        field.setLabelVisible(StringUtils.hasLength(label));
        field.setAlignmentType(FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_LEFT_WITH_18PX.getValue());
        field.setLabelStyleType(FieldLabelStyleTypeEnum.LABEL_GREY.getValue());
        field.setValueStyleType(FieldValueStyleTypeEnum.VALUE_BLACK.getValue());
        return field;
    }

    /**
     * alignmentType为4的field
     *
     * @param label
     * @param value
     * @return
     */
    public static Field createLabelValueLeftField(String label, String value) {
        Field field = Field.create(label, "LABEL", StringUtils.hasLength(value) ? value : "-");
        if (!StringUtils.hasLength(label)) {
            field.setLabelVisible(false);
            field.setValueStyleType(FieldValueStyleTypeEnum.VALUE_BIG_TITLE.getValue());
        } else {
            field.setLabelVisible(true);
        }
        field.setAlignmentType(FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_LEFT_NEWLINE.getValue());
        return field;
    }

    /**
     * 一些场景 只使用Field结构，不使用Field组件中各种类型控制
     * valueStyleType 0：一行显示...  1：换行显示
     *
     * @param label
     * @param value
     * @return
     */
    public static Field createFieldOfOtherScene(String label, String value) {
        Field field = Field.create(label, "LABEL", StringUtils.hasLength(value) ? value : "-");
        field.setValueStyleType(1);
        return field;
    }

    /**
     * alignmentType为4的field
     *
     * @param label
     * @param value
     * @return
     */
    public static Field createLabelValueLeftField(String label, String value, Integer labelType, Integer valueType, String hideValue) {
        Field field = Field.create(label, "LABEL", StringUtils.hasLength(value) ? value : "-");
        field.setLabelVisible(StringUtils.hasLength(label));
        field.setAlignmentType(FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_LEFT_NEWLINE.getValue());
        field.setLabelStyleType(labelType);
        field.setValueStyleType(valueType);
        field.setHideValue(hideValue);
        return field;
    }

    /**
     * alignmentType为1的field
     *
     * @param label
     * @param value
     * @return
     */
    public static Field createLabelLeftValueRightField(String label, String value, Integer labelType, Integer valueType) {
        Field field = Field.create(label, "LABEL", StringUtils.hasLength(value) ? value : "-");
        field.setLabelVisible(StringUtils.hasLength(label));
        field.setAlignmentType(FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_RIGHT.getValue());
        field.setLabelStyleType(labelType);
        field.setValueStyleType(valueType);
        return field;
    }

    /**
     * alignmentType为1,labelStyleType为2，valueStyleTyp为1的field
     *
     * @param label
     * @param value
     * @return
     */
    public static Field createLabelLeftValueRightField(String label, String value) {
        Field field = Field.create(label, "LABEL", StringUtils.hasLength(value) ? value : "-");
        field.setLabelVisible(StringUtils.hasLength(label));
        field.setAlignmentType(FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_RIGHT.getValue());
        field.setLabelStyleType(FieldLabelStyleTypeEnum.LABEL_GREY.getValue());
        field.setValueStyleType(FieldValueStyleTypeEnum.VALUE_BLACK.getValue());
        return field;
    }

    /**
     * alignmentType为1的field
     *
     * @param label
     * @param value
     * @return
     */
    public static Field createLabelLeftValueRightFieldBy18PX(String label, String value, Integer labelType, Integer valueType) {
        Field field = Field.create(label, "LABEL", StringUtils.hasLength(value) ? value : "-");
        field.setLabelVisible(StringUtils.hasLength(label));
        field.setAlignmentType(FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_RIGHT_WITH_18PX.getValue());
        field.setLabelStyleType(labelType);
        field.setValueStyleType(valueType);
        return field;
    }

    /**
     * alignmentType为3的field
     *
     * @param label
     * @param value
     * @return
     */
    public static Field createLabelValueLeftFieldNoNewLine(String label, String value) {
        Field field = Field.create(label, "LABEL", StringUtils.hasLength(value) ? value : "-");
        field.setLabelVisible(StringUtils.hasLength(label));
        field.setAlignmentType(FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_LEFT_NEWLINE.getValue());
        return field;
    }

    public static Field create(String label, String type, String value) {
        Field field = new Field();
        field.setLabel(label);
        field.setType(type);
        field.setValue(value);

        return field;
    }

    public static Field create(String label, String type, String value, String schema) {
        Field field = new Field();
        field.setLabel(label);
        field.setType(type);
        field.setValue(value);
        field.setSchema(schema);
        return field;
    }

    public static Field create(String label, String type, String value, String schema, Integer fieldAlignmentTypeEnum) {
        Field field = new Field();
        field.setLabel(label);
        field.setType(type);
        field.setValue(value);
        field.setSchema(schema);
        field.setAlignmentType(fieldAlignmentTypeEnum);
        return field;
    }

    public static Field create(String label, String value, FieldAlignmentTypeEnum fieldAlignmentTypeEnum, Boolean isComplete) {
        Field field = Field.create(label, "LABEL", value);
        if (null != fieldAlignmentTypeEnum) {
            field.setAlignmentType(fieldAlignmentTypeEnum.getValue());
        }
        field.setEnable(!isComplete);
        return field;
    }

    public static Field create(String label, String value, Integer labelStyleType, FieldAlignmentTypeEnum fieldAlignmentTypeEnum, Boolean isComplete) {
        Field field = Field.create(label, "LABEL", value);
        if (null != fieldAlignmentTypeEnum) {
            field.setAlignmentType(fieldAlignmentTypeEnum.getValue());
        }
        field.setLabelStyleType(labelStyleType);
        field.setEnable(!isComplete);
        return field;
    }

    public static Field create(String label, String value, Integer labelStyleType, FieldAlignmentTypeEnum fieldAlignmentTypeEnum, HashMap maps, Boolean isComplete) {
        Field field = Field.create(label, "LABEL", value);
        if (null != fieldAlignmentTypeEnum) {
            field.setAlignmentType(fieldAlignmentTypeEnum.getValue());
        }
        field.setLabelStyleType(labelStyleType);
        if (maps != null && maps.size() > 0) {
            Executor executor = new Executor();
            executor.setExecutorEmpId((String) maps.get("executorEmpId"));
            executor.setExecutorEmpName((String) maps.get("executorEmpName"));
            executor.setExecutorUserId((String) maps.get("executorUserId"));
            executor.setExecutorUserName((String) maps.get("executorUserName"));
            field.setExecutor(executor);
        }
        field.setEnable(!isComplete);
        return field;
    }

    public static Field create(UiBotTableColumn column, Map<String, Object> bizData) {
        return Field.create(column, column.getValueForDisplay(bizData));
    }

    public static Field create(UiBotTableColumn column, String value) {
        Field field = Field.create(column.getHeaderName(), column.getType(), value);
        field.setSchema(column.getSchema());

        return field;
    }

    public static Field create(UiBotTableColumn column, Map<String, Object> bizData, FieldAlignmentTypeEnum fieldAlignmentTypeEnum) {
        Field field = Field.create(column, column.getValueForDisplay(bizData));
        if (null != fieldAlignmentTypeEnum) {
            field.setAlignmentType(fieldAlignmentTypeEnum.getValue());
        }
        return field;
    }

    public static Field create(UiBotLayout fieldLayout, String value) {
        Field field = new Field();
        field.setSchema(fieldLayout.getSchema());
        field.setLabel(fieldLayout.getHeaderName());
        field.setType(fieldLayout.getType());
        field.setValue(value);
        return field;
    }

    public static Field create(ParseFieldData fieldData, String locale) {
        if (fieldData == null || fieldData.getData() == null || fieldData.getLabel() == null) {
            return null;
        }
        Field field = new Field();
        field.setLabel(fieldData.getLabel());
        field.setSchema(fieldData.getSchema());
        if ("TASK_PROGRESS_STATUS".equalsIgnoreCase(fieldData.getType())) {
            Map<String, Object> dataMap = (Map<String, Object>) fieldData.getData();
            //计划完成时间
            String planEndTime = (String) dataMap.get("planEndTime");
            //是否关闭（完成）
            Boolean close = (Boolean) dataMap.get("closed");
            //实际完成日期
            String endTime = (String) dataMap.get("endTime");

            if (close) {
                //已完成--正常完成、逾时完成
                if (DateTimeUtil.compareTime(endTime, planEndTime)) {
                    //实际完成日期早于计划完成日期--正常完成
                    field.setValue(LocaleUtil.getMobileTextByKey(locale, "已完成"));
                } else {
                    //实际完成日期晚于计划完成日期--逾时完成
                    //逾期天数
                    int days = DateTimeUtil.getDateTimePeriod(planEndTime, endTime);
                    field.setValue(String.format(LocaleUtil.getMobileTextByKey(locale, "逾期后完成"), days));
                }
            } else {
                //未完成
                if (!DateTimeUtil.isBeforeNowDateTime(planEndTime)) {
                    //未到计划完成日期--正在进行
                    field.setValue(LocaleUtil.getMobileTextByKey(locale, "正在进行"));
                } else {
                    //已到计划完成日期--逾时xx天
                    int days = DateTimeUtil.getDateTimePeriod(planEndTime, DateTimeUtil.getTodayTimeUseDefaultPattern());
                    field.setValue(String.format(LocaleUtil.getMobileTextByKey(locale, "逾期天数"), days));
                }
            }
        } else if (fieldData.getOptions() != null) {
            fieldData.getOptions().forEach(uiBotTableColumnEditorOption -> {
                if (fieldData.getData().equals(uiBotTableColumnEditorOption.getValue())) {
                    field.setValue(uiBotTableColumnEditorOption.getTitle());
                }
            });
        } else {
            field.setValue(StringUtils.hasLength(fieldData.getData().toString()) ? fieldData.getData().toString() : "-");
        }
        return field;
    }

    public static Field create(ParseFieldData fieldData, String locale, String value) {
        if (fieldData == null || fieldData.getData() == null || fieldData.getLabel() == null) {
            return null;
        }
        Field field = new Field();
        field.setLabel(localeService.getLanguageValue(locale, fieldData.getLabel()));
        field.setSchema(fieldData.getSchema());
        field.setValue(value);
        return field;
    }

    public static Field create(String label, Integer labelStyleType, Integer valueStyleType, String value) {
        Field field = new Field();
        field.setAlignmentType(FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_RIGHT.getValue());
        field.setLabel(label);
        field.setLabelStyleType(labelStyleType);
        field.setValueStyleType(valueStyleType);
        field.setValue(value);
        return field;
    }

    public static Field create(String label, String type, String value, Integer alignmentType, Boolean hasLabel) {
        Field field = new Field();
        field.setLabel(label);
        field.setType(type);
        field.setValue(value);
        field.setAlignmentType(alignmentType);
        field.setLabelStyleType(FieldLabelStyleTypeEnum.LABEL_GREY.getValue());
        if (!hasLabel) {
            field.setLabel("");
            field.setLabelVisible(hasLabel);
            field.setValueStyleType(FieldValueStyleTypeEnum.VALUE_BIG_TITLE.getValue());
        }

        return field;
    }

    public static Field create(String label, String value, Integer alignmentType, Integer valueStyleType, Boolean hidden) {
        Field field = new Field();
        field.setLabel(label);
        field.setValue(StringUtils.hasLength(value) ? value : "-");
        field.setAlignmentType(alignmentType);
        field.setValueStyleType(valueStyleType);
        field.setHidden(hidden);

        return field;
    }

    public static Field createZtbField(String label, Integer labelStyleType, Integer valueStyleType,
                                       String value, Integer alignmentType) {
        Field field = new Field();
        field.setAlignmentType(alignmentType);
        field.setLabel(label);
        field.setLabelStyleType(labelStyleType);
        field.setValueStyleType(valueStyleType);
        field.setValue(value);
        return field;
    }

    public static Field createApcField(String label, Integer labelStyleType, Integer valueStyleType,
                                       String value, Integer alignmentType, Boolean changeFlag) {
        Field field = new Field();
        field.setAlignmentType(alignmentType);
        field.setLabel(label);
        field.setLabelStyleType(labelStyleType);
        field.setValueStyleType(valueStyleType);
        field.setValue(value);
        field.setChangeFlag(changeFlag);
        return field;
    }

    public static Field createProcess(String label, String value) {
        Field field = new Field();
        field.setLabel(label);
        field.setLabelStyleType(FieldLabelStyleTypeEnum.LABEL_GREY.getValue());
        field.setValueStyleType(FieldValueStyleTypeEnum.VALUE_BLACK.getValue());
        field.setAlignmentType(FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_RIGHT.getValue());
        field.setValue(value);
        return field;
    }

    public String getLabel() {
        return label;
    }

    public void setLabel(String label) {
        this.label = label;
    }

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    public String getValue() {
        return value;
    }

    public void setValue(String value) {
        this.value = value;
    }

    @Override
    public String getSchema() {
        return schema;
    }

    @Override
    public void setSchema(String schema) {
        this.schema = schema;
    }

    public String getText() {
        return text;
    }

    public void setText(String text) {
        this.text = text;
    }

    public Integer getAlignmentType() {
        return alignmentType;
    }

    public void setAlignmentType(Integer alignmentType) {
        this.alignmentType = alignmentType;
    }

    public Integer getLabelStyleType() {
        return labelStyleType;
    }

    public void setLabelStyleType(Integer labelStyleType) {
        this.labelStyleType = labelStyleType;
    }

    public Integer getValueStyleType() {
        return valueStyleType;
    }

    public void setValueStyleType(Integer valueStyleType) {
        this.valueStyleType = valueStyleType;
    }

    @Override
    public String returnComponentType() {
        return Field.COMPONENT_TYPE;
    }

    public Boolean getHidden() {
        return hidden;
    }

    public void setHidden(Boolean hidden) {
        this.hidden = hidden;
    }

    public Boolean getEnable() {
        return enable;
    }

    public void setEnable(Boolean enable) {
        this.enable = enable;
    }

    public BaseMobileComponentPerson getPerson() {
        return person;
    }

    public void setPerson(BaseMobileComponentPerson person) {
        this.person = person;
    }

    public Boolean getLabelVisible() {
        return labelVisible;
    }

    public void setLabelVisible(Boolean labelVisible) {
        this.labelVisible = labelVisible;
    }

    public Executor getExecutor() {
        return executor;
    }

    public void setExecutor(Executor executor) {
        this.executor = executor;
    }

    public Action getAction() {
        return action;
    }

    public void setAction(Action action) {
        this.action = action;
    }

    public String getHideValue() {
        return hideValue;
    }

    public void setHideValue(String hideValue) {
        this.hideValue = hideValue;
    }

    @Override
    public void handleComponentParam(ComponentContext cmptContext, String mobilePath, String schema, Map<String, Object> data,
                                     PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        super.handleComponentParam(cmptContext, mobilePath, schema, data, executeContext, mobilePageRawData);
        String value = Optional.ofNullable(UiBotDesignerRenderService.getDataByPathAndSchema(data, mobilePath, schema))
                .map(Object::toString)
                .orElse(null);
        if (StringUtils.hasLength(value)) {
            if ("\n".equalsIgnoreCase(value)) {
                this.value = "-";
            } else {
                this.value = value;
            }
        } else {
            if (BooleanUtils.isTrue(this.enable)) {
                // 输入型，暂定无默认值
            } else {
                this.value = "-";
            }
        }
    }

    public static BaseMobileComponent cardContentConvertComponent(Card.DesignerRawCard.Content rawContent, ComponentContext cmptContext, Map<String, Object> data, PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        Field field = new Field();
        field.setComponentId(rawContent.getComponentId());
        field.setLabel(rawContent.getLabel());
        field.setEnable(false);
        field.setLabelStyleType(FieldLabelStyleTypeEnum.LABEL_GREY.getValue());
        field.setValueStyleType(FieldValueStyleTypeEnum.VALUE_GRAY.getValue());
        field.handleComponentParam(cmptContext, rawContent.getFullPath(), rawContent.getSchema(), data, executeContext, mobilePageRawData);
        return field;
    }

    @Override
    public void applyEditable(Boolean enable) {
        this.enable = enable;
    }

    @Override
    public void applyHidden(Boolean hidden) {
        this.hidden = hidden;
    }
}
