package com.digiwin.mobile.mobileuibot.core.component.business.bizprocess;

import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.core.columntag.ColumnTag;
import com.digiwin.mobile.mobileuibot.core.columntag.ColumnTagDefinitionCodeEnum;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponent;
import com.digiwin.mobile.mobileuibot.core.rule.Rule;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotActionSubmitType;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumn;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>功能描述：通用处理型任务明细数据列表</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: BizProcessDataList
 * @Author: Zaregoto
 * @Date: 2021/8/5 17:08
 */
public class BizProcessDataList extends BaseMobileComponent implements Serializable {
    public static final String COMPONENT_TYPE = "MULTI_LEVEL_SUBMITTABLE_LIST";
    private static final long serialVersionUID = 1380251415040439L;
    /**
     * 因事分组的数量最大值，大于它时会展示finishedText
     */
    private static final Integer MAX_THEME_SIZE_TO_USE_FINISHED_TEXT = 2;

    /**
     * 业务数据量的最大值，大于它时会展示finishedText
     */
    private static final Integer MAX_DATA_SIZE_TO_USE_FINISHED_TEXT = 2;

    /**
     * 分组处理数据时，主标题与副标题内容的分隔符
     * 注：为了防止数据本身与分隔符相同，才考虑使用emoji的unicode码
     */
    private static final String GROUPING_SPLITTER_TITLE = "\ue05a";
    /**
     * 分组副标题多value时的分隔符
     */
    private static final String GROUPING_SPLITTER_SUBTITLE = " ";
    private List<BizProcessDataItem> data;
    /**
     * 加载完成后的提示文字
     */
    private String finishedText = "";
    /**
     * 列表可被选择
     */
    private Boolean selectable;
    /**
     * toast提示信息，可放错误或提示信息
     */
    private String toastMsg;

    public BizProcessDataList() {
    }

    public static BizProcessDataList create(ApiRequest apiRequest,
                                            UiBotLayout bizProcessDataListLayout, List<Map<String, Object>> bizDataList,
                                            UiBotModel pcUiBotModel, List<UiBotTableColumn> tableColumns, List<ColumnTag> columnTagList,
                                            List<Rule> mobileRuleList) {
        String locale = apiRequest.getLocale();
        BizProcessDataList bizProcessDataList = new BizProcessDataList();
        /**
         * 判断是否可选择。
         * 数据支持分批提交时：selectable=true，反之selectable=false
         */
        boolean selectable = pcUiBotModel.getActions().stream().anyMatch(
                uiBotAction -> Optional.ofNullable(uiBotAction.getSubmitType())
                        .map(UiBotActionSubmitType::getIsBatch).orElse(false));
        bizProcessDataList.setSelectable(selectable);
        /**
         * 生成明细数据
         */
        // 查找标记为“因事分组”的主标题字段，如果没有找到，则使用数据中第一个只读的字段作为因事分组的主标题字段
        ColumnTag themeTitleTag;
        UiBotTableColumn themeTitleColumn;
        Optional<ColumnTag> optThemeTitleTag = columnTagList.stream().filter(
                ct -> ColumnTagDefinitionCodeEnum.DISPLAY_THEME_TITLE.getCode()
                        .equalsIgnoreCase(ct.getTagDefinition().getCode())).findFirst();
        if (!optThemeTitleTag.isPresent()) {
            themeTitleColumn =
                    tableColumns.stream().filter(uiBotTableColumn -> !uiBotTableColumn.canEdit())
                            .findFirst().orElse(null);
        } else {
            themeTitleTag = optThemeTitleTag.get();
            String themeTagSchema = themeTitleTag.getSchema();
            themeTitleColumn = tableColumns.stream()
                    .filter(col -> col.getSchema().equalsIgnoreCase(themeTagSchema)).findFirst()
                    .orElse(null);
        }
        // 查找标记为“因事分组”的副标题字段，如果没有找到则认为没有
        List<UiBotTableColumn> themeSubtitleColumnList = new ArrayList<>();
        List<ColumnTag> themeSubtitleColumnTagList = columnTagList.stream().filter(
                        ct -> ColumnTagDefinitionCodeEnum.DISPLAY_THEME_SUBTITLE.getCode()
                                .equalsIgnoreCase(ct.getTagDefinition().getCode()))
                .sorted(Comparator.comparingInt(ColumnTag::getOrderNo)).collect(Collectors.toList());
        themeSubtitleColumnTagList.forEach(ct -> {
            tableColumns.stream().filter(col -> col.getSchema().equalsIgnoreCase(ct.getSchema()))
                    .findFirst().ifPresent(themeSubtitleColumnList::add);
        });

        // 依找到栏位的schema分组后的结果。结果数据类型为map，key=栏位的分组值，value=相同分组的多笔数据
        Map<String, List<Map<String, Object>>> byThemeTagSchema =
                bizDataList.stream().collect(Collectors.groupingBy(bizData -> {
                    String themeTitleBizValue = themeTitleColumn.getValueForDisplay(bizData);
                    // 因事分组主字段一定会有，但副字段不一定会有，所以这里分组的值也会不同
                    if (themeSubtitleColumnList.isEmpty()) {
                        return themeTitleBizValue;
                    } else {
                        String themeSubtitleBizValueConcated = themeSubtitleColumnList.stream()
                                .map(column -> column.getValueForDisplay(bizData))
                                .collect(Collectors.joining(GROUPING_SPLITTER_SUBTITLE));
                        return themeTitleBizValue + GROUPING_SPLITTER_TITLE
                                + themeSubtitleBizValueConcated;
                    }
                }));

        // 根据不同的分组值生成不同的组内明细数据
        if (!byThemeTagSchema.isEmpty()) {
            for (String key : byThemeTagSchema.keySet()) {
                String[] splitKey = key.split(GROUPING_SPLITTER_TITLE);
                String title = splitKey[0];
                String subtitle = splitKey.length > 1 ? splitKey[1] : "";
                bizProcessDataList.getData().add(
                        BizProcessDataItem.create(apiRequest, bizProcessDataListLayout, title, subtitle,
                                pcUiBotModel, byThemeTagSchema.get(key), tableColumns, columnTagList,
                                mobileRuleList));
            }
        }

        /**
         * 处理加载完成时的文案
         * 分组数量>MAX_THEME_SIZE_TO_USE_FINISHED_TEXT 或 业务数据数量>MAX_DATA_SIZE_TO_USE_FINISHED_TEXT时，需要文案；
         * 反之不需要
         */
        if (bizProcessDataList.getData().size() > MAX_THEME_SIZE_TO_USE_FINISHED_TEXT
                || bizDataList.size() > MAX_DATA_SIZE_TO_USE_FINISHED_TEXT) {
            bizProcessDataList.setFinishedText(LocaleUtil.getMobileTextByKey(locale, "已经到底啦"));
        }

        // 提示信息放入错误提示
        bizProcessDataList.setToastMsg(LocaleUtil.getMobileTextByKey(locale, "还未勾选任何数据"));


        return bizProcessDataList;
    }

    public static BizProcessDataList createEmpty() {
        return new BizProcessDataList();
    }

    @Override
    public String returnComponentType() {
        return BizProcessDataList.COMPONENT_TYPE;
    }

    /**
     * 计算实际业务数据的条数
     *
     * @return
     */
    public int bizDataSize() {
        int sum = 0;
        if (null == this.data || this.data.isEmpty()) {
            return sum;
        }
        for (BizProcessDataItem dataItem : this.data) {
            if (null != dataItem.getChildren()) {
                int dataItemChildrenSize = dataItem.getChildren().size();
                sum += dataItemChildrenSize;
            }
        }
        return sum;
    }

    public Boolean getSelectable() {
        return selectable;
    }

    public void setSelectable(Boolean selectable) {
        this.selectable = selectable;
    }

    public String getFinishedText() {
        return finishedText;
    }

    public void setFinishedText(String finishedText) {
        this.finishedText = finishedText;
    }

    public List<BizProcessDataItem> getData() {
        if (null == this.data) {
            this.data = new ArrayList<>();
        }
        return this.data;
    }

    public void setData(List<BizProcessDataItem> data) {
        this.data = data;
    }

    public String getToastMsg() {
        return toastMsg;
    }

    public void setToastMsg(String toastMsg) {
        this.toastMsg = toastMsg;
    }
}
