package com.digiwin.mobile.mobileuibot.core.component.button;

import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.core.component.BaseGroupMobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponentWrapper;
import com.digiwin.mobile.mobileuibot.core.component.ComponentContext;
import com.digiwin.mobile.mobileuibot.core.component.MobilePageRawData;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.input.attachment.AttachmentTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.layout.ComponentLayoutTypeEnum;
import com.digiwin.mobile.mobileuibot.designer.uibot.enums.BusinessTypeEnum;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotActivityConstants;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotExecuteContext;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotAction;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotJumpAction;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.model.BaseDataStatusConfig;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.util.PcUiBotPatternUtil;
import com.digiwin.mobile.mobileuibot.proxy.uibot.pcservice.util.PcUiBotTagConstant;
import com.digiwin.mobile.mobileuibot.sso.enums.SsoSourceTypeEnum;
import lombok.Data;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.function.Predicate;

/**
 * <p>功能描述：按钮组</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: ButtonGroup
 * @Author: Zaregoto
 * @Date: 2023/2/9 12:08
 */
@Data
public class ButtonGroup extends BaseGroupMobileComponent<BottomButtonDigiwinAthena> {
    private static final long serialVersionUID = -6168875577301983184L;
    public static final String COMPONENT_TYPE = "BUTTON_GROUP";

    /**
     * 对齐方式 默認0(居中)、1(靠左)、2(靠右)
     * 提供前端按钮组放置位置，如：底部按钮居中、card组件中的按钮靠右
     *
     * @see com.digiwin.mobile.mobileuibot.core.component.button.ButtonGroupAlignEnum
     */
    private Integer alignmentType = ButtonGroupAlignEnum.CENTER.getValue();

    /**
     * 按钮组布局方式
     * 1:折叠 2:平铺
     *
     * @see ButtonLayoutTypeEnum
     */
    private Integer buttonLayoutType = ButtonLayoutTypeEnum.FOLD.getValue();

    @Override
    public String returnComponentType() {
        return ButtonGroup.COMPONENT_TYPE;
    }

    @Override
    public ComponentLayoutTypeEnum getComponentLayoutType() {
        return ComponentLayoutTypeEnum.GRID;
    }

    @Override
    public void handleComponentParam(ComponentContext cmptContext, String mobilePath, String schema, Map<String, Object> data, PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        // 根据配置处理隐藏按钮 -- 必须构建前处理
        this.handleHiddenButton(cmptContext, executeContext, mobilePageRawData.getExtendedFields(), data);
        super.handleComponentParam(cmptContext, mobilePath, schema, data, executeContext, mobilePageRawData);

        List<BaseMobileComponentWrapper<BottomButtonDigiwinAthena>> contentGroup = this.getContentGroup();
        if (!CollectionUtils.isEmpty(contentGroup)) {
            for (BaseMobileComponentWrapper<BottomButtonDigiwinAthena> wrapper : contentGroup) {
                Button<ButtonRawData> button = wrapper.getData();
                // 按钮组件
                button.handleComponentParam(cmptContext, wrapper.getMobilePath(), wrapper.getSchema(), data, executeContext, mobilePageRawData);
            }
        }
        // 构建全选按钮
        if (cmptContext.getCanBottomButton()) {
            //处理底部按钮超过3个显示不全 -- 必须构建后处理
            this.handleContentGroup(cmptContext);
        }
    }

    @Override
    public void applyEditable(Boolean enable) {
        if (BooleanUtils.isFalse(enable)) {
            this.handleHiddenButtonOfSubmit(submitAction -> BooleanUtils.isNotTrue(submitAction.getApplyAlthoughFinished()));
            this.handleHiddenButtonOfJump(submitAction -> ActionTypeEnum.RESET.getValue().equals(Optional.ofNullable(submitAction.getJumpAction()).map(UiBotJumpAction::getType).orElse(null)));
        }
    }

    /**
     * 处理底部按钮超过3个显示不全
     */
    private void handleContentGroup(ComponentContext cmptContext) {
        boolean addFlag = BooleanUtils.isTrue(cmptContext.getAddAllSelectButton());
        BaseMobileComponentWrapper<BottomButtonDigiwinAthena> allSelectWrapper = null;
        if (addFlag) {
            // 多选下，自动渲染全选按钮 --- 放在按钮组的左边
            BottomButtonDigiwinAthena allSelectButton = new BottomButtonDigiwinAthena();
            allSelectButton.setName(LocaleUtil.getMobileTextByKey(cmptContext.getLocale(), "全选"));
            allSelectButton.setType(BottomButtonStyleEnum.SELECT_ALL.getValue());
            allSelectWrapper = new BaseMobileComponentWrapper<>(allSelectButton, "DW_" + Button.COMPONENT_TYPE);
        }
        List<BaseMobileComponentWrapper<BottomButtonDigiwinAthena>> contentGroup = this.getContentGroup();
        if (addFlag) {
            if (CollectionUtils.isEmpty(contentGroup)) {
                contentGroup.add(allSelectWrapper);
            } else {
                contentGroup.add(0, allSelectWrapper);
            }
        }
    }

    /**
     * 根据状态配置 隐藏按钮
     *
     * @param executeContext
     * @param extendedFields
     * @param data
     */
    private void handleHiddenButton(ComponentContext cmptContext, PcUiBotExecuteContext executeContext, Map<String, Object> extendedFields, Map<String, Object> data) {
        if (!PcUiBotPatternUtil.isBaseData(executeContext.getPattern())) {
            // 不是基础数据录入或娜娜，返回
            return;
        }
        final String EDIT_TYPE_ADD = "add";
        if (EDIT_TYPE_ADD.equals(executeContext.getEditType())) {
            // 数据录入新增
            // 业务类型存在：隐藏非更新和保存类型按钮 -- 调用接口的按钮
            this.handleHiddenButtonOfSubmit(submitAction ->
                    org.springframework.util.StringUtils.hasLength(submitAction.getActionType()) &&
                            !(
                                    PcUiBotActivityConstants.BasicDataActionType.BASIC_DATA_UPDATE.equals(submitAction.getActionType())
                                            || PcUiBotActivityConstants.BasicDataActionType.BASIC_DATA_SAVE.equals(submitAction.getActionType())
                            )
            );
            // 隐藏本地删除按钮
            this.handleHiddenButtonOfJump(submitAction -> BusinessTypeEnum.LOCAL_DELETE.getValue().equals(Optional.ofNullable(submitAction.getJumpAction()).map(UiBotJumpAction::getBusinessType).orElse(null)));
            // 隐藏送审按钮
            this.handleHiddenButtonOfSubmit(submitAction -> StringUtils.isEmpty(submitAction.getActionType())
                    && "workflow-invoke".equals(submitAction.getActionServiceIdName()));
            // 隐藏撤审按钮
            this.handleHiddenButtonOfJump(submitAction -> StringUtils.isEmpty(submitAction.getActionType())
                    && "workflow-abort".equals(submitAction.getActionServiceIdName()));
            return;
        }

        // 数据录入修改
        BaseDataStatusConfig statusConfig = JsonUtil.objectToJavaObject(extendedFields, BaseDataStatusConfig.class);
        String manageStatus = this.getManageStatus(data);
        // 状态管理关闭下
        if (statusConfig == null || BooleanUtils.isNotTrue(statusConfig.getNeedStatusColumn())) {
            this.hiddenBottomButton(manageStatus, new ArrayList<>());
            return;
        }

        // 状态管理开启下
        List<String> ids = this.getShowButtonActionType(statusConfig, manageStatus);
        this.hiddenBottomButton(manageStatus, ids);
    }

    private void hiddenBottomButton(String manageStatus, List<String> ids) {
        // 业务类型存在：隐藏非删除、更新和保存类型按钮 -- 调用接口的按钮
        this.handleHiddenButtonOfSubmit(submitAction ->
                org.springframework.util.StringUtils.hasLength(submitAction.getActionType()) &&
                        !(
                                PcUiBotActivityConstants.BasicDataActionType.BASIC_DATA_UPDATE.equals(submitAction.getActionType())
                                        || PcUiBotActivityConstants.BasicDataActionType.BASIC_DATA_SAVE.equals(submitAction.getActionType())
                                        || PcUiBotActivityConstants.BasicDataActionType.BASIC_DATA_DELETE.equals(submitAction.getActionType())
                                        || ids.contains(submitAction.getActionType())
                        )
        );
        // 是否是模型驱动
        boolean isModelDrive = Optional.ofNullable(this.getContentGroup()).orElse(new ArrayList<>())
                .stream().anyMatch(content -> {
                    Button<ButtonRawData> button = content.getData();
                    UiBotAction submitAction = button.getSubmitAction();
                    if (submitAction == null) {
                        return false;
                    }
                    // 存在送审或撤审
                    return "workflow-invoke".equals(submitAction.getActionServiceIdName()) || "workflow-abort".equals(submitAction.getActionServiceIdName());
                });
        // 隐藏送审(manage_status!='N')按钮
        this.handleHiddenButtonOfSubmit(submitAction ->
                StringUtils.isEmpty(submitAction.getActionType()) &&
                        (
                                ("workflow-invoke".equals(submitAction.getActionServiceIdName()) && !"N".equals(manageStatus))
                        )
        );
        // 隐藏撤审(data?.manage_status!='I' && data?.manage_status!='Y')按钮
        this.handleHiddenButtonOfJump(submitAction ->
                StringUtils.isEmpty(submitAction.getActionType()) &&
                        (
                                ("workflow-abort".equals(submitAction.getActionServiceIdName()) && !"I".equals(manageStatus) && !"Y".equals(manageStatus))
                        )
        );
        if (isModelDrive) {
            // 隐藏保存(data.manage_status=='I' || data.manage_status=='Y')按钮
            // 隐藏删除(data.manage_status=='I' || data.manage_status=='Y')按钮
            this.handleHiddenButtonOfSubmit(submitAction ->
                    org.springframework.util.StringUtils.hasLength(submitAction.getActionType()) &&
                            (
                                    PcUiBotActivityConstants.BasicDataActionType.BASIC_DATA_UPDATE.equals(submitAction.getActionType())
                                            && ("I".equals(manageStatus) || "Y".equals(manageStatus))
                            )
            );
        }
    }

    private List<String> getShowButtonActionType(BaseDataStatusConfig statusConfig, String manageStatus) {
        List<String> ids = new ArrayList<>();
        if (StringUtils.isBlank(manageStatus)) {
            return ids;
        }
        Map<String, List<String>> showIdMap = statusConfig.getShowIdMap();
        if (CollectionUtils.isEmpty(showIdMap)) {
            return ids;
        }
        return Optional.ofNullable(showIdMap.get(manageStatus)).orElse(new ArrayList<>());
    }

    private String getManageStatus(Map<String, Object> data) {
        String manageStatus = "";
        if (data == null) {
            return manageStatus;
        }
        if (data.containsKey(PcUiBotTagConstant.MANAGE_STATUS)) {
            manageStatus = (String) data.get(PcUiBotTagConstant.MANAGE_STATUS);
            return manageStatus;
        }
        // 针对底部按钮处理
        for (Object v : data.values()) {
            if (v != null) {
                if (v instanceof Map) {
                    manageStatus = (String) ((Map<?, ?>) v).get(PcUiBotTagConstant.MANAGE_STATUS);
                } else if (v instanceof List) {
                    if (!CollectionUtils.isEmpty((Collection<?>) v)) {
                        Object o = ((List<?>) v).get(0);
                        if (o instanceof Map) {
                            manageStatus = (String) ((Map<?, ?>) o).get(PcUiBotTagConstant.MANAGE_STATUS);
                        }

                    }
                }
            }
        }
        return manageStatus;
    }

    private void handleHiddenButtonOfSubmit(Predicate<UiBotAction> predicate) {
        this.handleHiddenButton(submitAction -> ButtonTypeEnum.isNotJump(submitAction.getBtnType()) && predicate.test(submitAction));
    }

    private void handleHiddenButtonOfJump(Predicate<UiBotAction> predicate) {
        this.handleHiddenButton(submitAction -> ButtonTypeEnum.isJump(submitAction.getBtnType()) && predicate.test(submitAction));
    }

    private void handleHiddenButton(Predicate<UiBotAction> predicate) {
        List<BaseMobileComponentWrapper<BottomButtonDigiwinAthena>> contentGroup = this.getContentGroup();
        if (CollectionUtils.isEmpty(contentGroup)) {
            return;
        }
        contentGroup.removeIf(r -> {
            Button<ButtonRawData> button = r.getData();
            UiBotAction submitAction = button.getSubmitAction();
            if (submitAction == null) {
                return false;
            }
            return predicate.test(submitAction);
        });
    }
}
