package com.digiwin.mobile.mobileuibot.core.component.chart.chartcoordinate;

import com.digiwin.mobile.mobileuibot.core.component.chart.ChartTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.chart.chartcoordinate.axis.Axis;
import com.digiwin.mobile.mobileuibot.core.component.chart.chartcoordinate.axis.AxisTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.chart.chartcoordinate.axis.RectangleAxis;
import com.digiwin.mobile.mobileuibot.core.component.chart.chartelement.ChartElement;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.AgileDataIntermediateDataItem;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.DataObserveResult;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.chart.AgileDataChartFieldDisplayFormat;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.chart.AgileDataChartPointField;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.chart.AgileDataChartValueField;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.chart.ChartRenderSetting;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author wuyang
 * @desc 图表的坐标系
 */
@Slf4j
@Data
public abstract class ChartCoordinate extends ChartElement {

    private static final long serialVersionUID = 5534062764572878832L;
    /**
     * 坐标系类型：直角坐标(rectangle)、极坐标(polar)、无坐标（none）
     *
     * @see ChartCoordinateCategoryEnum
     */
    protected String category;
    /**
     * 坐标系中心在所呈现画面的像素位置
     */
    protected List<Double> centerPixel;
    /**
     * 坐标系中心在所呈现画面的百分百位置
     */
    protected List<String> centerPercent;

    public ChartCoordinate() {
        this.initCategory();
    }

    protected void initCategory() {
    }

    public void addXAxis(RectangleAxis axis) {
    }

    public void addMultipleXAxis(List<RectangleAxis> axes) {
    }

    public void addYAxis(RectangleAxis axis) {
    }

    public void addMultipleYAxis(List<RectangleAxis> axes) {
    }

    public void setRadiusAxis(Axis axis) {
    }

    public void setAngleAxis(Axis axis) {
    }

    /**
     * 根据一个中间态DSL的数据描述，创建一个移动端图表组件的坐标轴元素<br/>
     * 未来随着组件类挪入athenamobile-uibot-component模块，该方法要转到AgileDataIntermediateDataItem对象内
     *
     * @param intermediateDataItem
     * @return
     */
    public static ChartCoordinate buildChartCoordinate(AgileDataIntermediateDataItem intermediateDataItem, ChartTypeEnum chartTypeEnum) {
        ChartRenderSetting chartRenderSetting = intermediateDataItem.getChartRenderSetting();
        // 获取维度字段
        List<AgileDataChartPointField> points = chartRenderSetting.getPoints();
        // 获取统计字段
        List<AgileDataChartValueField> values = chartRenderSetting.getValues();

        ChartCoordinate coordinate;
        if (ChartTypeEnum.LINE == chartTypeEnum || ChartTypeEnum.BAR == chartTypeEnum || ChartTypeEnum.LINE_BAR == chartTypeEnum || ChartTypeEnum.MIXED == chartTypeEnum) {
            coordinate = new RectangleCoordinate();
            // 设置X坐标轴
            List<RectangleAxis> xAxes = points
                    .stream().map(point -> {
                        RectangleAxis xAxis = new RectangleAxis(AxisTypeEnum.CATEGORY.getValue());
                        xAxis.setName(point.getTitle());
                        // X轴的刻度数据
                        List<String> xAxisData = intermediateDataItem.getSeriesDataWithFieldId(point.getName());
                        xAxis.setData(xAxisData);

                        return xAxis;
                    }).collect(Collectors.toList());
            // 设置Y坐标轴。使用统计字段的group来明确Y轴的分组
            Map<Integer, List<AgileDataChartValueField>> groupedValueFields = intermediateDataItem.getGroupedValueFields();
            List<RectangleAxis> yAxes = new ArrayList<>(2);
            for (Integer groupNo : groupedValueFields.keySet()) {
                RectangleAxis yAxis = new RectangleAxis(AxisTypeEnum.VALUE.getValue());
                // 设置Y轴最小、最大值、刻度间距，以及刻度文字
                DataObserveResult observeResult = intermediateDataItem.getExactDataObserveResult(groupNo);
                List<String> yData = observeResult.calculatePreferredAxisDatas();
                setMinAndMaxValue(yData, yAxis);
                yAxis.setInterval(observeResult.getPreferredAxisInterval());
                yAxis.setData(yData);

                yAxes.add(yAxis);
            }
            coordinate.addMultipleXAxis(xAxes);
            coordinate.addMultipleYAxis(yAxes);

        } else if (chartTypeEnum == ChartTypeEnum.PIE) {
            coordinate = new NoneCoordinate();
        } else {
            // 没有匹配到
            coordinate = new NoneCoordinate();
        }

        return coordinate;
    }

    /**
     * 设置Y轴的最小值和最大值
     *
     * @param yData
     * @param yAxis
     */
    private static void setMinAndMaxValue(List<String> yData, RectangleAxis yAxis) {
        if (CollectionUtils.isEmpty(yData)) {
            return;
        }
        try {
            yAxis.setMin(parseValue(yData.get(0)));
            yAxis.setMax(parseValue(yData.get(yData.size() - 1)));
        } catch (NumberFormatException e) {
            // 处理解析错误
            log.error("Error parsing yData values: {}", e.getMessage());
        }
    }

    /**
     * 解析数值字符串，支持带%的百分比值
     * (可见AgileDataChartFieldDisplayFormat类找到定义来理解)
     *
     * @param value
     * @return
     * @see AgileDataChartFieldDisplayFormat
     */
    private static double parseValue(String value) {
        if (value.contains(",")) {
            value = value.replaceAll(",", "");
        } else if (value.contains("%")) {
            value = value.replaceAll("%", "");
            return Double.parseDouble(value) / 100;
        }
        return Double.parseDouble(value);
    }


}