package com.digiwin.mobile.mobileuibot.core.component.chart.chartseries;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.annotation.JSONField;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.core.component.chart.ChartTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.chart.chartelement.markline.ChartMarkLine;
import com.digiwin.mobile.mobileuibot.core.component.chart.chartstyle.ChartColorPalette;
import com.digiwin.mobile.mobileuibot.core.component.chart.chartstyle.ChartElementStyle;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.AgileDataIntermediateDataItem;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.chart.AgileDataChartPointField;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.chart.AgileDataChartValueField;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.chart.ChartRenderSetting;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.chart.measure.*;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.Accessors;
import org.springframework.util.CollectionUtils;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author wuyang
 * @desc 数据系列
 */
@Data
@NoArgsConstructor
@AllArgsConstructor
@Accessors(chain = true)
public class ChartSeries<T extends ChartData> implements Serializable {

    private static final long serialVersionUID = -7824082283273768445L;

    /**
     * 当前数据系列对应的字段信息。不参与序列化和反序列化，仅在处理时做传参用
     * 如：PAYMENT_AMOUNT
     */
    @JsonIgnore
    @JSONField(serialize = false, deserialize = false)
    private List<AgileDataChartValueField> valueFields = new ArrayList<>(2);

    /**
     * 待展示的图标类型（有且仅当Chart.type=mix时有值，且需要前端渲染）
     * 取值范围：参考 ChartTypeEnum   不包括mix
     *
     * @see com.digiwin.mobile.mobileuibot.core.component.chart.ChartTypeEnum
     */
    private String type;
    /**
     * 数据系列名称
     */
    private String name;
    /**
     * 数据
     */
    private List<T> data = new ArrayList<>();
    /**
     * 数据系列使用的X轴索引（出现2个X轴时使用）
     */
    @JSONField(name = "xAxisIndex")
    @JsonProperty(value = "xAxisIndex")
    private Integer xAxisIndex;
    /**
     * 数据系列使用的Y轴索引（出现2个Y轴时使用）
     */
    @JSONField(name = "yAxisIndex")
    @JsonProperty(value = "yAxisIndex")
    private Integer yAxisIndex;

    /**
     * 图表标线
     */
    private ChartMarkLine markLine;

    /**
     * 线条样式
     */
    private ChartElementStyle lineStyle;

    public void setLineStyle(ChartElementStyle lineStyle) {
        this.lineStyle = lineStyle;
    }

    public void addData(T chartData) {
        this.data.add(chartData);
    }

    public void addDataAll(List<T> chartData) {
        this.data.addAll(chartData);
    }

    /**
     * 添加一个度量字段信息
     *
     * @param valueField
     */
    public void addValueField(AgileDataChartValueField valueField) {
        this.valueFields.add(valueField);
    }

    /**
     * 根据一个中间态DSL的数据描述，创建折线图的数据系列
     *
     * @param locale               语言别
     * @param intermediateDataItem 中间态DSL的数据描述
     * @param chartTypeEnum        图表类型枚举
     * @return
     * @see RectangleCoordinateChartData
     * @see ChartTypeEnum
     */
    public static List<ChartSeries<RectangleCoordinateChartData>> buildLineChartSeries(
            String locale, AgileDataIntermediateDataItem intermediateDataItem, ChartTypeEnum chartTypeEnum) {
        ChartRenderSetting chartRenderSetting = intermediateDataItem.getChartRenderSetting();
        List<AgileDataChartPointField> points = chartRenderSetting.getPoints();
        List<AgileDataChartValueField> values = chartRenderSetting.getValues();

        List<JSONObject> pageData = intermediateDataItem.getPageData();

        // FIXME 需要有判断条件判断数据使用堆叠方式呈现，截止11.27中间态数据还没考虑到
        List<ChartSeries<RectangleCoordinateChartData>> dataSeries = new ArrayList<>();

        int colorIdx = 0;
        // 维度轴
        for (int i = 0; i < points.size(); i++) {
            AgileDataChartPointField pointField = points.get(i);
            // 统计轴
            for (int j = 0; j < values.size(); j++) {
                AgileDataChartValueField valueField = values.get(j);

                String lineColor = ChartColorPalette.getColorFromCategoryPalette(colorIdx, values.size());
                String dataColor = lineColor;

                ChartSeries<RectangleCoordinateChartData> series = new ChartSeries<>();
                series.setName(valueField.getTitle());
                series.setXAxisIndex(i);
                // 设置数据系列使用的Y轴索引
                series.setYAxisIndex(valueField.getGroup());
                series.setType(chartTypeEnum.getType());
                series.setLineStyle(new ChartElementStyle().setColor(lineColor));
                // 设置图表标线
                series.setMarkLine(ChartMarkLine.buildLineChartMarkLine(intermediateDataItem));

                for (JSONObject pageDatum : pageData) {
                    List<OneDimensionalChartData> oneDimDataList = new ArrayList<>();
                    OneDimensionalChartData oneDimData = new OneDimensionalChartData()
                            .setValue(pageDatum.getDoubleValue(valueField.getName()))
                            .setLabel(intermediateDataItem.getLabelWithValueField(locale, valueField,
                                    pageDatum.getDoubleValue(valueField.getName())))
                            .setShowLabel(false)
                            .setDimensionData(valueField.getDimensionData(pageDatum))
                            .setMeasurementData(valueField.getMeasurementData(pageDatum));
                    oneDimData.setColor(dataColor);
                    oneDimDataList.add(oneDimData);

                    RectangleCoordinateChartData data = new RectangleCoordinateChartData()
                            .setX(pageDatum.getString(pointField.getName()))
                            .setY(oneDimDataList);
                    series.addData(data);
                }
                dataSeries.add(series);
                colorIdx++;
            }
        }
        return dataSeries;
    }

    /**
     * 根据一个中间态DSL的数据描述，创建柱状图的数据系列
     *
     * @param locale               语言别
     * @param intermediateDataItem 中间态DSL的数据描述
     * @param chartTypeEnum        图表类型枚举
     * @return
     * @see RectangleCoordinateChartData
     * @see ChartTypeEnum
     */
    public static List<ChartSeries<RectangleCoordinateChartData>> buildBarChartSeries(
            String locale, AgileDataIntermediateDataItem intermediateDataItem, ChartTypeEnum chartTypeEnum) {
        ChartRenderSetting chartRenderSetting = intermediateDataItem.getChartRenderSetting();
        List<AgileDataChartPointField> points = chartRenderSetting.getPoints();
        List<AgileDataChartValueField> values = chartRenderSetting.getValues();
//        List<AgileDataChartGroupByField> groupBys = chartRenderSetting.getGroupBys();

        List<JSONObject> pageData = intermediateDataItem.getPageData();

        // 依公式ID分组后的临时数据
        Map<String, List<MeasurePresent>> formulationIdGroupedMap = null;
        if (null != chartRenderSetting.getMeasurePresents()) {
            formulationIdGroupedMap = chartRenderSetting.getMeasurePresents().
                    stream().collect(
                            // 指定使用有序Map和Collection，保存分组内的结果，保证分组顺序和原始数据中的顺序相同
                            Collectors.groupingBy(mp -> mp.getFormulationId(),
                                    LinkedHashMap::new,
                                    Collectors.toCollection(ArrayList::new)));
        }
        // FIXME 需要有判断条件判断数据使用堆叠方式呈现，截止11.27中间态数据还没考虑到

//        // 若是多柱（多簇）的情况，需要先将数据按维度字段形成预处理数据分组清单（维度字段数量取1-2闭区间，有两个时表示双X轴）
//        // 然后每个分组中的数据清单再按groupBys的字段做排序
//        List<BarChartPreprocessDataByPoint> pointCategorizedData = new ArrayList<>(points.size());
//        for (AgileDataChartPointField point : points) {
//            Map<String, List<JSONObject>> groupedPointValues = pageData
//                    .stream().collect(
//                            // 指定使用有序Map和Collection，保存分组内的结果，保证分组顺序和原始数据中的顺序相同
//                            Collectors.groupingBy(jsonObject -> jsonObject.getString(point.getName()),
//                                    LinkedHashMap::new,
//                                    Collectors.toCollection(ArrayList::new)));
//            // FIXME 确定多柱的数据系列排序。当前没有从中间态DSL的图谱中获取，未来应该会需要
//            pointCategorizedData.add(new BarChartPreprocessDataByPoint()
//                    .setFieldId(point.getName())
//                    .setFieldCategorizedValues(groupedPointValues)
//            );
//        }
//
//        List<ChartSeries<RectangleCoordinateChartData>> dataSeries = new ArrayList<>();
//
//        // 外循环：不同维度字段，表示不同X轴，加到不同series
//        for (int i = 0; i < pointCategorizedData.size(); i++) {
//            BarChartPreprocessDataByPoint barChartPreprocessDataByPoint = pointCategorizedData.get(i);
//
//            ChartSeries<RectangleCoordinateChartData> series = new ChartSeries<>();
//            series.setXAxisIndex(i);
//            series.setType(ChartTypeEnum.BAR.getType());
//            // 柱状图因前端实现的原因，同一个X轴下，无论多少个柱，都使用一个series，不同的数据放在内部data的y数组中
//            dataSeries.add(series);
//
//            Map<String, List<JSONObject>> fieldCategorizedValues = barChartPreprocessDataByPoint.getFieldCategorizedValues();
//            for (String x : fieldCategorizedValues.keySet()) {
//                RectangleCoordinateChartData chartData = new RectangleCoordinateChartData();
//                chartData.setX(x);
//
//                series.addData(chartData);
//                List<JSONObject> yList = fieldCategorizedValues.get(x);
//
//                List<OneDimensionalChartData> oneDimDataList = new ArrayList<>();
//                chartData.setY(oneDimDataList);
//
//                int colorIdx = 0;
//                // 统计数据
//                for (int j = 0; j < values.size(); j++) {
//                    AgileDataChartValueField valueField = values.get(j);
//                    for (int k = 0; k < yList.size(); k++) {
//                        JSONObject yObj = yList.get(k);
//                        // FIXME 堆叠展示的中间态数据还没确定
//                        String dataColor = ChartColorPalette.getColorFromCategoryPalette(colorIdx);
//                        colorIdx++;
//                        OneDimensionalChartData oneDimData = new OneDimensionalChartData()
//                                .setValue(yObj.getDoubleValue(valueField.getName()))
//                                .setLabel(intermediateDataItem.getLabelWithValueField(locale, valueField,
//                                        yObj.getDoubleValue(valueField.getName())))
//                                .setShowLabel(false)
//                                .setYAxisIndex(j);
//                        oneDimData.setName(valueField.getTitle());
//
//                        oneDimData.setColor(dataColor);
//                        oneDimDataList.add(oneDimData);
//                    }
//                }
//            }
//        }

        List<ChartSeries<RectangleCoordinateChartData>> dataSeries = new ArrayList<>();
        for (int i = 0; i < points.size(); i++) {
            AgileDataChartPointField pointField = points.get(i);

            ChartSeries<RectangleCoordinateChartData> series = new ChartSeries<>();
            // 存储Y轴字段使用的颜色字符串
            Map<String, String> valueFieldColorMap = new HashMap<>(values.size());

            for (JSONObject pageDatum : pageData) {
                int colorIdx = 0;

                RectangleCoordinateChartData chartData = new RectangleCoordinateChartData();
                series.addData(chartData);

                chartData.setX(pageDatum.getString(pointField.getName()));

                List<OneDimensionalChartData> oneDimDataList = new ArrayList<>();
                chartData.setY(oneDimDataList);

                for (AgileDataChartValueField valueField : values) {
                    String dataColor = ChartColorPalette.getColorFromCategoryPalette(colorIdx, values.size());
                    colorIdx++;

                    valueFieldColorMap.put(valueField.getName(), dataColor);

                    OneDimensionalChartData oneDimData = new OneDimensionalChartData()
                            .setValue(pageDatum.getDoubleValue(valueField.getName()))
                            .setLabel(intermediateDataItem.getLabelWithValueField(locale, valueField,
                                    pageDatum.getDoubleValue(valueField.getName())))
                            .setShowLabel(false)
                            // 设置数据系列使用的Y轴索引
                            .setYAxisIndex(valueField.getGroup())
                            .setDimensionData(valueField.getDimensionData(pageDatum))
                            .setMeasurementData(valueField.getMeasurementData(pageDatum));
                    oneDimData.setName(valueField.getTitle());
                    oneDimData.setColor(dataColor);

                    oneDimDataList.add(oneDimData);
                }
                // Y轴（度量）呈现增强
                if (!CollectionUtils.isEmpty(formulationIdGroupedMap)) {
                    // 差值标签
                    if (formulationIdGroupedMap.containsKey(MeasurePresetnFormulationIdEnum.GROWTH_RATE.getValue())) {
                        YDifferLabel yDifferLabel = new YDifferLabel();
                        chartData.setYDifferLabel(yDifferLabel);

                        GrowthRateInterpreter growthRateInterpreter = new GrowthRateInterpreter();
                        List<MeasurePresent> measurePresents = formulationIdGroupedMap.get(MeasurePresetnFormulationIdEnum.GROWTH_RATE.getValue());
                        measurePresents.forEach(mp -> {
                            GrowthRateInterpreterResult result =
                                    growthRateInterpreter.interpret(pageDatum, values, (GrowthRateMeasurePresent) mp);
                            if (null != result) {
                                YDifferLabelDataItem yDifferLabelDataItem = new YDifferLabelDataItem(mp.getFormat())
                                        .setDifferRate(result.getDifferRate())
                                        .setStartIndex(result.getStartIndex())
                                        .setEndIndex(result.getEndIndex());
                                yDifferLabel.addYDifferLabelDataItem(yDifferLabelDataItem);
                            }
                        });
                    }
                }
            }

            series.setXAxisIndex(i);
            series.setType(chartTypeEnum.getType());
            // 设置图表标线
            series.setMarkLine(ChartMarkLine.buildBarChartMarkLine(intermediateDataItem, valueFieldColorMap));

            // 柱状图因前端实现的原因，同一个X轴下，无论多少个柱，都使用一个series，不同的数据放在内部data的y数组中
            dataSeries.add(series);
        }

        return dataSeries;
    }

    /**
     * 根据一个中间态DSL的数据描述，创建混合图的数据系列
     *
     * @param locale               语言别
     * @param intermediateDataItem 中间态DSL的数据描述
     * @param chartTypeEnum        图表类型枚举
     * @return
     * @see RectangleCoordinateChartData
     * @see ChartTypeEnum
     */
    public static List<ChartSeries<RectangleCoordinateChartData>> buildMixedChartSeries(
            String locale, AgileDataIntermediateDataItem intermediateDataItem, ChartTypeEnum chartTypeEnum) {
        ChartRenderSetting chartRenderSetting = intermediateDataItem.getChartRenderSetting();
        List<AgileDataChartPointField> points = chartRenderSetting.getPoints();
        List<AgileDataChartValueField> values = chartRenderSetting.getValues();
        Map<Integer, List<AgileDataChartValueField>> groupedValueFields = intermediateDataItem.getGroupedValueFields();
        List<JSONObject> pageData = intermediateDataItem.getPageData();

        // 提前设置每个度量字段的颜色索引
        for (int i = 0; i < values.size(); i++) {
            AgileDataChartValueField valueField = values.get(i);
            valueField.setColorIdx(i);
        }

        // FIXME 需要有判断条件判断数据使用堆叠方式呈现，截止11.27中间态数据还没考虑到
        List<ChartSeries<RectangleCoordinateChartData>> dataSeries = new ArrayList<>();
        // 先根据字段分组情况，判断有多少series，并提前设置好每个series对应的字段名称
        // 如果已经有chartType=BAR的series，则后续遇到chartType=BAR的度量字段，则不再添加series
        boolean hasBarTypeSeries = false;
        for (Integer groupNo : groupedValueFields.keySet()) {
            List<AgileDataChartValueField> valueFields = groupedValueFields.get(groupNo);
            for (int i = 0; i < valueFields.size(); i++) {
                AgileDataChartValueField currentValueField = valueFields.get(i);
                String currentChartType = currentValueField.getChartType();
                if (hasBarTypeSeries && (ChartTypeEnum.BAR.getType().equalsIgnoreCase(currentChartType) || ChartTypeEnum.LINE_BAR.getType().equalsIgnoreCase(currentChartType))) {
                    ChartSeries<RectangleCoordinateChartData> barSeries =
                            dataSeries.stream()
                                    .filter(s -> s.getType().equals(ChartTypeEnum.BAR.getType())
                                            || s.getType().equals(ChartTypeEnum.LINE_BAR.getType()))
                                    .findFirst()
                                    .orElseThrow(() -> new ServiceException("bar type series lost..."));
                    barSeries.addValueField(currentValueField);
                } else {
                    ChartSeries<RectangleCoordinateChartData> series = new ChartSeries<>();
                    if (!ChartTypeEnum.BAR.getType().equals(currentChartType)) {
                        if (!ChartTypeEnum.LINE_BAR.getType().equals(currentChartType)) {
                            series.setYAxisIndex(groupNo);
                        }
                        
                    }
                    // TODO 数据系列的名称不一定就是度量字段的名称
//                    series.setName(currentValueField.getTitle());
                    series.setType(currentChartType);
                    series.addValueField(currentValueField);
                    dataSeries.add(series);
                    hasBarTypeSeries = ChartTypeEnum.BAR.getType().equals(currentChartType) || ChartTypeEnum.LINE_BAR.getType().equals(currentChartType);
                }
            }
        }

        // 维度轴
        for (int i = 0; i < points.size(); i++) {
            AgileDataChartPointField pointField = points.get(i);
            String lineColor = "";
            String dataColor = "";
            for (int j = 0; j < dataSeries.size(); j++) {
                ChartSeries<RectangleCoordinateChartData> series = dataSeries.get(j);
                boolean specifiedChartTypeIsBar = false;
                if (Objects.equals(ChartTypeEnum.BAR.getType(), series.getType()) || Objects.equals(ChartTypeEnum.LINE_BAR.getType(), series.getType())) {
                    specifiedChartTypeIsBar = true;
                }
                // FIXME 堆叠展示的中间态数据还没确定
                for (JSONObject pageDatum : pageData) {
                    List<AgileDataChartValueField> valueFields = series.getValueFields();
                    List<OneDimensionalChartData> oneDimDataList = new ArrayList<>();
                    for (int k = 0; k < valueFields.size(); k++) {
                        AgileDataChartValueField valueField = valueFields.get(k);
                        int colorIdx = valueField.getColorIdx();
                        OneDimensionalChartData oneDimData = new OneDimensionalChartData()
                                .setValue(pageDatum.getDoubleValue(valueField.getName()))
                                .setLabel(intermediateDataItem.getLabelWithValueField(locale, valueField,
                                        pageDatum.getDoubleValue(valueField.getName())))
                                .setShowLabel(false)
                                .setDimensionData(valueField.getDimensionData(pageDatum))
                                .setMeasurementData(valueField.getMeasurementData(pageDatum));
                        // 因柱状图的数据结构是以series中的Y轴数组来声明数据的，所以需在多柱状图时设计数据使用的Y轴索引。索引编号同中间态DSL中的编号。
                        if (specifiedChartTypeIsBar) {
                            oneDimData.setYAxisIndex(valueField.getGroup());
                        }
                        lineColor = ChartColorPalette.getColorFromCategoryPalette(colorIdx, valueFields.size());
                        dataColor = ChartColorPalette.getColorFromCategoryPalette(colorIdx, valueFields.size());

                        oneDimData.setName(valueField.getTitle());
                        oneDimData.setColor(dataColor);
                        oneDimDataList.add(oneDimData);
                    }
                    RectangleCoordinateChartData data = new RectangleCoordinateChartData()
                            .setX(pageDatum.getString(pointField.getName()))
                            .setY(oneDimDataList);
                    series.addData(data);
                }
                if (Objects.equals(ChartTypeEnum.LINE.getType(), series.getType())) {
                    series.setLineStyle(new ChartElementStyle().setColor(lineColor));
                }
            }
        }
        return dataSeries;
    }

    /**
     * 根据一个中间态DSL的数据描述，创建饼图的数据系列
     *
     * @param locale               语言别
     * @param intermediateDataItem 中间态DSL的数据描述
     * @return
     * @see RectangleCoordinateChartData
     * @see ChartTypeEnum
     */
    public static List<ChartSeries<PieChartData>> buildPieChartSeries(String locale,
                                                                      AgileDataIntermediateDataItem intermediateDataItem) {
        ChartRenderSetting chartRenderSetting = intermediateDataItem.getChartRenderSetting();
        List<AgileDataChartPointField> points = chartRenderSetting.getPoints();
        List<AgileDataChartValueField> values = chartRenderSetting.getValues();
        List<JSONObject> pageData = intermediateDataItem.getPageData();

        List<ChartSeries<PieChartData>> dataSeries = new ArrayList<>();
        // 维度轴
        for (int i = 0; i < points.size(); i++) {
            // 统计轴
            for (int j = 0; j < values.size(); j++) {
                AgileDataChartPointField pointField = points.get(i);
                AgileDataChartValueField valueField = values.get(j);
                ChartSeries<PieChartData> series = new ChartSeries<>();
                series.setName(valueField.getTitle());
                for (int k = 0; k < pageData.size(); k++) {
                    JSONObject pageDatum = pageData.get(k);
                    PieChartData data = new PieChartData()
                            .setValue(pageDatum.getDoubleValue(valueField.getName()))
                            .setLabel(pageDatum.getString(pointField.getName()))
                            .setShowLabel(true)
                            .setDimensionData(valueField.getDimensionData(pageDatum))
                            .setMeasurementData(valueField.getMeasurementData(pageDatum));
                    data.setColor(ChartColorPalette.getColorFromCategoryPalette(k, pageData.size()));
                    series.addData(data);
                }
                dataSeries.add(series);
            }
        }

        return dataSeries;
    }

}
