package com.digiwin.mobile.mobileuibot.core.component.chart.chartstyle;

import lombok.Data;

import java.io.Serializable;

/**
 * <p>功能描述：图表色板</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: ChartColorPalette
 * @Author: zaregoto
 * @Date: 2023/11/28 14:18
 */
@Data
public final class ChartColorPalette implements Serializable {
    private static final long serialVersionUID = 2461579055960450047L;

    /**
     * 类目色板（N<=10）：与设计规范一致的10色
     */
    private static final String[] CATEGORY_PALETTE_10 = {
            "#7D7DE3", "#71B6FB", "#B9DFFD", "#4CD4D0", "#76D285",
            "#F9D978", "#FFB361", "#F77D79", "#B488E3", "#E87DE1"
    };

    /**
     * 类目色板（10<N<=20）：与设计规范一致的20色
     */
    private static final String[] CATEGORY_PALETTE_20 = {
            // 第一行（左→右）
            "#7D7DE3", "#B5C0FB", "#71B6FB", "#B9DFFD", "#62D4D0",
            "#A1E9E2", "#76D285", "#92E19C", "#F7C84F", "#F9D978",
            // 第二行（左→右）
            "#FAE287", "#FFA647", "#FFC175", "#F77D79", "#F9A19A",
            "#A57ADA", "#C09FE3", "#CEADED", "#E894E2", "#F0BEEB"
    };

    public class GlobalColorPalette {
        /**
         * 背景色
         */
        public static final String BACKGROUND_COLOR = "#FFFFFF";

        /**
         * 标题文字色
         */
        public static final String TITLE_COLOR = "#383878";

        /**
         * 副标题文字色
         */
        public static final String SUBTITLE_COLOR = "#9995BD";

        /**
         * 标签文字色
         */
        public static final String LABEL_TEXT_COLOR = "#FFFFFF";

        /**
         * 坐标轴线色
         */
        public static final String AXIS_LINE_COLOR = "#D9D9D9";

        /**
         * 数据标注线色
         */
        public static final String MARK_LINE_COLOR = "#E8E8E8";

        /**
         * 坐标轴刻度色
         */
        public static final String SCALE_POINT_COLOR = "#D9D9D9";

        /**
         * 图例名称文字色
         */
        public static final String LEGEND_NAME_COLOR = "#666666";

        /**
         * 一般文字颜色
         */
        public static final String GENERAL_TEXT_COLOR = "#666666";

        /**
         * 一般可点击文字颜色
         */
        public static final String GENERAL_TOUCHABLE_TEXT_COLOR = "#6A4CFF";

    }

    /**
     * 根据输入的索引与总数量，获取色板上的颜色。
     * 规则：
     * - N<=10 使用10色固定色板；
     * - 10<N<=20 使用20色固定色板；
     * - N>20 使用HSL自动生成色（H步进30，S步进15，L步进10）。
     *
     * @param dataIndex   第几个（从0开始）
     * @param totalCount  总数量N
     * @return 颜色HEX字符串，如 #RRGGBB
     */
    public static String getColorFromCategoryPalette(int dataIndex, int totalCount) {
        if (totalCount <= 10) {
            int idx = mod(dataIndex, CATEGORY_PALETTE_10.length);
            return CATEGORY_PALETTE_10[idx];
        } else if (totalCount <= 20) {
            int idx = mod(dataIndex, CATEGORY_PALETTE_20.length);
            return CATEGORY_PALETTE_20[idx];
        } else {
            return generateAutoColor(dataIndex);
        }
    }

    // =============== 自动配色：HSL 网格生成 ===============

    private static final int[] H_STEPS = buildRange(0, 330, 30);   // 12 个
    private static final int[] S_STEPS = buildRange(15, 65, 10);   // 6 个
    private static final int[] L_STEPS = buildRange(30, 70, 10);   // 5 个

    /**
     * 基于网格按索引生成颜色；当索引超过 360 时循环。
     */
    private static String generateAutoColor(int index) {
        int total = H_STEPS.length * S_STEPS.length * L_STEPS.length; // 360
        int k = mod(index, total);
        int hIdx = k % H_STEPS.length;
        int sIdx = (k / H_STEPS.length) % S_STEPS.length;
        int lIdx = (k / (H_STEPS.length * S_STEPS.length)) % L_STEPS.length;

        double h = H_STEPS[hIdx];
        double s = S_STEPS[sIdx] / 100.0; // 0-1
        double l = L_STEPS[lIdx] / 100.0; // 0-1
        return hslToHex(h, s, l);
    }

    private static int[] buildRange(int start, int end, int step) {
        int size = (end - start) / step + 1;
        int[] arr = new int[size];
        int v = start;
        for (int i = 0; i < size; i++) {
            arr[i] = v;
            v += step;
        }
        return arr;
    }

    private static int mod(int a, int b) {
        int m = a % b;
        return m < 0 ? m + b : m;
    }

    /**
     * HSL 转 HEX（s,l 为 0-1 之间的小数）。
     */
    private static String hslToHex(double h, double s, double l) {
        double c = (1 - Math.abs(2 * l - 1)) * s;
        double x = c * (1 - Math.abs((h / 60.0) % 2 - 1));
        double m = l - c / 2.0;
        double r1, g1, b1;
        if (h < 60) {
            r1 = c;
            g1 = x;
            b1 = 0;
        } else if (h < 120) {
            r1 = x;
            g1 = c;
            b1 = 0;
        } else if (h < 180) {
            r1 = 0;
            g1 = c;
            b1 = x;
        } else if (h < 240) {
            r1 = 0;
            g1 = x;
            b1 = c;
        } else if (h < 300) {
            r1 = x;
            g1 = 0;
            b1 = c;
        } else {
            r1 = c;
            g1 = 0;
            b1 = x;
        }
        int r = (int) Math.round((r1 + m) * 255.0);
        int g = (int) Math.round((g1 + m) * 255.0);
        int b = (int) Math.round((b1 + m) * 255.0);
        return String.format("#%02X%02X%02X", clamp(r), clamp(g), clamp(b));
    }

    private static int clamp(int v) {
        if (v < 0) return 0;
        return Math.min(v, 255);
    }
}