package com.digiwin.mobile.mobileuibot.core.component.input.attachment;

import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.context.SpringContextHolder;
import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.file.FileUtil;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.common.map.MapUtil;
import com.digiwin.mobile.mobileuibot.common.string.StringUtil;
import com.digiwin.mobile.mobileuibot.core.common.ComponentChoiceTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.*;
import com.digiwin.mobile.mobileuibot.core.component.card.card.Card;
import com.digiwin.mobile.mobileuibot.core.rule.Rule;
import com.digiwin.mobile.mobileuibot.designer.uibot.service.UiBotDesignerRenderService;
import com.digiwin.mobile.mobileuibot.proxy.DigiwinAthenaApiResponse;
import com.digiwin.mobile.mobileuibot.proxy.atdm.model.DigiwinAtdmRequest;
import com.digiwin.mobile.mobileuibot.proxy.atdm.service.DigiwinAtdmProxyService;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcBacklogDetail;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcTaskProgressWorkItemAttachment;
import com.digiwin.mobile.mobileuibot.proxy.dmc.config.DigiwinDmcConfig;
import com.digiwin.mobile.mobileuibot.proxy.dmc.service.DigiwinDmcProxyService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotExecuteContext;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.attachment.UiBotAttachment;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.attachment.UiBotAttachmentData;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.rule.UiBotRule;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableAttachmentColumnAttribute;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumn;
import com.digiwin.mobile.mobileuibot.search.SearchItemDetailFixedNameEnum;
import com.digiwin.mobile.mobileuibot.search.SearchItemDetailStyleEnum;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>功能描述：附件组件。支持附件列表展示、附件上传、附件删除、附件预览</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: Attachment
 * @Author: Zaregoto
 * @Date: 2021/6/14 15:51
 */
@Data
@NoArgsConstructor
@AllArgsConstructor
public class Attachment<T extends AttachmentRawData> extends BaseInputMobileComponent implements Serializable {
    private static final long serialVersionUID = -7260016907362158686L;
    public static final String COMPONENT_TYPE = "ATTACHMENT";

    protected List<AttachmentFile> fileList = new ArrayList<>();
    protected FileMaxSize fileMaxSize;
    protected Long maxSize;
    protected T rawData;
    // 标题文字
    protected String title;

    /**
     * 标题的样式类型
     * 1-标题弱化，呈灰色， 2-标题正常，呈黑色， 3-标题黑色加粗，字号30px
     */
    protected Integer titleStyleType;

    /**
     * 操作类型
     *
     * @see AttachmentTypeEnum
     */
    protected Integer type = 2;

    /**
     * 样式类型，默认使用正常样式
     *
     * @see AttachmentStyleTypeEnum
     */
    protected Integer styleType = AttachmentStyleTypeEnum.TITLE_NORMAL.getValue();

    /**
     * 限制上传附件数量
     */
    protected Integer limit = 99;

    /**
     * 单选还是多选文件
     * 默认多选
     *
     * @see ComponentChoiceTypeEnum
     */
    protected String fileChoice = ComponentChoiceTypeEnum.MULTIPLE_CHOICE.getValue();

    /**
     * 当前字段是否是数组类型 -- 作用于与设计器集成后走V2版提交场景
     */
    protected Boolean isArray;

    /**
     * 是否折叠
     */
    Boolean isFold = false;

    /**
     * 折叠个数
     */
    Integer foldNumber = 3;

    /**
     * 默认展开，默认不展开为false
     */
    Boolean defaultExpand = false;

    /**
     * 上传类型, 附件组件限制只支持视频和图片
     *
     * @see AttachmentUploadTypeEnum
     */
    protected List<Integer> uploadTypes = new ArrayList<Integer>() {
        private static final long serialVersionUID = -7642929786167297837L;

        {
            add(AttachmentUploadTypeEnum.PICTURE.getValue());
            add(AttachmentUploadTypeEnum.VIDEO.getValue());
            add(AttachmentUploadTypeEnum.OTHER.getValue());
        }
    };

    public List<Integer> getUploadTypes() {
        // 兼容旧数据上传类型为空数组的情况
        if (CollectionUtils.isEmpty(this.uploadTypes)) {
            this.uploadTypes = new ArrayList<Integer>() {
                private static final long serialVersionUID = -4317825390163228210L;

                {
                    add(AttachmentUploadTypeEnum.PICTURE.getValue());
                    add(AttachmentUploadTypeEnum.VIDEO.getValue());
                    add(AttachmentUploadTypeEnum.OTHER.getValue());
                }
            };
        }
        return this.uploadTypes;
    }

    public static <T extends Attachment> T create(Class<T> clazz, String locale, UiBotTableColumn column, List<UiBotRule> columnRuleList,
                                                  Map<String, Object> bizData, DigiwinAtmcBacklogDetail digiwinAtmcBacklogDetail,
                                                  List<Rule> mobileValidateRuleList, Boolean disableAam) {
        Object data = bizData.get(column.getSchema());
        UiBotAttachment uiBotAttachment = JsonUtil.objectToJavaObject(data, UiBotAttachment.class);

        T attachment = null;
        try {
            attachment = setAttachmentRawData(attachment, clazz, uiBotAttachment, locale, column, columnRuleList, digiwinAtmcBacklogDetail, disableAam, null);
            // FIXME 写必填解析器
            Optional<UiBotRule> optColumnRequiredRule = columnRuleList.stream()
                    .filter(uiBotRule ->
                            null != uiBotRule.getSchema()
                                    && "required".equalsIgnoreCase(uiBotRule.getKey())
                                    && "EDIT".equalsIgnoreCase(uiBotRule.getScope()))
                    .findFirst();
            if (attachment != null) {
                attachment.setRequired(optColumnRequiredRule.isPresent());
                attachment.setTitle(column.getHeaderName());
                attachment.setType(column.canUpload() ?
                        AttachmentTypeEnum.READ_AND_WRITE.getValue() : AttachmentTypeEnum.READ_ONLY.getValue());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return attachment;
    }

    public static <T extends Attachment> T create(Class<T> clazz, String locale, UiBotTableColumn column, Map<String, Object> bizData) {
        Object data = bizData.get(column.getSchema());
        UiBotAttachment uiBotAttachment = JsonUtil.objectToJavaObject(data, UiBotAttachment.class);
        T attachment = null;
        try {
            UiBotTableAttachmentColumnAttribute columnAttribute = column.getAttribute();
            attachment = clazz.newInstance();
            for (UiBotAttachmentData uibotAttachData : uiBotAttachment.getData()) {
                attachment.getFileList().add(AttachmentFile.create(locale, uibotAttachData));
            }
            attachment.setRequired(true);
            attachment.setTitle(column.getHeaderName());
            attachment.setType(column.canUpload() ?
                    AttachmentTypeEnum.READ_AND_WRITE.getValue() : AttachmentTypeEnum.READ_ONLY.getValue());

            AttachmentRawDataDigiwinAthena attachmentRawDataDigiwinAthena = new AttachmentRawDataDigiwinAthena();
            attachmentRawDataDigiwinAthena.setDisableAam(Optional.ofNullable(column).map(UiBotTableColumn::getAttribute).map(UiBotTableAttachmentColumnAttribute::getDisableAam).orElse(true));
            attachmentRawDataDigiwinAthena.setUploadCategory(Optional.ofNullable(column).map(UiBotTableColumn::getAttribute).map(UiBotTableAttachmentColumnAttribute::getUploadCategory).orElse(""));
            attachmentRawDataDigiwinAthena.setUploadCategoryId("");
            attachmentRawDataDigiwinAthena.setPath("");
            attachmentRawDataDigiwinAthena.setRowDataKey("");
            attachmentRawDataDigiwinAthena.setBuckets(Optional.ofNullable(column).map(UiBotTableColumn::getAttribute).map(UiBotTableAttachmentColumnAttribute::getBuckets).orElse(DigiwinDmcConfig.ATHENA_BUCKET));

            attachment.setRawData(attachmentRawDataDigiwinAthena);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return attachment;
    }

    public static Attachment<AttachmentRawDataDigiwinAthena> create(UiBotLayout uiBotFieldLayout, UiBotRule uiBotRule) {
        Attachment<AttachmentRawDataDigiwinAthena> attachment = new Attachment<>();
        attachment.setSchema(uiBotFieldLayout.getSchema());
        attachment.setTitle(uiBotFieldLayout.getHeaderName());
        attachment.setType(AttachmentTypeEnum.READ_AND_WRITE.getValue());
        attachment.setRequired(false);
        if (uiBotRule != null) {
            attachment.setRequired("required".equalsIgnoreCase(uiBotRule.getKey()));
        }
        AttachmentRawDataDigiwinAthena attachmentRawDataDigiwinAthena = new AttachmentRawDataDigiwinAthena();
        if (uiBotFieldLayout.getAttribute() != null) {
            attachment.setFileMaxSize(new FileMaxSize(uiBotFieldLayout.getAttribute().getLong("fileMaxSfize")));
            attachmentRawDataDigiwinAthena.setDisableAam(uiBotFieldLayout.getAttribute().getBoolean("disableAam"));
            attachmentRawDataDigiwinAthena.setUploadCategory(uiBotFieldLayout.getAttribute().getString("uploadCategory"));
        }
        attachmentRawDataDigiwinAthena.setUploadCategoryId("");
        attachmentRawDataDigiwinAthena.setPath("");
        attachmentRawDataDigiwinAthena.setRowDataKey("");
        attachment.setRawData(attachmentRawDataDigiwinAthena);
        return attachment;
    }

    public static Attachment<AttachmentRawDataDigiwinAthena> create(UiBotLayout uiBotFieldLayout, UiBotRule uiBotRule, String bucket) {
        Attachment<AttachmentRawDataDigiwinAthena> attachment = new Attachment<>();
        attachment.setSchema(uiBotFieldLayout.getSchema());
        attachment.setTitle(uiBotFieldLayout.getHeaderName());
        attachment.setType(AttachmentTypeEnum.READ_AND_WRITE.getValue());
        attachment.setFileMaxSize(new FileMaxSize(uiBotFieldLayout.getAttribute().getLong("fileMaxSize")));
        attachment.setRequired(false);
        if (uiBotRule != null) {
            attachment.setRequired("required".equalsIgnoreCase(uiBotRule.getKey()));
        }
        AttachmentRawDataDigiwinAthena attachmentRawDataDigiwinAthena = new AttachmentRawDataDigiwinAthena();
        attachmentRawDataDigiwinAthena.setDisableAam(uiBotFieldLayout.getAttribute().getBoolean("disableAam"));
        attachmentRawDataDigiwinAthena.setUploadCategory(uiBotFieldLayout.getAttribute().getString("uploadCategory"));
        attachmentRawDataDigiwinAthena.setUploadCategoryId("");
        attachmentRawDataDigiwinAthena.setPath("");
        attachmentRawDataDigiwinAthena.setRowDataKey("");
        attachmentRawDataDigiwinAthena.setBuckets(bucket);
        attachment.setRawData(attachmentRawDataDigiwinAthena);
        return attachment;
    }

    /**
     * 创建只有一个文件的附件对象
     *
     * @param attachmentFile
     * @return
     */
    public static Attachment createWithOneFile(AttachmentFile attachmentFile) {
        Attachment attachment = new Attachment();
        if (null != attachmentFile) {
            attachment.getFileList().add(attachmentFile);
        }

        return attachment;
    }

    public static Attachment create(UiBotLayout uiBotLayout, Map<String, Object> data, String locale, DigiwinDmcProxyService digiwinDmcProxyService, Boolean isComplete) {
        Attachment attachment = new Attachment();
        attachment.setTitle(uiBotLayout.getmLabelText().get(locale));
        attachment.setStyleType(AttachmentStyleTypeEnum.TITLE_NORMAL.getValue());
        Map<String, Object> pcAttachment = (Map<String, Object>) data.get(uiBotLayout.getSchema());
        if (pcAttachment != null && pcAttachment.get("data") != null) {
            List<Map<String, Object>> attachmentDataList = (List<Map<String, Object>>) pcAttachment.get("data");
            if (attachmentDataList != null && attachmentDataList.size() > 0) {
                List<AttachmentFile> attachmentFiles = attachment.getFileList();
                for (Map<String, Object> attachmentData : attachmentDataList) {
                    attachment.setRequired(false);
                    AttachmentFile attachmentFile = new AttachmentFile();
                    String fileName = (String) attachmentData.get("name");
                    String id = (String) attachmentData.get("id");
                    attachmentFile.setFileExtension(AttachmentUtil.getAttachmentFileSuffix(fileName));
                    attachmentFile.setUploader((String) attachmentData.get("upload_user_name"));
                    attachmentFile.setFileSize(Long.valueOf((Integer) attachmentData.get("size")));
                    attachmentFile.setFileName(fileName);
//                    ShareInfo shareInfo = digiwinDmcProxyService.shareFile(id);
//                    if (shareInfo != null) {
//                        attachmentFile.setShareUrl(shareInfo.getUrl());
//                    }
                    attachmentFile.setFileId(id);
                    attachmentFile.setUploadTime((String) attachmentData.get("create_date"));
                    attachmentFiles.add(attachmentFile);
                }
                attachment.setType(AttachmentTypeEnum.READ_ONLY.getValue());
                attachment.setHidden(false);
                attachment.setEnable(!isComplete);
            } else {
                if (digiwinDmcProxyService != null && "attachment".equals(uiBotLayout.getSchema())) {
                    // 新建行事历页面要显示附件；行事历任务页面下无附件不显示
                    return null;
                }
                if (isComplete && "actualAttachment".equals(uiBotLayout.getSchema())) {
                    return null;
                }
                attachment.setHidden(true);
                attachment.setEnable(!isComplete);
                attachment.setRequired(false);
                if (isComplete) {
                    attachment.setType(AttachmentTypeEnum.READ_ONLY.getValue());
                } else {
                    attachment.setType(AttachmentTypeEnum.READ_AND_WRITE.getValue());
                }

            }
        } else {
            attachment.setEnable(!isComplete);
            attachment.setEnable(false);
            attachment.setRequired(false);
            if (isComplete) {
                attachment.setType(AttachmentTypeEnum.READ_ONLY.getValue());
            } else {
                attachment.setType(AttachmentTypeEnum.READ_AND_WRITE.getValue());
            }
        }
        if ("actualAttachment".equals(uiBotLayout.getSchema())) {
            attachment.setHidden(false);
            attachment.setEnable(!isComplete);
        }
        attachment.setFileMaxSize(new FileMaxSize(1024 * 1024 * 100L));
        attachment.setPlaceholder("大小限制100MB");
        attachment.setErrorMessage("上传失败");
        return attachment;
    }

    public static Attachment createAbnormalReasonAttachment(Map<String, Object> attachments, String title) {
        List<Map<String, Object>> fileInfoList = (List<Map<String, Object>>) attachments.get("data");
        List<AttachmentFile> fileList = new ArrayList<>();
        for (Map<String, Object> fileInfo : fileInfoList) {
            String fileExtension = null != AttachmentUtil.getAttachmentFileSuffix((String) fileInfo.get("name")) ?
                    AttachmentUtil.getAttachmentFileSuffix((String) fileInfo.get("name")) : "pdf";
            String fileId = null != fileInfo.get("id") ?
                    String.valueOf(fileInfo.get("id")) : "";
            String fileName = null != fileInfo.get("name") ?
                    String.valueOf(fileInfo.get("name")) : "";
            String shareUrl = null != fileInfo.get("file_download_url") ?
                    String.valueOf(fileInfo.get("file_download_url")) : "";
            String uploadTime = null != fileInfo.get("create_date") ?
                    String.valueOf(fileInfo.get("create_date")) : "";
            String uploader = null != fileInfo.get("upload_user_name") ?
                    String.valueOf(fileInfo.get("upload_user_name")) : "";
            String fileSize = null != fileInfo.get("size") ?
                    String.valueOf(fileInfo.get("size")) : "0";
            AttachmentFile attachmentFile = AttachmentFile.create(
                    fileExtension, fileId, fileName, Double.valueOf(fileSize).longValue(), shareUrl, uploadTime, uploader);
            fileList.add(attachmentFile);
        }

        Attachment attachment = new Attachment();
        attachment.setFileList(fileList);
        attachment.setRawData(new AttachmentRawDataDigiwinAthena());
        attachment.setStyleType(AttachmentStyleTypeEnum.TITLE_WEAKENED.getValue());
        attachment.setTitle(title);
        attachment.setTitleStyleType(SearchItemDetailStyleEnum.TITLE_STYLE_TYPE_ONE.getValue());
        attachment.setType(AttachmentTypeEnum.READ_ONLY.getValue());

        return attachment;
    }

    public static Attachment create(String title, Integer styleType, Integer titleStyleType, Integer type, Boolean abnormalReturns) {
        Attachment attachment = new Attachment();
        attachment.setFileList(Collections.emptyList());
        attachment.setRawData(new AttachmentRawDataDigiwinAthena());
        attachment.setStyleType(styleType);
        attachment.setTitle(title);
        attachment.setTitleStyleType(titleStyleType);
        attachment.setType(type);
        // S4 APC 异常回报任务-转派、处理-附件档案-附件大小限制20M，个数限制10M，不区分文档、图片还是视频、音频
        attachment.setFileMaxSize(new FileMaxSize(1024 * 1024 * 20L));
        attachment.setLimit(10);
        attachment.setHidden(!(ObjectUtils.isNotEmpty(abnormalReturns) && BooleanUtils.isTrue(abnormalReturns)));
        return attachment;
    }


    /**
     * 将web端的Attachement转为移动的attachment
     *
     * @param uiBotAttachment
     * @param locale
     * @param title
     * @return
     */
    public static Attachment uibotAttachmentparseAttachment(UiBotAttachment uiBotAttachment,
                                                            String locale, String title) {
        Attachment attachment = new Attachment();
        attachment.setTitle(title);
        for (UiBotAttachmentData uibotAttachData : uiBotAttachment.getData()) {
            attachment.getFileList().add(AttachmentFile.create(locale, uibotAttachData));
        }
        attachment.setRawData(new AttachmentRawDataDigiwinAthena());
        attachment.setStyleType(AttachmentStyleTypeEnum.TITLE_WEAKENED.getValue());
        attachment.setTitleStyleType(SearchItemDetailStyleEnum.TITLE_STYLE_TYPE_ONE.getValue());
        attachment.setType(AttachmentTypeEnum.READ_ONLY.getValue());
        return attachment;
    }

    /**
     * 将web端的Attachement转为移动的attachment
     *
     * @param digiwinAtmcTaskProgressWorkItemAttachmentList
     * @param locale
     * @param title
     * @return
     */
    public static Attachment uibotAttachmentparseAttachment(List<DigiwinAtmcTaskProgressWorkItemAttachment> digiwinAtmcTaskProgressWorkItemAttachmentList,
                                                            String locale, String title) {
        Attachment attachment = new Attachment();
        attachment.setTitle(title);
        for (DigiwinAtmcTaskProgressWorkItemAttachment digiwinAtmcTaskProgressWorkItemAttachment : digiwinAtmcTaskProgressWorkItemAttachmentList) {
            attachment.getFileList().add(AttachmentFile.create(locale, digiwinAtmcTaskProgressWorkItemAttachment));
        }
        attachment.setRawData(new AttachmentRawDataDigiwinAthena());
        attachment.setStyleType(AttachmentStyleTypeEnum.TITLE_WEAKENED.getValue());
        attachment.setTitleStyleType(SearchItemDetailStyleEnum.TITLE_STYLE_TYPE_ONE.getValue());
        attachment.setType(AttachmentTypeEnum.READ_ONLY.getValue());
        return attachment;
    }

    public static Attachment createTbdsFiles(List<Map<String, Object>> files, String title) {
        List<AttachmentFile> fileList = new ArrayList<>();
        for (Map<String, Object> fileInfo : files) {
            String fileExtension;
            if ("0".equals(fileInfo.get("resource_type"))) {
                fileExtension = "jpg";
            } else {
                fileExtension = "mp4";
            }
            String fileId = null != fileInfo.get("source_id") ?
                    String.valueOf(fileInfo.get("source_id")) : "";
            String fileName = null != fileInfo.get("source_id") ?
                    String.valueOf(fileInfo.get("source_id")) : "";
            String uploadTime = null != fileInfo.get("create_time") ?
                    String.valueOf(fileInfo.get("create_time")) : "";
            String uploader = null != fileInfo.get("create_name") ?
                    String.valueOf(fileInfo.get("create_name")) : "";
            String fileSize = null != fileInfo.get("file_size") ?
                    String.valueOf(fileInfo.get("file_size")) : "0";
            String videoCoverId = null != fileInfo.get("video_cover_id") ?
                    String.valueOf(fileInfo.get("video_cover_id")) : "";
            String shareUrl = AppContext.getApiUrlSetting().getDmcUrl() + "/api/dmc/v2/file/tbds/preview/" + fileId;
            AttachmentFile attachmentFile = AttachmentFile.create(
                    fileExtension, fileId, fileName, Long.valueOf(fileSize), shareUrl, uploadTime, uploader);
            if (StringUtils.isNotEmpty(videoCoverId)) {
                //视频封面 图片预览地址
                String videoCoverUrl = AppContext.getApiUrlSetting().getDmcUrl() + "/api/dmc/v2/file/tbds/preview/" + videoCoverId;
                attachmentFile.setPreviewUrl(videoCoverUrl);
                attachmentFile.setCoverId(videoCoverId);
            }
            fileList.add(attachmentFile);
        }
        Attachment attachment = new Attachment();
        attachment.setFileList(fileList);
        attachment.setRawData(new AttachmentRawDataDigiwinAthena());
        attachment.setStyleType(AttachmentStyleTypeEnum.TITLE_NORMAL.getValue());
        attachment.setTitle(title);
        attachment.setTitleStyleType(SearchItemDetailStyleEnum.TITLE_STYLE_TYPE_THREE.getValue());
        attachment.setType(AttachmentTypeEnum.READ_ONLY.getValue());
        return attachment;
    }


    public static Attachment createTbdsFiles(List<Map<String, Object>> files, Integer type, String title,
                                             Integer styleType, String bucketName, Integer titleStyleType) {
        List<AttachmentFile> fileList = new ArrayList<>();
        for (Map<String, Object> fileInfo : files) {
            String fileId = null != fileInfo.get("id") ?
                    String.valueOf(fileInfo.get("id")) : "";
            String fileName = null != fileInfo.get("name") ?
                    String.valueOf(fileInfo.get("name")) : "";
            String fileExtension = FileUtil.getFileExtension(fileName);
            String uploadTime = null != fileInfo.get("create_date") ?
                    String.valueOf(fileInfo.get("create_date")) : "";
            String uploader = null != fileInfo.get("upload_user_name") ?
                    String.valueOf(fileInfo.get("upload_user_name")) : "";
            Long fileSize;
            if (fileInfo.get("size") instanceof Double) {
                Double fileSizeDouble = null != fileInfo.get("size") ?
                        (Double) fileInfo.get("size") : 0.0;
                fileSize = fileSizeDouble.longValue();
            } else {
                String fileSizeString = null != fileInfo.get("file_size") ?
                        String.valueOf(fileInfo.get("file_size")) : "0";
                fileSize = Long.valueOf(fileSizeString);
            }
            String shareUrl = AppContext.getApiUrlSetting().getDmcUrl() + "/api/dmc/v2/file/" + bucketName + "/preview/" + fileId;
            AttachmentFile attachmentFile = AttachmentFile.create(
                    fileExtension, fileId, fileName, fileSize, shareUrl, uploadTime, uploader);

            fileList.add(attachmentFile);
        }

        Attachment attachment = new Attachment();
        attachment.setFileList(fileList);
        attachment.setRawData(new AttachmentRawDataDigiwinAthena());
        attachment.setStyleType(styleType);
        attachment.setTitle(title);
        attachment.setTitleStyleType(titleStyleType);
        attachment.setType(type);

        return attachment;
    }

    @Override
    public String returnComponentType() {
        return Attachment.COMPONENT_TYPE;
    }

    public static <T extends Attachment> T setAttachmentRawData(T oldAttachment, Class<T> clazz, UiBotAttachment uiBotAttachment,
                                                                String locale, UiBotTableColumn column,
                                                                List<UiBotRule> columnRuleList, DigiwinAtmcBacklogDetail digiwinAtmcBacklogDetail,
                                                                Boolean disableAam, String path) {
        UiBotTableAttachmentColumnAttribute columnAttribute = column.getAttribute();
        try {
            oldAttachment = clazz.newInstance();
            for (UiBotAttachmentData uibotAttachData : uiBotAttachment.getData()) {
                oldAttachment.getFileList().add(AttachmentFile.create(locale, uibotAttachData));
            }
            if (columnAttribute != null) {
                oldAttachment.setFileMaxSize(new FileMaxSize(columnAttribute.getFileMaxSize()));
            }
            // FIXME 因规则对象UiBotRule中的path在同一个JSON中是相同的，所以这里可以用取巧的方式获得
            if (path == null && !columnRuleList.isEmpty()) {
                path = columnRuleList.get(0).getPath();
            }
            AttachmentRawDataDigiwinAthena rawDataDigiwinAthena = new AttachmentRawDataDigiwinAthena();
            rawDataDigiwinAthena.setPath(path);
            rawDataDigiwinAthena.setRowDataKey(uiBotAttachment.getRowDataKey());
            rawDataDigiwinAthena.setRowData(uiBotAttachment.getRowData());
            rawDataDigiwinAthena.setTmActivityId(digiwinAtmcBacklogDetail.getTmActivityId());
            rawDataDigiwinAthena.setTmTaskId(digiwinAtmcBacklogDetail.getTmTaskId());
            rawDataDigiwinAthena.setBuckets(column.getBuckets());
            if (columnAttribute != null) {
                rawDataDigiwinAthena.setUploadCategory(columnAttribute.getUploadCategory());
            } else {
                if (column.getOcrType() == 0) {
                    rawDataDigiwinAthena.setUploadCategory("employeeInduction_identityCard");
                } else if (column.getOcrType() == 1) {
                    rawDataDigiwinAthena.setUploadCategory("employeeInduction_wagesCard");
                }

            }

            rawDataDigiwinAthena.setDisableAam(disableAam);

            oldAttachment.setRawData(rawDataDigiwinAthena);
        } catch (InstantiationException | IllegalAccessException e) {
            e.printStackTrace();
        }
        return oldAttachment;
    }

    public UiBotAttachment parseToUiBotAttachment(Attachment attachment, DigiwinDmcProxyService digiwinDmcProxyService) {
        AttachmentRawDataDigiwinAthena attachmentRawData =
                (AttachmentRawDataDigiwinAthena) attachment.getRawData();
        UiBotAttachment uiBotAttachment = new UiBotAttachment();
        // 会出现categoryId为空的情况，需要先查一下
        String uploadCategoryId;
        if (ObjectUtils.isEmpty(attachmentRawData)) {
            uploadCategoryId = "";
        } else {
            uploadCategoryId = Optional.ofNullable(attachmentRawData.getUploadCategoryId()).orElse(
                    digiwinDmcProxyService.getDirIdByName(attachmentRawData.getUploadCategory())
            );
        }
        attachment.getFileList().removeIf(e -> "DELETE".equals(((AttachmentFile) e).getStatus()));
        attachment.getFileList().forEach(obj -> {
            AttachmentFile attachmentFile = (AttachmentFile) obj;
            UiBotAttachmentData uiBotAttachmentData = new UiBotAttachmentData();
            uiBotAttachmentData.setId(attachmentFile.getFileId());
            uiBotAttachmentData.setName(attachmentFile.getFileName());
            uiBotAttachmentData.setUploadUserName(attachmentFile.getUploader());
            uiBotAttachmentData.setCreateDate(attachmentFile.getUploadTime());
            uiBotAttachmentData.setSize(Optional.ofNullable(attachmentFile.getFileSize()).orElse(new Long(0)).doubleValue());
            uiBotAttachmentData.setUibotStatus(attachmentFile.getStatus());
            uiBotAttachmentData.setCategory(attachmentRawData.getUploadCategory());
            uiBotAttachmentData.setCategoryId(Optional.ofNullable(uploadCategoryId).orElse(""));
            uiBotAttachmentData.setRowDataKey(attachmentRawData.getRowDataKey());
            uiBotAttachmentData.setRowData(attachmentRawData.getRowData());
            uiBotAttachmentData.setShareUrl(attachmentFile.getShareUrl());
            uiBotAttachmentData.setBase64Data(attachmentFile.getBase64Data());
            uiBotAttachmentData.setCoverBase64Data(attachmentFile.getCoverBase64Data());
            uiBotAttachmentData.setFileExtension(attachmentFile.getFileExtension());
            uiBotAttachmentData.setBase64DataId(attachmentFile.getBase64DataId());
            uiBotAttachmentData.setApprovalCreateDate(attachmentFile.getUploadTime());

            uiBotAttachment.addData(uiBotAttachmentData);
        });
        uiBotAttachment.setRowDataKey(attachmentRawData.getRowDataKey());
        uiBotAttachment.setRowData(attachmentRawData.getRowData());
        return uiBotAttachment;
    }

    public UiBotAttachment parseToUiBotAttachmentNotFilterDelete(Attachment attachment, DigiwinDmcProxyService digiwinDmcProxyService) {
        AttachmentRawDataDigiwinAthena attachmentRawData =
                (AttachmentRawDataDigiwinAthena) attachment.getRawData();
        UiBotAttachment uiBotAttachment = new UiBotAttachment();
        // 会出现categoryId为空的情况，需要先查一下
        String uploadCategoryId;
        if (ObjectUtils.isEmpty(attachmentRawData)) {
            uploadCategoryId = "";
        } else {
            uploadCategoryId = Optional.ofNullable(attachmentRawData.getUploadCategoryId()).orElse(
                    digiwinDmcProxyService.getDirIdByName(attachmentRawData.getUploadCategory())
            );
        }
        attachment.getFileList().forEach(obj -> {
            AttachmentFile attachmentFile = (AttachmentFile) obj;
            UiBotAttachmentData uiBotAttachmentData = new UiBotAttachmentData();
            uiBotAttachmentData.setId(attachmentFile.getFileId());
            uiBotAttachmentData.setName(attachmentFile.getFileName());
            uiBotAttachmentData.setUploadUserName(attachmentFile.getUploader());
            uiBotAttachmentData.setCreateDate(attachmentFile.getUploadTime());
            uiBotAttachmentData.setSize(Optional.ofNullable(attachmentFile.getFileSize()).orElse(new Long(0)).doubleValue());
            uiBotAttachmentData.setUibotStatus(attachmentFile.getStatus());
            uiBotAttachmentData.setCategory(attachmentRawData.getUploadCategory());
            uiBotAttachmentData.setCategoryId(Optional.ofNullable(uploadCategoryId).orElse(""));
            uiBotAttachmentData.setRowDataKey(attachmentRawData.getRowDataKey());
            uiBotAttachmentData.setRowData(attachmentRawData.getRowData());
            uiBotAttachmentData.setShareUrl(attachmentFile.getShareUrl());
            uiBotAttachmentData.setApprovalCreateDate(attachmentFile.getUploadTime());
            uiBotAttachmentData.setBase64Data(attachmentFile.getBase64Data());
            uiBotAttachmentData.setFileExtension(attachmentFile.getFileExtension());
            uiBotAttachmentData.setBase64DataId(attachmentFile.getBase64DataId());
            uiBotAttachmentData.setCoverBase64Data(attachmentFile.getCoverBase64Data());
            if ("DELETE".equals(attachmentFile.getStatus())) {
                uiBotAttachmentData.setUibotStatus("DELETE");
            }

            uiBotAttachment.addData(uiBotAttachmentData);
        });
        uiBotAttachment.setRowDataKey(attachmentRawData.getRowDataKey());
        uiBotAttachment.setRowData(attachmentRawData.getRowData());
        return uiBotAttachment;
    }

    public static Attachment createSolution(String locale, String iamUserToken, Map<String, Object> detailInfo) {
        Map<String, Object> knowledgeCustomize = (Map<String, Object>) detailInfo.get("knowledge_customize");
        String solutionName = String.valueOf(knowledgeCustomize.get("solution_name"));
        List<Map<String, Object>> questionAcceptanceInfoList = (List<Map<String, Object>>) knowledgeCustomize.get("question_acceptance_info");
        List<AttachmentFile> fileList = new ArrayList<>();
        for (Map<String, Object> questionAcceptanceInfo : questionAcceptanceInfoList) {
            String fileExtension = null != questionAcceptanceInfo.get("file_extension") ?
                    String.valueOf(questionAcceptanceInfo.get("file_extension")) : "pdf";
            String fileId = null != questionAcceptanceInfo.get("print_report_id") ?
                    String.valueOf(questionAcceptanceInfo.get("print_report_id")) : "";
            String fileName = null != questionAcceptanceInfo.get("file_name") ?
                    String.valueOf(questionAcceptanceInfo.get("file_name")) : solutionName;
            //临时测试用https://dmc-hw-test.digiwincloud.com.cn/api/dmc/v2/file/FRC/preview/ba2b457a-2d4b-4003-a7ec-f3575b0beebb，web端改造好后，null情况下赋空字符串
            String shareUrl = null != questionAcceptanceInfo.get("print_report_preview_url") ?
                    String.valueOf(questionAcceptanceInfo.get("print_report_preview_url")) : "";
            String uploadTime = null != questionAcceptanceInfo.get("create_time") ?
                    String.valueOf(questionAcceptanceInfo.get("create_time")) : "";
            String uploader = null != questionAcceptanceInfo.get("create_name") ?
                    String.valueOf(questionAcceptanceInfo.get("create_name")) : "";
            String fileSize = null != questionAcceptanceInfo.get("file_size") ?
                    String.valueOf(questionAcceptanceInfo.get("file_size")) : "0";
            AttachmentFile attachmentFile = AttachmentFile.create(
                    fileExtension, fileId, fileName, Long.valueOf(fileSize), shareUrl, uploadTime, uploader);

            fileList.add(attachmentFile);
        }
        Attachment attachment = new Attachment();
        attachment.setFileList(fileList);
        attachment.setRawData(new AttachmentRawDataDigiwinAthena());
        attachment.setStyleType(AttachmentStyleTypeEnum.TITLE_NORMAL.getValue());
        attachment.setTitle(
                LocaleUtil.getMobileTextByKey(locale, SearchItemDetailFixedNameEnum.QUESTION_SOLUTION.getValue())
        );
        attachment.setTitleStyleType(SearchItemDetailStyleEnum.TITLE_STYLE_TYPE_THREE.getValue());
        attachment.setType(AttachmentTypeEnum.READ_ONLY.getValue());

        return attachment;
    }

    public static Attachment createActualAttachment(String locale, String iamUserToken, Map<String, Object> detailInfo) {
        List<Map<String, Object>> fileInfoList = (List<Map<String, Object>>) detailInfo.get("file_info");
        List<AttachmentFile> fileList = new ArrayList<>();
        for (Map<String, Object> fileInfo : fileInfoList) {
            String fileExtension = null != fileInfo.get("file_extension") ?
                    String.valueOf(fileInfo.get("file_extension")) : "pdf";
            String fileId = null != fileInfo.get("id") ?
                    String.valueOf(fileInfo.get("id")) : "";
            String fileName = null != fileInfo.get("file_name") ?
                    String.valueOf(fileInfo.get("file_name")) : "";
            String shareUrl = null != fileInfo.get("file_download_url") ?
                    String.valueOf(fileInfo.get("file_download_url")) : "";
            String uploadTime = null != fileInfo.get("create_time") ?
                    String.valueOf(fileInfo.get("create_time")) : "";
            String uploader = null != fileInfo.get("create_name") ?
                    String.valueOf(fileInfo.get("create_name")) : "";
            String fileSize = null != fileInfo.get("file_size") ?
                    String.valueOf(fileInfo.get("file_size")) : "0";
            AttachmentFile attachmentFile = AttachmentFile.create(
                    fileExtension, fileId, fileName, Long.valueOf(fileSize), shareUrl, uploadTime, uploader);

            fileList.add(attachmentFile);
        }

        Attachment attachment = new Attachment();
        attachment.setFileList(fileList);
        attachment.setRawData(new AttachmentRawDataDigiwinAthena());
        attachment.setStyleType(AttachmentStyleTypeEnum.TITLE_NORMAL.getValue());
        attachment.setTitle(
                LocaleUtil.getMobileTextByKey(locale, SearchItemDetailFixedNameEnum.SEARCH_ITEM_DETAIL_ATTACHMENT.getValue())
        );
        attachment.setTitleStyleType(SearchItemDetailStyleEnum.TITLE_STYLE_TYPE_THREE.getValue());
        attachment.setType(AttachmentTypeEnum.READ_ONLY.getValue());

        return attachment;
    }

    /**
     * 附件--多数据源配置
     */
    protected DataSource dataSource;

    @Override
    public void handleComponentParam(ComponentContext cmptContext, String mobilePath, String schema, Map<String, Object> data, PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        super.resetCmptMultipleDataSourceData(cmptContext, executeContext, this.dataSource, data);
        super.handleComponentParam(cmptContext, mobilePath, schema, data, executeContext, mobilePageRawData);
        // 定制：PCM_项目预算签核
        this.customizePcmBudgetProjectApprovals(mobilePath, schema, data, executeContext, mobilePageRawData);
        if (this.fileMaxSize != null) {
            this.fileMaxSize.MB2B();
        }
        UiBotAttachment uiBotAttachment =
                Optional.ofNullable(UiBotDesignerRenderService.getDataByPathAndSchema(data, mobilePath, schema))
                        .map(r ->
                        {
                            if (r instanceof Map) {
                                this.isArray = false;
                                return JsonUtil.objectToJavaObject(r, new TypeReference<UiBotAttachment>() {
                                });
                            } else if (r instanceof UiBotAttachment) {
                                this.isArray = false;
                                return (UiBotAttachment) r;
                            } else if (r instanceof List) {
                                this.isArray = true;
                                if (!CollectionUtils.isEmpty((List) r)) {
                                    return JsonUtil.objectToJavaObject(r, new TypeReference<List<UiBotAttachment>>() {
                                    }).get(0);
                                }
                            }
                            return null;
                        })
                        .orElse(new UiBotAttachment());

        for (UiBotAttachmentData uibotAttachData : uiBotAttachment.getData()) {
            this.fileList.add(AttachmentFile.create(executeContext.getLocale(), uibotAttachData));
        }
        AttachmentRawDataDigiwinAthena rawData = Optional
                .ofNullable((AttachmentRawDataDigiwinAthena) this.getRawData())
                .orElse(new AttachmentRawDataDigiwinAthena());

        AttachmentRawDataDigiwinAthena newRawData = new AttachmentRawDataDigiwinAthena();
        newRawData.setPath("");
        newRawData.setRowDataKey(uiBotAttachment.getRowDataKey());
        newRawData.setRowData(uiBotAttachment.getRowData());
        newRawData.setTmActivityId(executeContext.getTmActivityId());
        newRawData.setTmTaskId(executeContext.getTmProjectId());
        newRawData.setBuckets(rawData.getBuckets());
        newRawData.setUploadCategory(Optional.ofNullable(rawData.getUploadCategory()).orElse(""));
        newRawData.setDisableAam(Optional.ofNullable(rawData.getDisableAam()).orElse(true));
        this.rawData = (T) newRawData;

        specialHandle(executeContext, mobilePath, schema, data);
    }

    private void specialHandle(PcUiBotExecuteContext executeContext, String mobilePath, String schema, Map<String, Object> data) {
        // 试验优测定制-报工图片和视频栏位只能支持上传图片和视频
        if ("UT_InspectionResultsRegistration".equalsIgnoreCase(executeContext.getTmActivityId())) {
            if ("video_attachment".equals(schema)) {
                setSupportImageAndVideoType();
            }
        }
        // 问题评审，应用接口返回的问题图片和附件不是标准格式，需要自己处理组装
        if ("questionIdentify".equalsIgnoreCase(executeContext.getTmActivityId())) {
            if ("problem_review_attachment".equals(schema)) {
                this.isArray = true;
            }
            Object attachmentInfo = UiBotDesignerRenderService.getDataByPathAndSchema(data, mobilePath, schema);
            if (ObjectUtils.isNotEmpty(attachmentInfo)) {
                if (attachmentInfo instanceof List && !CollectionUtils.isEmpty((List) attachmentInfo)) {
                    List<Map<String, Object>> attachmentInfoList = (List<Map<String, Object>>) attachmentInfo;
                    if ("attachment_info".equals(schema)) {
                        List<AttachmentFile> attachmentFileList = attachmentInfoList.stream()
                                .filter(e -> !CollectionUtils.isEmpty(e))
                                .map(e -> {
                                    AttachmentFile attachmentFile = new AttachmentFile();
                                    attachmentFile.setFileId(MapUtils.getString(e, "attachment_id"));
                                    attachmentFile.setFileName(MapUtils.getString(e, "attachment_name"));
                                    attachmentFile.setFileExtension(AttachmentUtil.getAttachmentFileSuffix(MapUtils.getString(e, "attachment_name")));
                                    attachmentFile.setUploader(MapUtils.getString(e, "upload_person_name"));
                                    return attachmentFile;
                                }).collect(Collectors.toList());

                        // 所属阶段需要区分是问题反馈还是问题评审的，并在提交时候处理带上
                        AttachmentRawDataDigiwinAthena attachmentRawDataDigiwinAthena = JsonUtil.objectToJavaObject(this.rawData, AttachmentRawDataDigiwinAthena.class);
                        List<String> problemFeedbackFileIdList = attachmentFileList.stream().map(e -> e.getFileId()).collect(Collectors.toList());
                        Map<String, Object> problemFeedbackFileIdMap = new HashMap<>(1);
                        problemFeedbackFileIdMap.put("problemFeedbackFileIdList", problemFeedbackFileIdList);
                        attachmentRawDataDigiwinAthena.setExtendParam(problemFeedbackFileIdMap);
                        this.setRawData((T) attachmentRawDataDigiwinAthena);
                        this.fileList = attachmentFileList;
                    }

                    if ("question_picture".equals(schema)) {
                        List<AttachmentFile> attachmentFileList = attachmentInfoList.stream()
                                .filter(e -> !CollectionUtils.isEmpty(e))
                                .map(e -> {
                                    AttachmentFile attachmentFile = new AttachmentFile();
                                    attachmentFile.setFileId(MapUtils.getString(e, "picture_id"));
                                    attachmentFile.setFileName(MapUtils.getString(e, "picture_id"));
                                    attachmentFile.setFileExtension("jpg");
                                    return attachmentFile;
                                }).collect(Collectors.toList());
                        this.fileList = attachmentFileList;
                    }
                }
            }
        }
    }

    private void customizePcmBudgetProjectApprovals(String mobilePath, String schema, Map<String, Object> data, PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        String cmptSchema = "PCM_budgetProjectApprovals_DTD_attachment";
        if (!cmptSchema.equals(schema)) {
            return;
        }
        Map<String, Object> bpmData = Optional.ofNullable(executeContext.getBpmData()).orElse(new HashMap<>());
        Map<String, Object> parameter = new HashMap<>();
        parameter.put("attachment_type", "1");
        parameter.put("project_no", bpmData.get("project_no"));
        parameter.put("version_no", bpmData.get("version_no"));
        if (bpmData.get("version_no") == null) {
            return;
        }
        DigiwinAtdmRequest request = DigiwinAtdmRequest.create("bm.pcsc.project.budget.attachment.get",
                parameter,
                JsonUtil.objectToJavaObject(executeContext, Map.class),
                JsonUtil.objectToJavaObject(executeContext.getBusinessUnit(), Map.class)
        );
        DigiwinAthenaApiResponse<Map<String, Object>> athenaApiResponse = SpringContextHolder.getBean(DigiwinAtdmProxyService.class)
                .dataQueryByActionId(
                        AppRequestContext.getContextEntity().getLocale(),
                        AppRequestContext.getContextEntity().getIamUserToken(),
                        JsonUtil.javaObjectToJsonString(request),
                        AppRequestContext.getContextEntity().getTenantId()
                );
        if (athenaApiResponse == null) {
            return;
        }
        Map<String, Object> map = athenaApiResponse.getResponse();
        if (map == null) {
            return;
        }
        List<Map<String, Object>> budgetAttachmentInfoList = (List<Map<String, Object>>) map.get("budget_attachment_info");
        if (CollectionUtils.isEmpty(budgetAttachmentInfoList)) {
            return;
        }
        String attachment = (String) budgetAttachmentInfoList.get(0).get("attachment");
        if (StringUtils.isEmpty(attachment)) {
            return;
        }
        List<Map<String, Object>> rawAttachment = JsonUtil.jsonStringToObject(attachment, List.class);
        if (CollectionUtils.isEmpty(rawAttachment)) {
            return;
        }

        // 定义map对象，不能定义UiBotAttachment对象，否则获取data中数据转UiBotAttachment对象时有问题
        Map<String, Object> uiBotAttachment = new HashMap<>();
        uiBotAttachment.put("data", rawAttachment.stream().map(raw -> {
            UiBotAttachmentData uiBotAttachmentData = new UiBotAttachmentData();
            uiBotAttachmentData.setId((String) raw.get("id"));
            uiBotAttachmentData.setName((String) raw.get("fileName"));
            uiBotAttachmentData.setUploadUserName("");
            uiBotAttachmentData.setCreateDate(StringUtil.valueOf(raw.get("createDate")).replace("/", "-"));
            uiBotAttachmentData.setSize(new Double(StringUtil.valueOf(raw.getOrDefault("totalSize", 0))));
            uiBotAttachmentData.setShareUrl("");
            return uiBotAttachmentData;
        }).collect(Collectors.toList()));
        // 存储该schema节点下的附件信息
        data.put(cmptSchema, uiBotAttachment);
    }


    public static Attachment createAttachment(List<Map<String, Object>> fileInfoList, Attachment attachment, String title) {
        List<AttachmentFile> fileList = new ArrayList<>();
        for (Map<String, Object> fileInfo : fileInfoList) {
            String fileExtension = null != AttachmentUtil.getAttachmentFileSuffix((String) fileInfo.get("name")) ?
                    AttachmentUtil.getAttachmentFileSuffix((String) fileInfo.get("name")) : "pdf";
            String fileId = null != fileInfo.get("id") ?
                    String.valueOf(fileInfo.get("id")) : "";
            String fileName = null != fileInfo.get("name") ?
                    String.valueOf(fileInfo.get("name")) : "";
            String shareUrl = null != fileInfo.get("file_download_url") ?
                    String.valueOf(fileInfo.get("file_download_url")) : "";
            String uploadTime = null != fileInfo.get("create_date") ?
                    String.valueOf(fileInfo.get("create_date")) : "";
            String uploader = null != fileInfo.get("upload_user_name") ?
                    String.valueOf(fileInfo.get("upload_user_name")) : "";
            String fileSize = null != fileInfo.get("size") ?
                    String.valueOf(fileInfo.get("size")) : "0";
            AttachmentFile attachmentFile = AttachmentFile.create(
                    fileExtension, fileId, fileName, Double.valueOf(fileSize).longValue(), shareUrl, uploadTime, uploader);
            fileList.add(attachmentFile);
        }
        if (ObjectUtils.isEmpty(attachment)) {
            attachment = new Attachment();
            attachment.setStyleType(AttachmentStyleTypeEnum.TITLE_WEAKENED.getValue());
            attachment.setTitleStyleType(SearchItemDetailStyleEnum.TITLE_STYLE_TYPE_ONE.getValue());
            attachment.setType(AttachmentTypeEnum.READ_ONLY.getValue());
        }
        attachment.setFileList(fileList);
        attachment.setTitle(title);
        return attachment;
    }

    public void setSupportImageAndVideoType() {
        this.uploadTypes = new ArrayList<Integer>() {
            private static final long serialVersionUID = -887051684180235461L;

            {
                add(AttachmentUploadTypeEnum.PICTURE.getValue());
                add(AttachmentUploadTypeEnum.VIDEO.getValue());
            }
        };
    }


    /**
     * 创建体系云手动报工交付物和历史交付物两字段附件
     *
     * @param column
     * @param digiwinAtmcBacklogDetail
     * @return
     */
    public static Attachment createManualReportingAttachment(String locale, UiBotTableColumn column,
                                                             Map<String, Object> bizData, DigiwinAtmcBacklogDetail digiwinAtmcBacklogDetail) {
        Attachment attachment = new Attachment();
        //交付物
        if ("sch-manual-task-delivery".equalsIgnoreCase(column.getType())) {
            attachment.setRequired(true);
            attachment.setType(AttachmentTypeEnum.READ_AND_WRITE.getValue());
        }
        //历史交付物
        if ("sch-manual-task-delivery-history".equalsIgnoreCase(column.getType())) {
            attachment.setRequired(false);
            attachment.setType(AttachmentTypeEnum.READ_ONLY.getValue());
        }

        Object data = bizData.get(column.getSchema());

        // 体系云管家交付物日期展示
        translateDoubleTimeToString(data, "create_date");
        UiBotAttachment uiBotAttachment = JsonUtil.objectToJavaObject(data, UiBotAttachment.class);
        if (ObjectUtils.isNotEmpty(uiBotAttachment) && !CollectionUtils.isEmpty(uiBotAttachment.getData())) {
            for (UiBotAttachmentData uibotAttachData : uiBotAttachment.getData()) {
                attachment.getFileList().add(AttachmentFile.create(locale, uibotAttachData));
            }
        }

        attachment.setTitle(column.getHeaderName());
        attachment.setFileMaxSize(new FileMaxSize(1024 * 1024 * 100L));
        attachment.setPlaceholder("大小限制100MB");
        attachment.setErrorMessage("上传失败");

        AttachmentRawDataDigiwinAthena rawDataDigiwinAthena = new AttachmentRawDataDigiwinAthena();
        rawDataDigiwinAthena.setPath("task_info");
        rawDataDigiwinAthena.setTmActivityId(digiwinAtmcBacklogDetail.getTmActivityId());
        rawDataDigiwinAthena.setTmTaskId(digiwinAtmcBacklogDetail.getTmTaskId());
        rawDataDigiwinAthena.setBuckets(DigiwinDmcConfig.SCH_BUCKET);
        rawDataDigiwinAthena.setUploadCategory("schManualAssignmentAttachment");
        rawDataDigiwinAthena.setDisableAam(true);
        attachment.setRawData(rawDataDigiwinAthena);

        return attachment;
    }

    /**
     * 从data中获取到的日期为double类型，需要转化为对应日期格式，否则展示有问题
     */
    public static void translateDoubleTimeToString(Object data, String dateFieldName) {
        if (Objects.isNull(data)) {
            return;
        }
        List<Map<String, Object>> dataList = MapUtil.getOrDefault((Map) data, "data", Collections.emptyList());
        dataList.forEach(item -> {
            if (Objects.nonNull(item.get(dateFieldName))) {
                double createDate = MapUtils.getDouble(item, dateFieldName);
                String translateCreateDate = new SimpleDateFormat(DateTimeUtil.DEFAULT_DATETIME_PATTERN)
                        .format(new Date(Long.valueOf((long) createDate)));
                item.put(dateFieldName, translateCreateDate);
            }
        });
    }

    public static BaseMobileComponent cardContentConvertComponent(Card.DesignerRawCard.Content rawContent, ComponentContext cmptContext, Map<String, Object> data, PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        Attachment<AttachmentRawDataDigiwinAthena> attachment = new Attachment<>();
        attachment.setComponentId(rawContent.getComponentId());
        attachment.setTitle(rawContent.getLabel());
        attachment.setEnable(rawContent.getEditable());
        attachment.setType(BooleanUtils.isTrue(rawContent.getEditable()) ? AttachmentTypeEnum.READ_AND_WRITE.getValue() : AttachmentTypeEnum.READ_ONLY.getValue());
        attachment.setStyleType(AttachmentStyleTypeEnum.TITLE_WEAKENED.getValue());
        Card.DesignerRawCard.Content.AttachmentObj attachmentObj = JsonUtil.objectToJavaObject(rawContent.getSetting(), Card.DesignerRawCard.Content.AttachmentObj.class);
        if (attachmentObj != null) {
            attachment.setPlaceholder(attachmentObj.getPlaceholder());
            attachment.setLimit(attachmentObj.getLimit());
            attachment.setFileMaxSize(attachmentObj.getFileMaxSize());
            attachment.setUploadTypes(attachmentObj.getUploadTypes());
            attachment.setIsFold(attachmentObj.getIsFold());
            attachment.setFoldNumber(attachmentObj.getFoldNumber());
            attachment.setDefaultExpand(attachmentObj.getDefaultExpand());
            AttachmentRawDataDigiwinAthena rawDataDigiwinAthena = new AttachmentRawDataDigiwinAthena();
            rawDataDigiwinAthena.setBuckets(attachmentObj.getBuckets());
            if (attachmentObj.getRawData() != null) {
                rawDataDigiwinAthena.setDisableAam(attachmentObj.getRawData().getDisableAam());
                rawDataDigiwinAthena.setUploadCategory(attachmentObj.getRawData().getUploadCategory());
            }
            attachment.setRawData(rawDataDigiwinAthena);
        }
        attachment.handleComponentParam(cmptContext, rawContent.getFullPath(), rawContent.getSchema(), data, executeContext, mobilePageRawData);
        return attachment;
    }

    @Override
    public void applyEditable(Boolean enable) {
        super.applyEditable(enable);
        if (BooleanUtils.isTrue(enable)) {
            // 可编辑
            this.type = AttachmentTypeEnum.READ_AND_WRITE.getValue();
        } else {
            // 不可编辑
            this.type = AttachmentTypeEnum.READ_ONLY.getValue();
        }
    }
}