package com.digiwin.mobile.mobileuibot.core.component.input.calendar;

import com.alibaba.fastjson.annotation.JSONField;
import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.core.component.BaseInputMobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.ComponentContext;
import com.digiwin.mobile.mobileuibot.core.component.MobilePageRawData;
import com.digiwin.mobile.mobileuibot.core.rule.Rule;
import com.digiwin.mobile.mobileuibot.designer.uibot.service.UiBotDesignerRenderService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotExecuteContext;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.rule.UiBotRule;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumn;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.Data;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.time.LocalDate;
import java.util.*;

/**
 * <p>功能描述：日历选择输入组件</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: InputCalendar
 * @Author: Zaregoto
 * @Date: 2021/6/14 15:30
 */
@Data
public class InputCalendar extends BaseInputMobileComponent implements Serializable {
    private static final long serialVersionUID = 963291030278645004L;
    public static final String COMPONENT_TYPE = "INPUT_CALENDAR";

    /**
     * 日历的日期列表。
     * 展示时，里面的值是后台输入的值；提交时，里面的值是用户前端输入的值
     */
    private List<InputCalendarDate> dateValueList = new ArrayList<>();

    /**
     * 日历的最大日期
     */
    private InputCalendarDate maxDate = InputCalendarDate.create("2121", "12", "31");

    /**
     * 日历的最小日志
     */
    private InputCalendarDate minDate = InputCalendarDate.create("2000", "01", "01");

    /**
     * 可选的最大日期
     */
    private InputCalendarDate maxDateSelectable;

    /**
     * 可选的最小日期
     */
    private InputCalendarDate minDateSelectable;
    private String title;
    /**
     * 1：单选，2：区间
     *
     * @see InputCalendarTypeEnum
     */
    private Integer type;

    /**
     * 日历录入的日期文字。格式为：YYYY-MM-DD 00:00:00.000
     * FIXME 以下字段兼容App端的提交数据格式，后续需要与端上沟通依据不同组件传入对应的Model
     */
    @JsonProperty(access = JsonProperty.Access.WRITE_ONLY, value = "text")
    @JSONField(serialize = false, deserialize = true, name = "text")
    private String inputText;

    /**
     * 设计器新增属性
     * 当type = 2，区间选择时，此字段才有值，提交时会对inputText值进行截取，开始日期赋值给当前schema，结束日期赋值给endSchema
     */
    private String endSchema;
    private String endSchemaPath;

    /**
     * 样式类型：1、默认样式，2、平铺
     *
     * @see InputCalendarStyleTypeEnum
     */
    private Integer styleType = InputCalendarStyleTypeEnum.DEFAULT.getValue();

    public InputCalendar() {

    }

    /**
     * 默认回填日历
     *
     * @param title
     * @param type
     * @param value
     * @param required
     * @param hidden
     * @param enable
     * @param errorMessage
     * @return
     */
    public static InputCalendar create(String title, Integer type, String value,
                                       Boolean required, Boolean hidden, Boolean enable,
                                       String errorMessage) {
        InputCalendar inputCalendar = new InputCalendar();
        inputCalendar.setTitle(title);
        inputCalendar.setType(type);
        inputCalendar.setErrorMessage(errorMessage);
        inputCalendar.setPlaceholder(errorMessage);
        if (StringUtils.hasLength(value) && !StringUtils.isEmpty(value)) {
            String[] values;
            if (value.contains("-")) {
                values = value.split("-");
            } else {
                values = value.split("\\.");
            }
            inputCalendar.getDateValueList().add(InputCalendarDate.create(values[0], values[1], values[2]));
        }
        if (value == null && !required) {
            inputCalendar.setErrorMessage("-");
            inputCalendar.setPlaceholder("-");
        } else if (required) {
            inputCalendar.setErrorMessage(errorMessage);
            inputCalendar.setPlaceholder(errorMessage);
        }
        inputCalendar.setMinDate(InputCalendarDate.create("1900", "01", "01"));
        inputCalendar.setMinDateSelectable(InputCalendarDate.create("1900", "01", "01"));
        inputCalendar.setRequired(required);
        inputCalendar.setHidden(hidden);
        inputCalendar.setEnable(enable);
        return inputCalendar;
    }

    /**
     * 设定日历区间
     *
     * @param title
     * @param type
     * @param minDateSelectable
     * @param maxDateSelectable
     * @param required
     * @param hidden
     * @param enable
     * @return
     */
    public static InputCalendar create(String title, Integer type, InputCalendarDate minDateSelectable,
                                       InputCalendarDate maxDateSelectable, Boolean required,
                                       Boolean hidden, Boolean enable, String errorMessage) {
        InputCalendar inputCalendar = new InputCalendar();
        inputCalendar.setTitle(title);
        inputCalendar.setType(type);
        inputCalendar.setErrorMessage(errorMessage);
        inputCalendar.setPlaceholder(errorMessage);
        inputCalendar.setMaxDateSelectable(maxDateSelectable);
        inputCalendar.setMinDateSelectable(minDateSelectable);
        inputCalendar.setRequired(required);
        inputCalendar.setHidden(hidden);
        inputCalendar.setEnable(enable);
        return inputCalendar;
    }

    public static InputCalendar create(String locale, UiBotTableColumn column, List<UiBotRule> columnRuleList,
                                       Map<String, Object> bizData, List<Rule> mobileRuleList, boolean enable, boolean required,
                                       ApiRequest apiRequest) {
        InputCalendar calendar = new InputCalendar();

        String dateData = (String) bizData.get(column.getSchema());
        if (null != dateData && !dateData.isEmpty()) {
            LocalDate localDate = DateTimeUtil.parseLocalDateUseDefaultPattern(dateData);
            calendar.addDateValue(
                    DateTimeUtil.getFullYear(localDate.getYear()),
                    DateTimeUtil.getFullMonth(localDate.getMonthValue()),
                    DateTimeUtil.getFullDate(localDate.getDayOfMonth())
            );
        }

        calendar.setMinDate(InputCalendarDate.create("2000", "01", "01"));
        calendar.setMaxDate(InputCalendarDate.create("2121", "12", "31"));
        // FIXME 20221011 临时方案：解决PCC应用的实际完成日字段可选择未来日期。后面要改成读数据库配置
        if ("project_info".equalsIgnoreCase(column.getPath()) &&
                ("actual_start_date".equalsIgnoreCase(column.getSchema()) ||
                        "actual_finish_date".equalsIgnoreCase(column.getSchema()))
        ) {
            calendar.setMaxDateSelectable(InputCalendarDate.createToday());
        }
        if ("task_info".equalsIgnoreCase(column.getPath())
                && ("actual_finish_date".equalsIgnoreCase(column.getSchema())
                || "actual_start_date".equalsIgnoreCase(column.getSchema()))
        ) {
            calendar.setMaxDateSelectable(InputCalendarDate.createToday());
        }

        calendar.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请选择"));

        // 是否必填
        calendar.setRequired(required);
        calendar.setTitle(column.getHeaderName());
        // FIXME 目前没有规则来知道日历选择是选几个日期，先固定选择单个日期
        calendar.setType(InputCalendarTypeEnum.SELECT_SINGLE_DATE.getValue());
        // FIXME 目前先使用以下提示
        calendar.setErrorMessage(LocaleUtil.getMobileTextByKey(locale, "完成率100%后可填"));
        calendar.setEnable(enable);

        calendar.addAllMobileRule(bizData, column.getSchema(), mobileRuleList, locale, apiRequest, Collections.emptyMap(), "");

        return calendar;
    }

    public static InputCalendar create(UiBotLayout uiBotFieldLayout, UiBotRule uiBotRule, String locale) {
        InputCalendar inputCalendar = new InputCalendar();
        inputCalendar.setSchema(uiBotFieldLayout.getSchema());
        inputCalendar.setTitle(uiBotFieldLayout.getHeaderName());
        inputCalendar.setType(InputCalendarTypeEnum.SELECT_SINGLE_DATE.getValue());
        inputCalendar.setMinDate(InputCalendarDate.create("2000", "01", "01"));
        inputCalendar.setMaxDate(InputCalendarDate.create("2121", "12", "31"));
        inputCalendar.setMinDateSelectable(InputCalendarDate.create("2000", "01", "01"));
        inputCalendar.setMaxDateSelectable(InputCalendarDate.create("2121", "12", "31"));
        inputCalendar.setEnable(true);
        inputCalendar.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请选择"));
        if (uiBotRule != null) {
            inputCalendar.setRequired("required".equalsIgnoreCase(uiBotRule.getKey()));
        } else {
            inputCalendar.setRequired(false);
        }
        return inputCalendar;
    }

    public static InputCalendar create(UiBotLayout groupLayout, List<String> values, Integer type) {
        InputCalendar inputCalendar = new InputCalendar();
        inputCalendar.setType(type);
        inputCalendar.setSchema(groupLayout.getSchema());
        inputCalendar.setTitle(StringUtils.hasLength(groupLayout.getHeaderName()) ? groupLayout.getHeaderName() : groupLayout.getPlaceholder());
        inputCalendar.setMinDate(InputCalendarDate.create("2000", "01", "01"));
        inputCalendar.setMaxDate(InputCalendarDate.create("2121", "12", "31"));
        inputCalendar.setMinDateSelectable(InputCalendarDate.create("2000", "01", "01"));
        inputCalendar.setMaxDateSelectable(InputCalendarDate.create("2121", "12", "31"));
        inputCalendar.setEnable(true);
        inputCalendar.setRequired(true);
        return inputCalendar;
    }

    public void addDateValue(String year, String month, String date) {
        this.dateValueList.add(InputCalendarDate.create(year, month, date));
    }

    @Override
    public String returnComponentType() {
        return InputCalendar.COMPONENT_TYPE;
    }

    @Override
    public void handleComponentParam(ComponentContext cmptContext, String mobilePath, String schema, Map<String, Object> data,
                                     PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        super.handleComponentParam(cmptContext, mobilePath, schema, data, executeContext, mobilePageRawData);
        String value = Optional.ofNullable(UiBotDesignerRenderService.getDataByPathAndSchema(data, mobilePath, schema))
                .map(Object::toString)
                .orElse(null);
        if (StringUtils.hasLength(value)) {
            String[] values;
            if (value.contains("-")) {
                values = value.split("-");
            } else {
                values = value.split("\\.");
            }
            this.dateValueList.add(InputCalendarDate.create(values[0], values[1], values[2]));
        }
    }
}
