package com.digiwin.mobile.mobileuibot.core.component.input.cascade;

import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.context.SpringContextHolder;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.string.StringUtil;
import com.digiwin.mobile.mobileuibot.core.component.BaseInputMobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.ComponentContext;
import com.digiwin.mobile.mobileuibot.core.component.MobilePageRawData;
import com.digiwin.mobile.mobileuibot.designer.uibot.service.UiBotDesignerRenderService;
import com.digiwin.mobile.mobileuibot.proxy.eoc.model.EocOrg;
import com.digiwin.mobile.mobileuibot.proxy.eoc.service.impl.DigiwinEocProxyServiceImpl;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotExecuteContext;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * <p>功能描述：运营单元组件--v2版组织对应组件</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: InputEocSelect.java
 * @Author: wangjwc
 * @Date: created at 2025/5/27 16:58
 */
@Data
public class InputEocSelect extends BaseInputMobileComponent {
    public static final String COMPONENT_TYPE = "EOC_SELECT";
    private static final long serialVersionUID = -3445324895812288367L;
    /**
     * 当前栏位的名称
     */
    private String title;
    /**
     * 最大层级数
     */
    private Integer levelNums;
    /**
     * eoc结构数据
     */
    private EocOrg content;
    /**
     * 默认选中的项
     */
    private EocOrg select;

    /**
     * 提交的值 -- 选中的节点
     */
    private EocOrg data;

    @Override
    public String returnComponentType() {
        return InputEocSelect.COMPONENT_TYPE;
    }

    @Override
    public void handleComponentParam(ComponentContext cmptContext, String mobilePath, String schema, Map<String, Object> data, PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        super.handleComponentParam(cmptContext, mobilePath, schema, data, executeContext, mobilePageRawData);
        this.content = SpringContextHolder.getBean(DigiwinEocProxyServiceImpl.class).getEocOrgTree(AppRequestContext.getContextEntity().getIamUserToken());
        this.levelNums = getMaxDepthRecursive(this.content);
        // 构建默认选中项
        this.select = handleDefaultSelect(this.content, data, mobilePath, schema);
    }

    /**
     * 递归计算树的最大深度
     * @param root 树的根节点
     * @return 最大深度（根节点为第1层）
     */
    public static int getMaxDepthRecursive(EocOrg root) {
        if (root == null) {
            return 0;
        }
        int maxChildDepth = 0;
        for (EocOrg child : root.getChildren()) {
            int childDepth = getMaxDepthRecursive(child);
            if (childDepth > maxChildDepth) {
                maxChildDepth = childDepth;
            }
        }
        // 当前层 + 子节点的最大深度
        return maxChildDepth + 1;
    }

    private EocOrg handleDefaultSelect(EocOrg content, Map<String, Object> data, String mobilePath, String schema) {
        if (content == null) {
            return null;
        }
        // wjw TODO: 2025/5/29 假设api中存储的Eoc组织信息为当前选择的组织ID
        Object o = UiBotDesignerRenderService.getDataByPathAndSchema(data, mobilePath, schema);
        if (o != null) {
            // 获取默认选项
            return findPathTree(content, StringUtil.valueOf(o));
        }
        return null;
    }

    /**
     * 在树中查找指定ID的节点路径（树形结构表示）
     * @param root 树的根节点
     * @param targetId 要查找的目标ID
     * @return 包含完整路径的树结构，如果未找到则返回null
     */
    public static EocOrg findPathTree(EocOrg root, String targetId) {
        if (root == null) {
            return null;
        }
        if (StringUtils.isEmpty(targetId)) {
            return null;
        }
        // 1. 查找从根节点到目标节点的路径
        List<EocOrg> path = new ArrayList<>();
        if (!findPath(root, targetId, path)) {
            return null;
        }
        // 2. 根据路径构建树形结构
        return buildPathTree(path);
    }

    /**
     * 递归查找节点路径
     * @param node 当前节点
     * @param targetId 目标ID
     * @param path 当前路径
     * @return 是否找到目标节点
     */
    private static boolean findPath(EocOrg node, String targetId, List<EocOrg> path) {
        // 添加当前节点到路径
        path.add(node);
        // 检查是否找到目标节点
        if (node.getId().equals(targetId)) {
            return true;
        }
        // 递归搜索子节点
        for (EocOrg child : node.getChildren()) {
            if (findPath(child, targetId, path)) {
                // 找到目标节点，直接返回
                return true;
            }
        }
        // 当前分支未找到，回溯移除当前节点
        path.remove(path.size() - 1);
        return false;
    }

    /**
     * 根据路径构建树形结构
     * @param path 节点路径列表
     * @return 路径树的根节点
     */
    private static EocOrg buildPathTree(List<EocOrg> path) {
        if (path.isEmpty()) {
            return null;
        }
        // 复制路径中的节点（深拷贝）
        List<EocOrg> clonedPath = new ArrayList<>();
        for (EocOrg node : path) {
            // 创建新节点（仅复制ID和名称）
            EocOrg eocOrg = JsonUtil.objectToJavaObject(node, EocOrg.class);
            eocOrg.setChildren(new ArrayList<>());
            clonedPath.add(eocOrg);
        }
        // 构建树形结构
        for (int i = 0; i < clonedPath.size() - 1; i++) {
            // 当前节点只包含路径中的下一个节点作为子节点
            clonedPath.get(i).getChildren().add(clonedPath.get(i + 1));
        }
        // 返回根节点
        return clonedPath.get(0);
    }
}
