package com.digiwin.mobile.mobileuibot.core.component.input.multitext;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.ObjectUtil;
import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.core.component.BaseInputMobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.ComponentContext;
import com.digiwin.mobile.mobileuibot.core.component.MobilePageRawData;
import com.digiwin.mobile.mobileuibot.core.component.basic.HiddenConfig;
import com.digiwin.mobile.mobileuibot.core.component.card.card.Card;
import com.digiwin.mobile.mobileuibot.core.rule.Rule;
import com.digiwin.mobile.mobileuibot.core.rule.RuleInitConfig;
import com.digiwin.mobile.mobileuibot.core.rule.validate.ValidateRuleTypeEnum;
import com.digiwin.mobile.mobileuibot.designer.uibot.service.UiBotDesignerRenderService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotExecuteContext;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.rule.UiBotRule;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumn;
import lombok.Data;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static com.digiwin.mobile.mobileuibot.core.layout.doublepattern.util.ModuleUtils.localeService;

/**
 * <p>功能描述：多行文本录入组件</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: InputMultiText
 * @Author: Zaregoto
 * @Date: 2021/6/14 15:14
 */
@Data
public class InputMultiText extends BaseInputMobileComponent implements Serializable {
    public static final String COMPONENT_TYPE = "INPUT_MULTI";
    private static final long serialVersionUID = 2299255092657306155L;
    private String text;
    private String title;
    /**
     * 是否展示输入框背景阴影
     */
    private Boolean showShadow;
    /**
     * 最多字数
     */
    private Integer maxLength;
    /**
     * 支持最大行数
     */
    private Integer maxLines = 4;
    /**
     * 是否启用收起展开功能
     */
    private Boolean useExpand = true;
    private HiddenConfig hiddenConfig = new HiddenConfig();

    public InputMultiText() {
    }

    @Deprecated
    public static InputMultiText create(String title, String text, Boolean enable,
                                        Boolean showShadow, String locale) {
        InputMultiText inputMultiText = new InputMultiText();
        inputMultiText.setTitle(title);
        inputMultiText.setText(StringUtils.hasLength(text) ? text : "-");
        inputMultiText.setEnable(enable);
        inputMultiText.setRequired(false);
        inputMultiText.setShowShadow(showShadow);
        buildHiddenConfig(inputMultiText, locale);
        return inputMultiText;
    }

    public static InputMultiText create(String title, String text, Boolean enable,
                                        Boolean showShadow, String locale, String placeholder) {
        InputMultiText inputMultiText = new InputMultiText();
        inputMultiText.setTitle(title);
        inputMultiText.setText(text);
        inputMultiText.setEnable(enable);
        inputMultiText.setPlaceholder(placeholder);
        inputMultiText.setRequired(false);
        inputMultiText.setShowShadow(showShadow);
        buildHiddenConfig(inputMultiText, locale);
        return inputMultiText;
    }

    public static InputMultiText create(String schema, String title, String text, Boolean enable, Boolean required,
                                        Boolean showShadow, String locale) {
        InputMultiText inputMultiText = new InputMultiText();
        inputMultiText.setSchema(schema);
        inputMultiText.setTitle(title);
        inputMultiText.setText(text);
        inputMultiText.setEnable(enable);
        inputMultiText.setRequired(required);
        inputMultiText.setShowShadow(showShadow);
        buildHiddenConfig(inputMultiText, locale);
        inputMultiText.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        return inputMultiText;
    }

    public static InputMultiText create(String schema, String title, String text, Boolean enable, Boolean required,
                                        Boolean showShadow, String locale, Integer maxLength) {
        InputMultiText inputMultiText = new InputMultiText();
        inputMultiText.setSchema(schema);
        inputMultiText.setTitle(title);
        inputMultiText.setText(text);
        inputMultiText.setEnable(enable);
        inputMultiText.setRequired(required);
        inputMultiText.setShowShadow(showShadow);
        buildHiddenConfig(inputMultiText, locale);
        inputMultiText.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        inputMultiText.setMaxLength(maxLength);
        return inputMultiText;
    }

    public static InputMultiText create(String title, String text, Boolean enable,
                                        Boolean showShadow, List<UiBotRule> ruleList, String schema, String locale) {
        InputMultiText inputMultiText = new InputMultiText();
        inputMultiText.setTitle(title);
        inputMultiText.setText(StringUtils.hasLength(text) ? text : "-");
        inputMultiText.setEnable(enable == null ? false : enable);
        inputMultiText.setRequired(false);
        if (ruleList != null && ruleList.size() > 0) {
            ruleList.stream().filter(uiBotRule -> schema.equalsIgnoreCase(uiBotRule.getSchema()))
                    .forEach(uiBotRule -> {
                        if (ValidateRuleTypeEnum.REQUIRED.getType().equalsIgnoreCase(uiBotRule.getKey())) {
                            inputMultiText.setRequired("required".equalsIgnoreCase(uiBotRule.getKey()));
                        }
                    });
        }
        inputMultiText.setShowShadow(showShadow);
        buildHiddenConfig(inputMultiText, locale);
        return inputMultiText;
    }

    public static InputMultiText create(String locale, UiBotTableColumn column,
                                        List<UiBotRule> columnRuleList, Map<String, Object> bizData, List<Rule> mobileRuleList,
                                        boolean showShadow, Integer maxLength,
                                        Map<String, Object> variableMaps, String errorMessage) {
        InputMultiText inputMultiText = new InputMultiText();
        inputMultiText.setTitle(column.getHeaderName());
        inputMultiText.setText(column.getValueForDisplay(bizData));
        inputMultiText.setSchema(column.getSchema());
        inputMultiText.setEnable(true);
        inputMultiText.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        if (maxLength != -1) {
            inputMultiText.setMaxLength(maxLength);
        }
        Boolean hasRule = false;
        for (Rule rule : mobileRuleList) {
            if (rule.getSchema().equalsIgnoreCase(column.getSchema())) {
                RuleInitConfig initConfig = rule.getInitConfig();
                if (ObjectUtils.isNotEmpty(initConfig)) {
                    if (variableMaps != null && (Boolean) variableMaps.get(initConfig.getKey())) {
                        inputMultiText.setRequired(false);
                    } else {
                        inputMultiText.setRequired(initConfig.getRequired());
                    }
                }
                hasRule = true;
            }
        }
        if (column.getMRequired() != null && !hasRule) {
            inputMultiText.setRequired(column.getMRequired());
        }
        /**
         * TODO PCC任务客制，太过于复杂，还没想好如何定值规则
         * progressCancel -是否支持进度回退
         * isWorkReportRequired -  报工说明， 是否必填
         * 1.progressCancel为true，则报工说明字段必填，false则非必填（S3迭代需求）
         * 2.isWorkReportRequired为true、实际完成日非空且实际完成日大于预计完成日，则必填，反之为非必填（S5迭代需求）
         */
        if ("report_work_description".equalsIgnoreCase(column.getSchema()) && variableMaps != null) {
            if ((Boolean) variableMaps.get("isWorkReportRequired")) {
                inputMultiText.setRequired(false);
            }
        }
        /*
        处理云管家-手动报工逾期报工原因:
            依据交付设计器.逾期报工开关开开，则判断是否超过可预期天数，超过则这里要求必填，不超过则非必填
            获取逾期报工开关 overdue_report
            获取可逾期天数（天）overdue_report_day
         */
        if ("task_info".equalsIgnoreCase(column.getPath())
                && "overdue_task_report_reason".equalsIgnoreCase(column.getSchema())
                && CollUtil.isNotEmpty(variableMaps)
                && ObjectUtil.isNotEmpty(variableMaps.get("overdue_report"))
                && (Boolean) variableMaps.get("overdue_report")) {
            //计划结束日期
            String planFinishDate = variableMaps.get("plan_finish_date").toString();
            //获取可逾期天数（天）
            double overdueReportDay = (double) variableMaps.get("overdue_report_day");
            DateTime dateTime = DateUtil.parseDate(planFinishDate);
            //偏移的天数 这里加1 是因为isBeforeNowDateTime获取当前的有时分秒
            DateTime offsetDay = DateUtil.offsetDay(dateTime, (int) overdueReportDay + 1);
            Boolean beforeNowDateTime = DateTimeUtil.isBeforeNowDateTime(offsetDay.toString());
            inputMultiText.setRequired(beforeNowDateTime);
        }

        buildHiddenConfig(inputMultiText, locale);
        inputMultiText.setShowShadow(showShadow);

        return inputMultiText;
    }

    public static InputMultiText create(UiBotLayout uiBotFieldLayout, UiBotRule uiBotRule, String locale,
                                        Boolean showShadow, String fieldHandlingInstructionsRequired) {
        InputMultiText inputMultiText = new InputMultiText();
        if (uiBotRule != null) {
            inputMultiText.setRequired("required".equalsIgnoreCase(uiBotRule.getKey()));
        }
        if (uiBotFieldLayout != null) {
            inputMultiText.setTitle(uiBotFieldLayout.getHeaderName());
            if ("question_info".equalsIgnoreCase(uiBotFieldLayout.getPath())) {
                // 1，交付设计器-「现场处理说明是否必填」=启用，则该栏位要求必填
                // 2，交付设计器-「现场处理说明是否必填」=不启用，则该栏位要求非必填
                if ("scene_process_illustrate".equalsIgnoreCase(uiBotFieldLayout.getSchema())) {
                    inputMultiText.setRequired("Y".equalsIgnoreCase(fieldHandlingInstructionsRequired));
                }
                inputMultiText.setMaxLength(400);
            }
        }
        inputMultiText.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        inputMultiText.setEnable(true);
        inputMultiText.setShowShadow(true);
        if (showShadow != null) {
            inputMultiText.setShowShadow(showShadow);
        }
        buildHiddenConfig(inputMultiText, locale);
        return inputMultiText;
    }

    public static InputMultiText create(UiBotLayout uiBotLayout, List<Rule> mobileRuleList, Map<String, Object> data, String locale, boolean showShadow, Boolean isComplete) {
        InputMultiText inputMultiText = new InputMultiText();
        String desc = (String) data.get(uiBotLayout.getSchema());
        if (StringUtils.hasLength(desc)) {
            inputMultiText.setText(desc);
        }
        inputMultiText.setTitle(uiBotLayout.getmLabelText().get(locale));
        inputMultiText.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        inputMultiText.setEnable(!isComplete);
        inputMultiText.setRequired(!isComplete);
        for (Rule rule : mobileRuleList) {
            if (uiBotLayout.getSchema().equals(rule.getSchema())) {
                inputMultiText.setRequired(rule.getValid());
            }
        }
        buildHiddenConfig(inputMultiText, locale);
        inputMultiText.setShowShadow(showShadow);
        return inputMultiText;
    }

    public static HiddenConfig buildHiddenConfig(InputMultiText inputMultiText, String locale) {
        HiddenConfig hiddenConfig = inputMultiText.getHiddenConfig();
        hiddenConfig.setHiddenCount(4);
        hiddenConfig.setHiddenUnexpandContent(LocaleUtil.getMobileTextByKey(locale, "收起"));
        hiddenConfig.setHiddenExpandContent(LocaleUtil.getMobileTextByKey(locale, "查看更多"));
        return hiddenConfig;
    }

    public static InputMultiText create(String locale, String schema, String headerName) {
        InputMultiText inputMultiText = new InputMultiText();
        inputMultiText.setTitle(headerName);
        inputMultiText.setSchema(schema);
        inputMultiText.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        inputMultiText.setEnable(false);
        inputMultiText.setShowShadow(false);
        return inputMultiText;
    }

    @Override
    public String returnComponentType() {
        return InputMultiText.COMPONENT_TYPE;
    }

    @Override
    public void handleComponentParam(ComponentContext cmptContext, String mobilePath, String schema, Map<String, Object> data,
                                     PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        super.handleComponentParam(cmptContext, mobilePath, schema, data, executeContext, mobilePageRawData);
        if ("PWD_RTP_InquiryReply".equals(executeContext.getTmActivityId()) && "supplier_remark".equals(schema)) {
            // 带图采购-询价回复-更多选填-供应商备注-定制
            this.maxLength = 255;
        }
        this.showShadow = true;
        String value = Optional.ofNullable(UiBotDesignerRenderService.getDataByPathAndSchema(data, mobilePath, schema))
                .map(Object::toString)
                .orElse(null);
        if (StringUtils.hasLength(value)) {
            this.text = value;
        } else {
            if (BooleanUtils.isTrue(this.enable)) {
                // 输入型，暂定无默认值
            } else {
                this.text = "-";
            }
        }
        //提示语给默认值
        if (ObjectUtils.isEmpty(this.placeholder)) {
            this.placeholder = localeService.getLanguageValue(executeContext.getLocale(), "请输入");
        }
    }

    public static BaseMobileComponent cardContentConvertComponent(Card.DesignerRawCard.Content rawContent, ComponentContext cmptContext, Map<String, Object> data, PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        InputMultiText inputMultiText = new InputMultiText();
        inputMultiText.setComponentId(rawContent.getComponentId());
        inputMultiText.setTitle(rawContent.getLabel());
        inputMultiText.setEnable(rawContent.getEditable());
        Card.DesignerRawCard.Content.MultiTextObj multiTextObj = JsonUtil.objectToJavaObject(rawContent.getSetting(), Card.DesignerRawCard.Content.MultiTextObj.class);
        if (multiTextObj != null) {
            inputMultiText.setPlaceholder(multiTextObj.getPlaceholder());
            inputMultiText.setMaxLength(multiTextObj.getMaxLength());
        }
        inputMultiText.handleComponentParam(cmptContext, rawContent.getFullPath(), rawContent.getSchema(), data, executeContext, mobilePageRawData);
        return inputMultiText;
    }
}
