package com.digiwin.mobile.mobileuibot.core.component.input.numeric;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.common.context.SpringContextHolder;
import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.common.math.MathUtil;
import com.digiwin.mobile.mobileuibot.core.component.BaseInputMobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.ComponentContext;
import com.digiwin.mobile.mobileuibot.core.component.MobilePageRawData;
import com.digiwin.mobile.mobileuibot.core.component.button.Button;
import com.digiwin.mobile.mobileuibot.core.component.card.card.Card;
import com.digiwin.mobile.mobileuibot.core.layout.doublepattern.bean.PcModuleEnum;
import com.digiwin.mobile.mobileuibot.core.rule.Rule;
import com.digiwin.mobile.mobileuibot.core.rule.relation.RelationRule;
import com.digiwin.mobile.mobileuibot.core.rule.relation.RelationRuleText;
import com.digiwin.mobile.mobileuibot.core.rule.validate.ValidateRule;
import com.digiwin.mobile.mobileuibot.core.rule.validate.ValidateRuleScopeEnum;
import com.digiwin.mobile.mobileuibot.core.rule.validate.ValidateRuleTypeEnum;
import com.digiwin.mobile.mobileuibot.designer.uibot.service.UiBotDesignerRenderService;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotExecuteContext;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.rule.UiBotMaxLengthRuleAnalyzer;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.rule.UiBotRule;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumn;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumnEditor;
import com.digiwin.mobile.mobileuibot.proxy.uibot.parser.ParseFieldData;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static com.digiwin.mobile.mobileuibot.core.layout.doublepattern.util.ModuleUtils.localeService;

/**
 * <p>功能描述：数值型录入组件</p>
 * <p>百分比类型：前端展示百分比符号，展示数值为整数。提交时也原样提交后后端，后端解析时需要除以100.0</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: InputNumeric
 * @Author: Zaregoto
 * @Date: 2021/6/14 15:13
 */
@AllArgsConstructor
@NoArgsConstructor
@Data
@Slf4j
public class InputNumeric extends BaseInputMobileComponent implements Serializable {
    private static final long serialVersionUID = -5052577231874837283L;
    public static final String COMPONENT_TYPE = "INPUT_NUMERIC";

    /**
     * 允许输入的最大字符数,不包括小数点
     */
    private Integer maxLength;
    private Integer precision;
    // 初始展示文字
    private String text;
    private String title;
    private Integer type;
    /**
     * 当type为21时有单位
     */
    private String unit;
    private Button button;
    private String buttonValue;

    private Map<String, Object> rawData;


    //是否展示千分位,展示为true，不展示为fasle
    private Boolean thousandthPlace;

    public static InputNumeric create(Integer maxLength, Integer precision, String text, String title,
                                      Integer type) {
        InputNumeric inputNumeric = new InputNumeric();
        inputNumeric.setMaxLength(maxLength);
        inputNumeric.setPrecision(precision);
        inputNumeric.setText(text);
        inputNumeric.setTitle(title);
        inputNumeric.setType(type);
        inputNumeric.setErrorMessage("");
        return inputNumeric;
    }

    public static InputNumeric create(Integer maxLength, Integer precision, String text, String title,
                                      Integer type, Boolean isRequired, String unit) {
        InputNumeric inputNumeric = new InputNumeric();
        inputNumeric.setMaxLength(maxLength);
        inputNumeric.setPrecision(precision);
        inputNumeric.setText(text);
        inputNumeric.setTitle(title);
        inputNumeric.setType(type);
        inputNumeric.setErrorMessage("");
        inputNumeric.setRequired(isRequired);
        inputNumeric.setUnit(unit);
        return inputNumeric;
    }

    public static InputNumeric create(Integer maxLength, Integer precision, String text, String title,
                                      Integer type, RelationRule relationRule, Boolean required) {
        InputNumeric inputNumeric = new InputNumeric();
        inputNumeric.setMaxLength(maxLength);
        inputNumeric.setPrecision(precision);
        inputNumeric.setText(text);
        inputNumeric.setTitle(title);
        inputNumeric.setType(type);
        inputNumeric.setErrorMessage("");
        inputNumeric.setRelationRule(relationRule);
        inputNumeric.setRequired(required);
        inputNumeric.setEnable(required);
        return inputNumeric;
    }

    /**
     * 非必填，只能输入O及正整数
     *
     * @param maxLength
     * @param precision
     * @param title
     * @param type
     * @param required
     * @param locale
     * @param uiBotRuleList
     * @param schema
     * @return
     */
    public static InputNumeric create(Integer maxLength, Integer precision, String title,
                                      Integer type, Boolean required, Boolean enable, String locale,
                                      List<UiBotRule> uiBotRuleList, String schema, LocaleService localeService) {
        InputNumeric inputNumeric = new InputNumeric();
        inputNumeric.setMaxLength(maxLength);
        inputNumeric.setPrecision(precision);
        inputNumeric.setTitle(title);
        inputNumeric.setType(type);
        inputNumeric.setRequired(required);
        inputNumeric.setEnable(enable);

        List<ValidateRule> validateRules = new ArrayList<>(1);
        ValidateRule validateRule = new ValidateRule();
        validateRule.setSchema(schema);
        validateRule.setParameter("^\\d*\\.{0,1}\\d{0,1}$");
        validateRule.setScope("EDIT");
        validateRule.setType("pattern");
        validateRules.add(validateRule);
        inputNumeric.setValidateRuleList(validateRules);
        inputNumeric.setErrorMessage(localeService.getLanguageValue(locale, "请输入正确数字"));

        return inputNumeric;
    }

    /**
     * 创建一个字段可使用的数值型输入组件
     *
     * @param locale
     * @param column         字段信息
     * @param uiBotRuleList  字段的所有规则信息，包括字段本身的，以及字段相关的关联规则
     * @param bizData        业务数据
     * @param mobileRuleList 移动端上本字段所有规则信息
     * @param variableMaps
     * @param localeService
     * @return
     */
    public static InputNumeric create(String locale, UiBotTableColumn column, List<UiBotRule> uiBotRuleList,
                                      Map<String, Object> bizData, List<Rule> mobileRuleList,
                                      ApiRequest apiRequest, Map<String, Object> variableMaps, LocaleService localeService) {
        InputNumeric inputNumeric = new InputNumeric();
        inputNumeric.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        inputNumeric.setTitle(column.getHeaderName());

        /**
         * 类型设置
         * 先默认是整数型，再依下方顺序判断需要设置成什么类型。
         * -- 百分比类型
         * -- 其他
         */
        inputNumeric.setType(InputNumericTypeEnum.INTEGER.getValue());
        String rawValueString = String.valueOf(bizData.get(column.getSchema()) == null ? 0 : bizData.get(column.getSchema()));
        rawValueString = rawValueString.equalsIgnoreCase("0.0") ? "0" : rawValueString;
        BigDecimal bd = new BigDecimal(rawValueString);
        rawValueString = bd.toPlainString();
        inputNumeric.setText(rawValueString);
        if ("PERCENT_INPUT".equalsIgnoreCase(
                Optional.ofNullable(column.getEditor()).map(UiBotTableColumnEditor::getType)
                        .orElse(""))) {
            Double value = Double.parseDouble(rawValueString) * 100.0;
            inputNumeric.setType(InputNumericTypeEnum.PERCENTAGE.getValue());
            inputNumeric.setText(String.valueOf(MathUtil.getDecimalPoint(value, 2, true)));
        }

        /**
         * FIXME 因目前PCUiBot中，一个字段可以有很多个Rule，每个Rule都会有可能有errorMessage；
         * 但这种情况下errorMessage的触发顺序暂时没有定义，所以这里只能把所有错误提示拼在一起，根据语言别在中间拼上“或”，或者“or”
         */
        StringBuffer errMsgBuffer = new StringBuffer();
        //处理云管家-手动报工完成率效验提示语
        if ("task_info".equalsIgnoreCase(column.getPath()) && "complete_rate".equalsIgnoreCase(column.getSchema())) {
            String text = (new BigDecimal(rawValueString).multiply(new BigDecimal(100))).stripTrailingZeros().toPlainString();
            inputNumeric.setText(text);
            //前置任务未完成不可100%报工
            if (CollUtil.isNotEmpty(variableMaps) && ObjectUtil.isNotEmpty(variableMaps.get("isReportableWork"))) {
                if (!(Boolean) variableMaps.get("isReportableWork")) {
                    List<ValidateRule> validateRuleList = new ArrayList<>(1);
                    ValidateRule validateRule1 = new ValidateRule();
                    validateRule1.setSchema(column.getSchema());
                    validateRule1.setScope(ValidateRuleScopeEnum.EDIT.getScope());
                    validateRule1.setType(ValidateRuleTypeEnum.APPOINT.getType());
                    validateRule1.setParameter("100");
                    validateRule1.setErrorMessage(LocaleUtil.getMobileTextByKey(locale, "前置任务未完成不可100%报工"));
                    validateRuleList.add(validateRule1);
                    inputNumeric.setValidateRuleList(validateRuleList);
                }
            }

            uiBotRuleList.forEach(a -> {
                if (null != a.getErrorMessage()) {
                    if (errMsgBuffer.length() == 0) {
                        errMsgBuffer.append(a.getErrorMessage());
                    }
                }
            });
        } else {
            uiBotRuleList.forEach(uiBotRule -> {
                if (null != uiBotRule.getErrorMessage()) {
                    if (errMsgBuffer.length() == 0) {
                        errMsgBuffer.append(uiBotRule.getErrorMessage());
                    } else {
                        errMsgBuffer
                                .append(" ")
                                .append(LocaleUtil.getMobileTextByKey(locale, "或"))
                                .append(" ")
                                .append(uiBotRule.getErrorMessage());
                    }
                }
            });
        }

        inputNumeric.setErrorMessage(errMsgBuffer.toString());

        // 是否必填
        // FIXME 后面改用必填规则解析器来判断
        Optional<UiBotRule> optColumnRequiredRule = uiBotRuleList.stream()
                .filter(uiBotRule ->
                        null != uiBotRule.getSchema()
                                && "required".equalsIgnoreCase(uiBotRule.getKey())
                                && "EDIT".equalsIgnoreCase(uiBotRule.getScope()))
                .findFirst();
        inputNumeric.setRequired(optColumnRequiredRule.isPresent());
        Optional<Rule> optionalRequiredRule = mobileRuleList.stream().filter(
                rule -> column.getSchema().equalsIgnoreCase(rule.getSchema())
                        && "required".equalsIgnoreCase(rule.getType())
        ).findFirst();
        if (optionalRequiredRule.isPresent()) {
            inputNumeric.setRequired(true);
        }

        // FIXME 因为数据上还没看到有最大值的限制，但在移动端应该有限制，否则输入体验不佳，故先在移动端设置
        UiBotMaxLengthRuleAnalyzer ruleAnalyzer = SpringContextHolder.getBean(UiBotMaxLengthRuleAnalyzer.class);
        UiBotRule artificialUiBotRule = new UiBotRule();
        artificialUiBotRule.setSchema(column.getSchema());
        artificialUiBotRule.setKey("maxLength");
        inputNumeric.setMaxLength(ruleAnalyzer.analyzeByDefault(artificialUiBotRule));

        // FIXME 目前无客制也无规则限制，暂定默认可用
        inputNumeric.setEnable(true);

        // 如果页面上有撤回按钮，完成率和工时如果是不可编辑，规则和必填不需要添加，否则会影响提交按钮高亮
        if (!CollectionUtils.isEmpty(variableMaps) && MapUtils.getBooleanValue(variableMaps, "withdraw")
                && (("complete_rate".equalsIgnoreCase(column.getSchema()) && !column.canEdit())
                || ("work_hours".equalsIgnoreCase(column.getSchema()) && !column.canEdit()))) {
            inputNumeric.setValidateRuleList(Collections.emptyList());
            inputNumeric.setRelationRule(null);
            inputNumeric.setRequired(false);
        } else {
            inputNumeric.addAllMobileRule(bizData, column.getSchema(), mobileRuleList, locale, apiRequest, variableMaps, inputNumeric.getText());
        }

        RelationRule relationRule = inputNumeric.getRelationRule();
        if (relationRule != null && !CollectionUtils.isEmpty(relationRule.getTextRuleList())) {
            List<RelationRuleText> textRuleList = relationRule.getTextRuleList();
            List<RelationRuleText> newTextRuleList = new ArrayList<>(textRuleList.size());
            for (RelationRuleText relationRuleText : textRuleList) {
                if ("report_work_description".equalsIgnoreCase(relationRuleText.getTargetSchema())) {
                    String data = (String) bizData.get("plan_finish_date");
                    Boolean isWorkReportRequired = false;
                    if (!CollectionUtils.isEmpty(variableMaps)) {
                        isWorkReportRequired = (Boolean) variableMaps.get("isWorkReportRequired");
                    }
                    if (isWorkReportRequired &&
                            StringUtils.hasLength(data) && DateTimeUtil.isBeforeNowDate(data)) {
                        relationRuleText.setTargetPlaceholder(localeService.getLanguageValue(locale, relationRuleText.getTargetPlaceholder()));
                        newTextRuleList.add(relationRuleText);
                    }
                } else {
                    newTextRuleList.add(relationRuleText);
                }
            }
            relationRule.setTextRuleList(newTextRuleList);
        }
        return inputNumeric;
    }

    /**
     * 创建一个字段可使用的数值型输入组件
     *
     * @param locale
     * @param fieldData     字段所以信息
     * @param uiBotRuleList 字段的所有规则信息，包括字段本身的，以及字段相关的关联规则
     * @return
     */
    public static InputNumeric create(String locale, ParseFieldData fieldData, List<UiBotRule> uiBotRuleList) {
        InputNumeric inputNumeric = new InputNumeric();
        inputNumeric.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        inputNumeric.setTitle(fieldData.getLabel());

        /**
         * 类型设置
         * 先默认是整数型，再依下方顺序判断需要设置成什么类型。
         * -- 百分比类型
         * -- 其他
         */
        String rawValueString;
        if (fieldData.getData() instanceof Integer
                || fieldData.getData() instanceof Double
                || fieldData.getData() instanceof Float) {
            if (PcModuleEnum.PERCENT_INPUT.getValue().equals(fieldData.getType())) {
                inputNumeric.setType(InputNumericTypeEnum.PERCENTAGE.getValue());
            } else {
                inputNumeric.setType(InputNumericTypeEnum.INTEGER.getValue());
            }

            rawValueString = String.valueOf(fieldData.getData());
        } else {
            inputNumeric.setType(InputNumericTypeEnum.INTEGER.getValue());
            rawValueString = "0";
        }

        inputNumeric.setText(rawValueString);

        /**
         * FIXME 因目前PCUiBot中，一个字段可以有很多个Rule，每个Rule都会有可能有errorMessage；
         * 但这种情况下errorMessage的触发顺序暂时没有定义，所以这里只能把所有错误提示拼在一起，根据语言别在中间拼上“或”，或者“or”
         */
        StringBuffer errMsgBuffer = new StringBuffer();
        uiBotRuleList.forEach(uiBotRule -> {
            if (null != uiBotRule.getErrorMessage()) {
                if (errMsgBuffer.length() == 0) {
                    errMsgBuffer.append(uiBotRule.getErrorMessage());
                } else {
                    errMsgBuffer
                            .append(" ")
                            .append(LocaleUtil.getMobileTextByKey(locale, "或"))
                            .append(" ")
                            .append(uiBotRule.getErrorMessage());
                }
            }
        });
        inputNumeric.setErrorMessage(errMsgBuffer.toString());

        // 是否必填
        // FIXME 后面改用必填规则解析器来判断
        Optional<UiBotRule> optColumnRequiredRule = uiBotRuleList.stream()
                .filter(uiBotRule ->
                        null != uiBotRule.getSchema()
                                && "required".equalsIgnoreCase(uiBotRule.getKey())
                                && "EDIT".equalsIgnoreCase(uiBotRule.getScope()))
                .findFirst();
        inputNumeric.setRequired(optColumnRequiredRule.isPresent());

        // FIXME 因为数据上还没看到有最大值的限制，但在移动端应该有限制，否则输入体验不佳，故先在移动端设置
        UiBotMaxLengthRuleAnalyzer ruleAnalyzer = SpringContextHolder.getBean(UiBotMaxLengthRuleAnalyzer.class);
        UiBotRule artificialUiBotRule = new UiBotRule();
        artificialUiBotRule.setSchema(fieldData.getSchema());
        artificialUiBotRule.setKey("maxLength");
        inputNumeric.setMaxLength(ruleAnalyzer.analyzeByDefault(artificialUiBotRule));

        // FIXME 目前无客制也无规则限制，暂定默认可用
        inputNumeric.setEnable(true);

        return inputNumeric;
    }

    /**
     * 创建一个字段可使用的数值型输入组件
     *
     * @param locale
     * @param fieldData     字段所以信息
     * @param uiBotRuleList 字段的所有规则信息，包括字段本身的，以及字段相关的关联规则
     * @param button        值右侧按钮
     * @param buttonValue   值右侧按钮点击值
     * @return
     */
    public static InputNumeric create(String locale, ParseFieldData fieldData, List<UiBotRule> uiBotRuleList, Button button, String buttonValue, boolean isRequire) {
        InputNumeric inputNumeric = new InputNumeric();
        inputNumeric.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        inputNumeric.setTitle(fieldData.getLabel());

        /**
         * 类型设置
         * 先默认是整数型，再依下方顺序判断需要设置成什么类型。
         * -- 百分比类型
         * -- 其他
         */
        if (fieldData.getData() instanceof Integer
                || fieldData.getData() instanceof Double
                || fieldData.getData() instanceof Float) {
            if (PcModuleEnum.PERCENT_INPUT.getValue().equals(fieldData.getType())) {
                inputNumeric.setType(InputNumericTypeEnum.PERCENTAGE.getValue());
            } else {
                inputNumeric.setType(InputNumericTypeEnum.INTEGER.getValue());
            }
        } else {
            inputNumeric.setType(InputNumericTypeEnum.INTEGER.getValue());
        }

        /**
         * FIXME 因目前PCUiBot中，一个字段可以有很多个Rule，每个Rule都会有可能有errorMessage；
         * 但这种情况下errorMessage的触发顺序暂时没有定义，所以这里只能把所有错误提示拼在一起，根据语言别在中间拼上“或”，或者“or”
         */
        StringBuffer errMsgBuffer = new StringBuffer();
        uiBotRuleList.forEach(uiBotRule -> {
            if (null != uiBotRule.getErrorMessage()) {
                if (errMsgBuffer.length() == 0) {
                    errMsgBuffer.append(uiBotRule.getErrorMessage());
                } else {
                    errMsgBuffer
                            .append(" ")
                            .append(LocaleUtil.getMobileTextByKey(locale, "或"))
                            .append(" ")
                            .append(uiBotRule.getErrorMessage());
                }
            }
        });
        inputNumeric.setErrorMessage(errMsgBuffer.toString());

        // 是否必填
        // FIXME 后面改用必填规则解析器来判断
        Optional<UiBotRule> optColumnRequiredRule = uiBotRuleList.stream()
                .filter(uiBotRule ->
                        null != uiBotRule.getSchema()
                                && "required".equalsIgnoreCase(uiBotRule.getKey())
                                && "EDIT".equalsIgnoreCase(uiBotRule.getScope()))
                .findFirst();
        inputNumeric.setRequired(optColumnRequiredRule.isPresent());

        // FIXME 因为数据上还没看到有最大值的限制，但在移动端应该有限制，否则输入体验不佳，故先在移动端设置
        UiBotMaxLengthRuleAnalyzer ruleAnalyzer = SpringContextHolder.getBean(UiBotMaxLengthRuleAnalyzer.class);
        UiBotRule artificialUiBotRule = new UiBotRule();
        artificialUiBotRule.setSchema(fieldData.getSchema());
        artificialUiBotRule.setKey("maxLength");
        inputNumeric.setMaxLength(ruleAnalyzer.analyzeByDefault(artificialUiBotRule));

        // FIXME 目前无客制也无规则限制，暂定默认可用
        inputNumeric.setEnable(true);
        inputNumeric.setRequired(isRequire);

        inputNumeric.setButton(button);
        inputNumeric.setButtonValue(buttonValue);

        return inputNumeric;
    }

    public void addValidateRule(ValidateRule validateRule) {
        if (null != validateRule && !validateRule.ruleIsEmpty()) {
            this.getValidateRuleList().add(validateRule);
        }
    }

    public Integer getMaxLength() {
        return maxLength;
    }

    public void setMaxLength(Integer maxLength) {
        this.maxLength = maxLength;
    }

    public Integer getPrecision() {
        return precision;
    }

    public void setPrecision(Integer precision) {
        this.precision = precision;
    }

    public String getText() {
        return text;
    }

    public void setText(String text) {
        this.text = text;
    }

    public String getTitle() {
        return title;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public Integer getType() {
        return type;
    }

    public void setType(Integer type) {
        this.type = type;
    }

    @Override
    public String returnComponentType() {
        return InputNumeric.COMPONENT_TYPE;
    }

    @Override
    public void handleComponentParam(ComponentContext cmptContext, String mobilePath, String schema, Map<String, Object> data,
                                     PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        super.handleComponentParam(cmptContext, mobilePath, schema, data, executeContext, mobilePageRawData);
        String value = Optional.ofNullable(UiBotDesignerRenderService.getDataByPathAndSchema(data, mobilePath, schema))
                .map(Object::toString)
                .orElse(null);
        if (StringUtils.hasLength(value)) {
            this.text = removeExcessZero(value);
        } else {
            this.text = "";
        }
        //TODO:3/30月底发版，临时注释
        //if (ObjectUtil.equals(InputNumericTypeEnum.INTEGER.getValue(), this.type)) {
        //    //正整数的时候要给precision为0
        //    this.precision = 0;
        //}
        //提示语给默认值
        if (ObjectUtils.isEmpty(this.placeholder)) {
            this.placeholder = localeService.getLanguageValue(executeContext.getLocale(), "请输入");
        }

        //整数设置精度 precision = 0
        if (Objects.equals(InputNumericTypeEnum.INTEGER.getValue(), this.type)) {
            this.precision = 0;
        }
    }

    /**
     * 移除小数点后多余0
     */
    private String removeExcessZero(String value) {
        try {
            return new BigDecimal(value).stripTrailingZeros().toPlainString();
        } catch (Exception e) {
            log.error("removeExcessZero error, text:{}, error:{}", value, e.getMessage());
        }
        return value;
    }

    public static BaseMobileComponent cardContentConvertComponent(Card.DesignerRawCard.Content rawContent, ComponentContext cmptContext, Map<String, Object> data, PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        InputNumeric inputNumeric = new InputNumeric();
        inputNumeric.setComponentId(rawContent.getComponentId());
        inputNumeric.setTitle(rawContent.getLabel());
        inputNumeric.setEnable(rawContent.getEditable());
        Card.DesignerRawCard.Content.NumericalObj numericalObj = JsonUtil.objectToJavaObject(rawContent.getSetting(), Card.DesignerRawCard.Content.NumericalObj.class);
        if (numericalObj != null) {
            inputNumeric.setPlaceholder(numericalObj.getPlaceholder());
            inputNumeric.setUnit(numericalObj.getUnit());
            inputNumeric.setPrecision(numericalObj.getDecimalPoint());
            inputNumeric.setThousandthPlace(numericalObj.getThousandthPlace());
            inputNumeric.setType(numericalObj.getType());
        }
        inputNumeric.setType(InputNumericTypeEnum.INTEGER.getValue());
        inputNumeric.handleComponentParam(cmptContext, rawContent.getFullPath(), rawContent.getSchema(), data, executeContext, mobilePageRawData);
        return inputNumeric;
    }
}
