package com.digiwin.mobile.mobileuibot.core.component.input.singletext;

import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.core.CommonRawDataParentType;
import com.digiwin.mobile.mobileuibot.core.component.BaseInputMobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.ComponentContext;
import com.digiwin.mobile.mobileuibot.core.component.MobilePageRawData;
import com.digiwin.mobile.mobileuibot.core.component.action.Action;
import com.digiwin.mobile.mobileuibot.core.component.button.Button;
import com.digiwin.mobile.mobileuibot.core.component.card.card.Card;
import com.digiwin.mobile.mobileuibot.core.rule.Rule;
import com.digiwin.mobile.mobileuibot.core.rule.RuleCategoryEnum;
import com.digiwin.mobile.mobileuibot.core.rule.validate.ValidateRule;
import com.digiwin.mobile.mobileuibot.core.rule.validate.ValidateRuleScopeEnum;
import com.digiwin.mobile.mobileuibot.core.rule.validate.ValidateRuleTypeEnum;
import com.digiwin.mobile.mobileuibot.designer.uibot.service.UiBotDesignerRenderService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotExecuteContext;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotAction;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.rule.UiBotRule;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumn;
import lombok.Data;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static com.digiwin.mobile.mobileuibot.core.layout.doublepattern.util.ModuleUtils.localeService;

/**
 * <p>功能描述：单行文本录入组件</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: InputSingleText
 * @Author: Zaregoto
 * @Date: 2021/6/14 15:14
 */
@Data
public class InputSingleText extends BaseInputMobileComponent implements Serializable {
    private static final long serialVersionUID = -2804973497917889622L;
    public static final String COMPONENT_TYPE = "INPUT_SINGLE";

    private String text;
    private String title;
    /**
     * 输入框可输入的类型,1：普通键盘；2：车牌输入
     */
    private Integer inputType = InputTypeEnum.NORMAL_INPUT.getValue();

    /**
     * 单位
     */
    private String unit;

    /**
     * 设计态点击跳转事件
     */
    private UiBotAction inputButton;

    /**
     * 返回给前端
     */
    private Action inputAction;

    //跳转的描述
    private String actionName;

    /**
     * 最多字数
     */
    private Integer maxLength;

    public InputSingleText() {
    }

    public static InputSingleText create(String title, String text) {
        InputSingleText inputSingleText = new InputSingleText();
        inputSingleText.setTitle(title);
        inputSingleText.setText(text);

        return inputSingleText;
    }

    public static InputSingleText create(String title, String text, boolean enable) {
        InputSingleText inputSingleText = new InputSingleText();
        inputSingleText.setTitle(title);
        inputSingleText.setText(text);
        inputSingleText.setEnable(enable);

        return inputSingleText;
    }

    public static InputSingleText create(String title, String text, String locale, String schema, List<UiBotRule> ruleList, Boolean enable) {
        InputSingleText inputSingleText = new InputSingleText();
        inputSingleText.setTitle(title);
        inputSingleText.setText(text);
        inputSingleText.setSchema(schema);
        inputSingleText.setEnable(enable == null ? false : enable);
        inputSingleText.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        if (ruleList != null && ruleList.size() > 0) {
            ruleList.stream().filter(uiBotRule -> schema.equalsIgnoreCase(uiBotRule.getSchema()))
                    .forEach(uiBotRule -> {
                        if (ValidateRuleTypeEnum.REQUIRED.getType().equalsIgnoreCase(uiBotRule.getKey())) {
                            inputSingleText.setRequired("required".equalsIgnoreCase(uiBotRule.getKey()));
                        }
                    });
        }

        return inputSingleText;
    }

    public static InputSingleText create(UiBotLayout uiBotLayout, String locale, String text) {
        InputSingleText inputSingleText = new InputSingleText();
        inputSingleText.setSchema(uiBotLayout.getSchema());
        inputSingleText.setTitle(uiBotLayout.getmLabelText().get(locale));
        if (StringUtils.hasLength(text)) {
            inputSingleText.setText(text);
        }
        inputSingleText.setEnable(true);
        inputSingleText.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        inputSingleText.setRequired(true);
        return inputSingleText;
    }

    public static InputSingleText create(UiBotLayout uiBotLayout, String locale, String text, Integer inputType) {
        InputSingleText inputSingleText = new InputSingleText();
        inputSingleText.setSchema(uiBotLayout.getSchema());
        inputSingleText.setTitle(uiBotLayout.getmLabelText().get(locale));
        if (StringUtils.hasLength(text)) {
            inputSingleText.setText(text);
        }
        inputSingleText.setInputType(inputType);
        inputSingleText.setEnable(true);
        inputSingleText.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        inputSingleText.setRequired(true);
        return inputSingleText;
    }

    public static InputSingleText create(String locale, UiBotTableColumn column, List<UiBotRule> columnRuleList,
                                         Map<String, Object> bizData, List<Rule> mobileRuleList) {
        InputSingleText inputSingleText = new InputSingleText();
        inputSingleText.setTitle(column.getHeaderName());
        inputSingleText.setText(bizData.get(column.getSchema()) == null ? "" : String.valueOf(bizData.get(column.getSchema())));
        inputSingleText.setSchema(column.getSchema());
        inputSingleText.setEnable(true);
        inputSingleText.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        Optional<Rule> optRequiredRule = mobileRuleList.stream()
                .filter(r ->
                        RuleCategoryEnum.VALIDATE.getCategory().equalsIgnoreCase(r.getCategory())
                                && ValidateRuleTypeEnum.REQUIRED.getType().equalsIgnoreCase(r.getType())
                ).findFirst();
        if (optRequiredRule.isPresent()) {
            inputSingleText.setRequired(true);
        } else {
            inputSingleText.setRequired(false);
        }

        return inputSingleText;
    }

    public static InputSingleText create(UiBotLayout uiBotFieldLayout, UiBotRule uiBotRule, String locale) {
        InputSingleText inputSingleText = new InputSingleText();
        if (uiBotFieldLayout != null) {
            inputSingleText.setTitle(uiBotFieldLayout.getHeaderName());
            inputSingleText.setSchema(uiBotFieldLayout.getSchema());
            inputSingleText.setEnable(true);
        }
        inputSingleText.setRequired(false);
        if (uiBotRule != null) {
            inputSingleText.setRequired("required".equalsIgnoreCase(uiBotRule.getKey()));
        }
        inputSingleText.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        return inputSingleText;
    }

    public static InputSingleText create(UiBotLayout uiBotFieldLayout, List<UiBotRule> ruleList, String locale) {
        InputSingleText inputSingleText = new InputSingleText();
        if (uiBotFieldLayout != null) {
            inputSingleText.setTitle(uiBotFieldLayout.getHeaderName());
            inputSingleText.setSchema(uiBotFieldLayout.getSchema());
            inputSingleText.setEnable(true);
        }
        inputSingleText.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        inputSingleText.setRequired(false);
        List<ValidateRule> validateRuleList = new ArrayList<>();
        if (ruleList != null && ruleList.size() > 0) {
            ruleList.stream().filter(uiBotRule -> uiBotFieldLayout.getSchema().equalsIgnoreCase(uiBotRule.getSchema()))
                    .forEach(uiBotRule -> {
                        if (ValidateRuleTypeEnum.REQUIRED.getType().equalsIgnoreCase(uiBotRule.getKey())) {
                            inputSingleText.setRequired("required".equalsIgnoreCase(uiBotRule.getKey()));
                        } else if (ValidateRuleTypeEnum.PATTERN.getType().equalsIgnoreCase(uiBotRule.getKey())) {
                            ValidateRule validateRule = new ValidateRule();
                            validateRule.setSchema(uiBotFieldLayout.getSchema());
                            validateRule.setParameter(uiBotRule.getParameter());
                            validateRule.setType(ValidateRuleTypeEnum.PATTERN.getType());
                            validateRule.setScope(ValidateRuleScopeEnum.EDIT.getScope());
                            validateRuleList.add(validateRule);
                            inputSingleText.setErrorMessage(uiBotRule.getErrorMessage());
                        }
                    });
        }

        inputSingleText.setValidateRuleList(validateRuleList);
        return inputSingleText;
    }

    public static InputSingleText create(UiBotLayout uiBotLayout, String text, Map<String, Object> data, String locale, Boolean isComplete) {
        InputSingleText inputSingleText = new InputSingleText();
        inputSingleText.setTitle(uiBotLayout.getmLabelText().get(locale));
        inputSingleText.setText(text);
        inputSingleText.setEnable(!isComplete);
        inputSingleText.setRequired(false);
        if ("actualMinutes".equals(uiBotLayout.getSchema()) || "minutes".equals(uiBotLayout.getSchema())) {
            inputSingleText.setErrorMessage(LocaleUtil.getMobileTextByKey(locale, "actualMinutesErrors"));
            List<ValidateRule> validateRules = new ArrayList<>(1);
            ValidateRule validateRule = new ValidateRule();
            validateRule.setSchema(uiBotLayout.getSchema());
            validateRule.setParameter("^([0-5]?[0-9]|59)$");
            validateRule.setScope("EDIT");
            validateRule.setType("pattern");
            validateRules.add(validateRule);
            inputSingleText.setValidateRuleList(validateRules);
        } else if ("actualHours".equals(uiBotLayout.getSchema()) || "hours".equals(uiBotLayout.getSchema())) {
            inputSingleText.setErrorMessage(LocaleUtil.getMobileTextByKey(locale, "actualHoursErrors"));
            List<ValidateRule> validateRules = new ArrayList<>(1);
            ValidateRule validateRule = new ValidateRule();
            validateRule.setSchema(uiBotLayout.getSchema());
            validateRule.setParameter("^[0-9]*$");
            validateRule.setScope("EDIT");
            validateRule.setType("pattern");
            validateRules.add(validateRule);
            inputSingleText.setValidateRuleList(validateRules);
        } else if ("taskName".equals(uiBotLayout.getSchema())) {
            inputSingleText.setRequired(true);
        }
        inputSingleText.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        inputSingleText.setSchema(uiBotLayout.getSchema());
        return inputSingleText;
    }

    public static InputSingleText create(String schema, String title, Boolean required,
                                         Boolean enable, String locale, String data, Boolean hidden) {
        InputSingleText inputSingleText = new InputSingleText();
        inputSingleText.setText(data);
        inputSingleText.setSchema(schema);
        inputSingleText.setTitle(title);
        inputSingleText.setEnable(enable);
        inputSingleText.setPlaceholder(LocaleUtil.getMobileTextByKey(locale, "请输入"));
        inputSingleText.setRequired(required);
        inputSingleText.setHidden(hidden);
        return inputSingleText;
    }

    @Override
    public String returnComponentType() {
        return InputSingleText.COMPONENT_TYPE;
    }

    @Override
    public void handleComponentParam(ComponentContext cmptContext, String mobilePath, String schema, Map<String, Object> data,
                                     PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        super.handleComponentParam(cmptContext, mobilePath, schema, data, executeContext, mobilePageRawData);
        String value = Optional.ofNullable(UiBotDesignerRenderService.getDataByPathAndSchema(data, mobilePath, schema))
                .map(Object::toString)
                .orElse(null);
        if (StringUtils.hasLength(value)) {
            this.text = value;
        } else {
            if (BooleanUtils.isTrue(this.enable)) {
                // 输入型，暂定无默认值
            } else {
                this.text = "";
            }
        }
        //提示语给默认值
        if (ObjectUtils.isEmpty(this.placeholder)) {
            this.placeholder = localeService.getLanguageValue(executeContext.getLocale(), "请输入");
        }
        if (ObjectUtils.isEmpty(this.actionName)) {
            this.actionName = localeService.getLanguageValue(executeContext.getLocale(), "请选择");
        }
        //构建跳转
        if (ObjectUtils.isEmpty(this.inputButton)) {
            return;
        }
        if(ObjectUtils.isEmpty(this.inputButton.getBtnType())){
            return;
        }
        Button<UiBotAction> button = new Button<>();
        button.setSubmitAction(this.inputButton);
        cmptContext.setCommonRawDataParentType(CommonRawDataParentType.ALONE_BUTTON.getValue());
        button.handleComponentParam(cmptContext, mobilePath, schema, data, executeContext, mobilePageRawData);
        this.inputButton = null;
        this.inputAction = button.getAction();
    }

    public static BaseMobileComponent cardContentConvertComponent(Card.DesignerRawCard.Content rawContent, ComponentContext cmptContext, Map<String, Object> data, PcUiBotExecuteContext executeContext, MobilePageRawData mobilePageRawData) {
        InputSingleText inputSingleText = new InputSingleText();
        inputSingleText.setComponentId(rawContent.getComponentId());
        inputSingleText.setTitle(rawContent.getLabel());
        inputSingleText.setEnable(rawContent.getEditable());
        Card.DesignerRawCard.Content.SingleTextObj singleTextObj = JsonUtil.objectToJavaObject(rawContent.getSetting(), Card.DesignerRawCard.Content.SingleTextObj.class);
        if (singleTextObj != null) {
            inputSingleText.setPlaceholder(singleTextObj.getPlaceholder());
            inputSingleText.setUnit(singleTextObj.getUnit());
        }
        inputSingleText.handleComponentParam(cmptContext, rawContent.getFullPath(), rawContent.getSchema(), data, executeContext, mobilePageRawData);
        return inputSingleText;
    }
}
