package com.digiwin.mobile.mobileuibot.core.component.table;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.mobile.mobileuibot.core.component.action.Action;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotExecuteContext;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.table.AgileDataTableColumn;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.table.TableColumnDefinition;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumn;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * <p>功能描述：表格下钻配置</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: DrillDownConfig
 * @Author: zaregoto
 * @Date: 2023/11/21 14:29
 */
@Data
@NoArgsConstructor
@AllArgsConstructor
public class TableColumnConfig implements Serializable {

    private static final long serialVersionUID = -783269710476001751L;

    /**
     * 是否可以下钻
     */
    private Boolean canDrillDown;

    /**
     * 单元格列ID
     */
    private String columnId;

    /**
     * 下钻的事件
     */
    private Action action;

    /**
     * 单元格字体的颜色,默认为黑色
     */
    private String fontColor = TableColumnColorEnum.NORMAL.getColor();

    /**
     * 复合列的子列信息
     */
    private List<TableCompoundColumn> compoundCols;

    public void clearFontColor() {
        this.fontColor = null;
    }

    public static TableColumnConfig create(AgileDataTableColumn column, UiBotExecuteContext executeContext) {
        TableColumnConfig tableColumnConfig = new TableColumnConfig();
        tableColumnConfig.buildFontColor(tableColumnConfig, column.getOperations());
        tableColumnConfig.setColumnId(column.getSchema());
        tableColumnConfig.setAction(Action.createDrilDown(column, executeContext));
        return tableColumnConfig;
    }

    /**
     * 创建表格行中每个单元格的配置
     *
     * @param column
     * @param executeContext
     * @return
     */
    public static TableColumnConfig create(UiBotTableColumn column, UiBotExecuteContext executeContext) {
        TableColumnConfig tableColumnConfig = new TableColumnConfig();
        tableColumnConfig.setColumnId(column.getSchema());
        tableColumnConfig.buildFontColor(tableColumnConfig, column.getOperations());
        tableColumnConfig.setAction(Action.createDrilDown(column, executeContext));
        return tableColumnConfig;
    }

    /**
     * 创建一个用复合列（组合列）的单元格配置
     *
     * @param tableColumnDefinition 复合列定义
     * @param compoundCols          复合列的子列
     * @param tableData             表格业务数据
     * @param executeContext        数据上下文
     * @return
     */
    public static TableColumnConfig create(TableColumnDefinition tableColumnDefinition, List<TableCompoundColumn> compoundCols,
                                           List<JSONObject> tableData, UiBotExecuteContext executeContext) {
        TableColumnConfig tableColumnConfig = new TableColumnConfig();
        tableColumnConfig.setColumnId(tableColumnDefinition.getSchema());
        tableColumnConfig.setCompoundCols(compoundCols);
        // 复合列未设计下钻，先不做
//        tableColumnConfig.setAction(Action.createDrilDown(column, executeContext));
        return tableColumnConfig;
    }

    public void buildFontColor(TableColumnConfig tableColumnConfig, JSONArray operations) {
        if (AgileDataTableColumn.canDrillDown(operations)) {
            tableColumnConfig.setCanDrillDown(true);
            /**
             * 下钻的颜色
             */
            tableColumnConfig.setFontColor(TableColumnColorEnum.CAN_DRILL_DOWN.getColor());
        } else if (AgileDataTableColumn.isHighlight(operations)) {
            /**
             * 强调的颜色
             */
            tableColumnConfig.setFontColor(TableColumnColorEnum.HIGHLIGHT.getColor());
        } else {
            tableColumnConfig.setCanDrillDown(false);
        }
    }

    /**
     * 获取移动端表格行的每列单元格配置
     *
     * @param columnDefinitionList 中间态DSL定义的表格数据列清单
     * @param decomposedColumnList 已全部拆分的表格列清单
     * @param executeContext       数据上下文
     * @return
     */
    public static List<TableColumnConfig> getTableColumnConfigs(
            List<TableColumnDefinition> columnDefinitionList,
            List<UiBotTableColumn> decomposedColumnList,
            List<JSONObject> tableData,
            UiBotExecuteContext executeContext) {
        List<TableColumnConfig> columnConfigs = new ArrayList<>(columnDefinitionList.size());
        for (TableColumnDefinition tableColumnDefinition : columnDefinitionList) {
            TableColumnConfig tableColumnConfig;
            if (tableColumnDefinition.preferCompoundMode()) {
                tableColumnConfig = TableColumnConfig.create(tableColumnDefinition,
                        TableCompoundColumn.getCompoundColumns(tableColumnDefinition), tableData, executeContext);
            } else {
                UiBotTableColumn tableColumn = tableColumnDefinition.getColumns().get(0);
                tableColumnConfig = TableColumnConfig.create(tableColumn, executeContext);
            }
            columnConfigs.add(tableColumnConfig);
        }
        return columnConfigs;
    }
}
