package com.digiwin.mobile.mobileuibot.core.component.table;

import com.alibaba.fastjson.annotation.JSONField;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.table.TableColumnDefinition;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumn;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableDecimalRule;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableUnit;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.Accessors;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author wuyang
 */
@Data
@AllArgsConstructor
@NoArgsConstructor
@Accessors(chain = true)
public class TableHeaderDataSet implements Serializable {

    private static final long serialVersionUID = -8728448922722288993L;

    /**
     * 列的字段id
     */
    private String id;

    /**
     * 列的名称
     */
    private String content;

    /**
     * 列的数据类型
     */
    private String dataType;

    /**
     * 列是否支持排序
     */
    private Boolean canSort;

    /**
     * 列是否支持筛选
     */
    private Boolean canFilter;

    /**
     * 列的筛选样式
     *
     * @see TableHeaderFilterStyleEnum
     */
    private String filterStyle;
    /**
     * 列是否为复合列（组合列）
     */
    private Boolean compound = false;

    /**
     * 列内容水平对齐方式（默认左对齐）
     *
     * @see TableHeaderHAlignTypeEnum
     */
    @JsonProperty(value = "hAlignType")
    @JSONField(name = "hAlignType")
    private String hAlignType = TableHeaderHAlignTypeEnum.LEFT.getValue();

    /**
     * 列栏位宽度
     */
    private Integer width;

    /**
     * 指标单位
     */
    private String businessType;

    /**
     * 数据计算逻辑说明
     */
    private String description;

    /**
     * 舍入方式
     */
    private UiBotTableDecimalRule decimalRule;

    /**
     * 计算单位
     */
    private UiBotTableUnit unit;

    /**
     * 统计指标是否需要使用百分号，或千分位显示
     * 当使用百分号时，值为percent；
     * 当使用千分位时，值为thousand
     */
    private String percent;

    /**
     * 统计指标的显示精度
     */
    private Integer decimal;

    /**
     * 强制设置当前列为复合列，会同时将过滤关闭、排序关闭、对齐方式设为null
     */
    public void forceSetCompoundMode() {
        this.compound = true;
        this.canFilter = false;
        this.canSort = false;
        this.hAlignType = null;
    }

    /**
     * 生成不带组合列的表格表头，并判断列是否支持排序和筛选
     *
     * @param uiBotTableColumns
     * @param needFilterAndSort
     * @return
     */
    public static List<TableHeaderDataSet> getTableHeaderDataSetList(List<UiBotTableColumn> uiBotTableColumns,
                                                                     boolean needFilterAndSort) {
        List<TableHeaderDataSet> tableHeaderDataSetList = new ArrayList<>(uiBotTableColumns.size());

        for (int i = 0; i < uiBotTableColumns.size(); i++) {
            UiBotTableColumn column = uiBotTableColumns.get(i);
            String columnDataType = Optional.ofNullable(column.getDataType()).orElse("string");
            TableHeaderDataSet tableHeaderData = new TableHeaderDataSet()
                    .setId(column.getSchema())
                    .setContent(column.getHeaderName())
                    // 如果发现pc端是用百分比呈现，那默认将该列数据类型设置为数值型
                    .setDataType(column.isRenderInPercentType() ? "numeric" : column.getDataType())
                    .setCanSort(needFilterAndSort ? Optional.ofNullable(column.getSortable()).orElse(false) : false)
                    .setCanFilter(needFilterAndSort ? Optional.ofNullable(column.getFilterable()).orElse(false) : false)
                    .setFilterStyle(TableHeaderFilterStyleEnum.FILTER_WITH_ENTRY_AND_SELECT.getValue())
                    .setUnit(column.getUnit())
                    .setDecimalRule(column.getDecimalRule())
                    .setPercent(column.getPercent())
                    .setDecimal(-1 == column.getDecimal() ? null : column.getDecimal());
            if (i != 0) {
                if (column.isNumericData()) {
                    tableHeaderData.setHAlignType(TableHeaderHAlignTypeEnum.RIGHT.getValue());
                }
            }
            tableHeaderDataSetList.add(tableHeaderData);
        }

        return tableHeaderDataSetList;
    }

    /**
     * 生成带组合列的表格表头，并判断列是否支持排序和筛选
     *
     * @param tableColumnDefinitionList 中间态DSL所有表格列的定义
     * @param needFilterAndSort         是否需要过滤和排序
     * @return
     */
    public static List<TableHeaderDataSet> getCompoundTableHeaderDataSetList(List<TableColumnDefinition> tableColumnDefinitionList,
                                                                             boolean needFilterAndSort) {
        List<TableHeaderDataSet> tableHeaderDataSetList = new ArrayList<>(tableColumnDefinitionList.size());
        for (int i = 0; i < tableColumnDefinitionList.size(); i++) {
            TableColumnDefinition columnDefinition = tableColumnDefinitionList.get(i);
            TableHeaderDataSet tableHeaderData = getCompoundTableHeaderDataSet(columnDefinition, i == 0, needFilterAndSort);
            tableHeaderDataSetList.add(tableHeaderData);
        }

        return tableHeaderDataSetList;
    }

    /**
     * 根据中间态DSL中表格列的定义，生成带组合列的表格表头
     *
     * @param columnDefinition        中间态DSL的某个表格列定义
     * @param isFirstColumnDefinition 是否是第一列，只影响非复合列。若是第一列，永远左对齐；否则需依列数据类型判断：如果是数值型则右对齐，否则默认左对齐
     * @param needFilterAndSort       是否需要过滤和排序
     * @return
     */
    private static TableHeaderDataSet getCompoundTableHeaderDataSet(TableColumnDefinition columnDefinition,
                                                                    boolean isFirstColumnDefinition,
                                                                    boolean needFilterAndSort) {
        List<UiBotTableColumn> columns = columnDefinition.getColumns();

        TableHeaderDataSet tableHeaderData = new TableHeaderDataSet();
        // 复合列（组合列）的情况下，统一设置为字符串，默认左对齐，无排序，无过滤
        if (columnDefinition.preferCompoundMode()) {
            tableHeaderData
                    .setId(columnDefinition.getSchema())
                    .setDataType("string")
                    .setContent(columnDefinition.getHeaderName())
                    .setCanSort(false)
                    .setCanFilter(false);
            tableHeaderData.forceSetCompoundMode();
        } else {
            UiBotTableColumn singleColumn = columns.get(0);
            tableHeaderData
                    .setId(singleColumn.getSchema())
                    .setContent(singleColumn.getHeaderName())
                    // 如果发现pc端是用百分比呈现，那默认将该列数据类型设置为数值型
                    .setDataType(singleColumn.isRenderInPercentType() ? "numeric" : singleColumn.getDataType())
                    .setCanSort(needFilterAndSort ? Optional.ofNullable(singleColumn.getSortable()).orElse(false) : false)
                    .setCanFilter(needFilterAndSort ? Optional.ofNullable(singleColumn.getFilterable()).orElse(false) : false)
                    .setFilterStyle(TableHeaderFilterStyleEnum.FILTER_WITH_ENTRY_AND_SELECT.getValue())
                    .setBusinessType(singleColumn.getBusinessType())
                    .setDescription(columnDefinition.getDescription())
                    .setDecimalRule(singleColumn.getDecimalRule())
                    .setUnit(singleColumn.getUnit())
                    .setPercent(singleColumn.getPercent())
                    .setDecimal(-1 == singleColumn.getDecimal() ? null : singleColumn.getDecimal());
        }
        // 非复合列（组合列），单独设置对齐方式
        if (!tableHeaderData.getCompound() && !isFirstColumnDefinition) {
            if (Objects.equals("numeric", tableHeaderData.getDataType().toLowerCase())) {
                tableHeaderData.setHAlignType(TableHeaderHAlignTypeEnum.RIGHT.getValue());
            }
        }

        return tableHeaderData;
    }
}