package com.digiwin.mobile.mobileuibot.core.layout.analyzer;

import com.digiwin.mobile.mobileuibot.core.component.MobileLayoutTypeEnum;
import com.digiwin.mobile.mobileuibot.core.layout.LayoutGeneratorPreprocessData;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * <p>功能描述：基于业务数据的布局分析器</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: BizDataLayoutAnalyzer
 * @Author: zaregoto
 * @Date: 2023/3/15 14:48
 */
@Component("bizDataLayoutAnalyzer")
public class BizDataLayoutAnalyzer implements LayoutAnalyzer<Object> {

    @Override
    public String getType() {
        return LayoutAnalyzerTypeEnum.BIZ_DATA_TYPE.toString();
    }

    @Override
    public MobileLayoutTypeEnum getLayoutType(LayoutGeneratorPreprocessData preprocessData) {
        Map<String, Object> bizDataMap = preprocessData.getBizDataMap();
        Map<String, Object> bizMetadataMap = preprocessData.getBizMetadataMap();

        if (CollectionUtils.isEmpty(bizDataMap)) {
            // 无任何业务数据，默认使用多层布局
            return MobileLayoutTypeEnum.MULTI_PATTERN_LAYOUT;
        }

        // 只有一组业务数据时
        if (bizDataMap.size() == 1) {
            String bizDataKey = bizDataMap.keySet().iterator().next();
            Object bizData = bizDataMap.getOrDefault(bizDataKey, Collections.emptyMap());
            return this.getLayoutTypeBySingleBizData(bizData, bizMetadataMap);
        }

        /**
         * 有多组业务数据时，对每组业务数据进行布局类型的判断。
         * 只要再出过一次多层布局，则最终使用多层布局；否则使用单层布局
         */
        boolean hasMultiLevelLayout = false;
        for (Map.Entry<String, Object> bizDataKey : bizDataMap.entrySet()) {
            Object bizData = bizDataMap.getOrDefault(bizDataKey, Collections.emptyMap());
            if (bizData instanceof Map) {
                MobileLayoutTypeEnum mobileLayoutTypeEnum =
                        this.getLayoutTypeBySingleBizData(bizData, bizMetadataMap);
                if (Objects.equals(mobileLayoutTypeEnum, MobileLayoutTypeEnum.MULTI_PATTERN_LAYOUT)) {
                    hasMultiLevelLayout = true;
                    break;
                }
            } else {
                // 处理 bizDataObj 不是 Map 的情况，例如抛出异常或返回默认值
                throw new IllegalArgumentException("Expected a Map but found: " + bizData.getClass().getName());
            }
        }

        return hasMultiLevelLayout ?
                MobileLayoutTypeEnum.MULTI_PATTERN_LAYOUT : MobileLayoutTypeEnum.SINGLE_PATTERN_LAYOUT;
    }

    /**
     * 根据实际业务数据，以及业务数据的元数据判断合适的布局类型
     * v2.5.0: 仅使用业务数据，暂未使用元数据
     *
     * @param bizData
     * @param bizMetadata
     * @return
     */
    private MobileLayoutTypeEnum getLayoutTypeBySingleBizData(Object bizData, Map<String, Object> bizMetadata) {
        // 数据为null，默认使用多层布局
        if (Objects.isNull(bizData)) {
            return MobileLayoutTypeEnum.MULTI_PATTERN_LAYOUT;
        }
        if (bizData instanceof List) {
            List listData = (List) bizData;
            if (listData.size() <= 1) {
                return MobileLayoutTypeEnum.SINGLE_PATTERN_LAYOUT;
            } else {
                return MobileLayoutTypeEnum.MULTI_PATTERN_LAYOUT;
            }
        }
        if (bizData instanceof Map) {
            return MobileLayoutTypeEnum.SINGLE_PATTERN_LAYOUT;
        }
        // 未知数据类型，默认使用多层布局
        return MobileLayoutTypeEnum.MULTI_PATTERN_LAYOUT;
    }
}
