package com.digiwin.mobile.mobileuibot.core.layout.analyzer;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>功能描述：布局分析器权重读取器</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: LayoutAnalyzerWeightReader
 * @Author: zaregoto
 * @Date: 2023/3/21 15:27
 */
@Slf4j
@Component("layoutAnalyzerWeightReader")
public class LayoutAnalyzerWeightReader {

    /**
     * 改用线程安全的类
     */
    private List<LayoutAnalyzerWeight> layoutAnalyzerWeightList = Collections.synchronizedList(new ArrayList<>());

    /**
     * 读取布局分析器的权重数据（默认不使用基于AI的权重分析）
     *
     * @return
     */
    public LayoutAnalyzerWeight getMaxWeight() {
        return this.getMaxWeight(false);
    }

    /**
     * 读取布局分析器的权重数据
     *
     * @param basedOnAugmentedIntelligence
     * @return
     */
    public LayoutAnalyzerWeight getMaxWeight(boolean basedOnAugmentedIntelligence) {
        if (CollectionUtils.isEmpty(layoutAnalyzerWeightList)) {
            synchronized (LayoutAnalyzerWeightReader.class) {
                if (CollectionUtils.isEmpty(layoutAnalyzerWeightList)) {
                    this.layoutAnalyzerWeightList = new ArrayList<>(LayoutAnalyzerTypeEnum.values().length);
                    if (!basedOnAugmentedIntelligence) {
                        this.layoutAnalyzerWeightList.add(
                                new LayoutAnalyzerWeight()
                                        .setLayoutAnalyzerTypeEnum(LayoutAnalyzerTypeEnum.BIZ_DATA_TYPE)
                                        .setWeight(LayoutAnalyzerWeight.MAX_WEIGHT)
                        );
                        this.layoutAnalyzerWeightList.add(
                                new LayoutAnalyzerWeight()
                                        .setLayoutAnalyzerTypeEnum(LayoutAnalyzerTypeEnum.SCENERIO_DATA_TYPE)
                                        .setWeight(LayoutAnalyzerWeight.MIN_WEIGHT)
                        );
                    } else {
                        // TODO 扩展点：集成AI的布局分析结果
                    }
                }
            }
        }
        // 权重值纠正
        this.executeEcc(this.layoutAnalyzerWeightList);

        this.layoutAnalyzerWeightList.sort(Comparator.comparingDouble(LayoutAnalyzerWeight::getWeight));
        return this.layoutAnalyzerWeightList.get(this.layoutAnalyzerWeightList.size() - 1);
    }

    /**
     * 执行权重值纠正，防止出现各个分析器的权重相等，从而无法得到布局分析结果
     * ECC:Error Checking and Correcting，错误检查和纠正，借用RAM中的技术名词
     *
     * @param layoutAnalyzerWeights
     */
    private void executeEcc(List<LayoutAnalyzerWeight> layoutAnalyzerWeights) {
        if (CollectionUtils.isEmpty(layoutAnalyzerWeights)) {
            return;
        }
        Optional<LayoutAnalyzerWeight> optMaxLayoutAnalyzerWeight = layoutAnalyzerWeights
                .stream()
                .max(Comparator.comparingDouble(LayoutAnalyzerWeight::getWeight));
        if (optMaxLayoutAnalyzerWeight.isPresent()) {
            Double maxWeight = optMaxLayoutAnalyzerWeight.get().getWeight();
            List<LayoutAnalyzerWeight> maxWeightAnalyzerList = layoutAnalyzerWeights.stream()
                    .filter(w -> Objects.equals(maxWeight, w.getWeight()))
                    .collect(Collectors.toList());
            // 如果权重最大值大于1项，则需要做规范化处理，使得只有1项是最大值
            // FIXME 暂时做法是提高第1项的权重
            if (maxWeightAnalyzerList.size() > 1) {
                if (maxWeight >= LayoutAnalyzerWeight.MAX_WEIGHT) {
                    for (int i = 0; i < maxWeightAnalyzerList.size(); i++) {
                        LayoutAnalyzerWeight layoutAnalyzerWeight = maxWeightAnalyzerList.get(i);
                        if (i == 0) {
                            layoutAnalyzerWeight.setWeight(LayoutAnalyzerWeight.MAX_WEIGHT);
                        } else {
                            layoutAnalyzerWeight.setWeight(LayoutAnalyzerWeight.MAX_WEIGHT - LayoutAnalyzerWeight.WEIGHT_STEP);
                        }
                    }
                } else {
                    LayoutAnalyzerWeight layoutAnalyzerWeight = maxWeightAnalyzerList.get(0);
                    Double weight = layoutAnalyzerWeight.getWeight();
                    Double newWeight = weight + LayoutAnalyzerWeight.WEIGHT_STEP;
                    if (newWeight >= LayoutAnalyzerWeight.MAX_WEIGHT) {
                        layoutAnalyzerWeight.setWeight(LayoutAnalyzerWeight.MAX_WEIGHT);
                    } else {
                        layoutAnalyzerWeight.setWeight(newWeight);
                    }
                }
            }
        }
    }
}
