package com.digiwin.mobile.mobileuibot.core.rule;

import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.core.rule.relation.CommonRule;
import com.digiwin.mobile.mobileuibot.core.rule.validate.ValidateRuleTypeEnum;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.springframework.data.mongodb.core.mapping.Document;
import org.springframework.data.mongodb.core.mapping.Field;
import org.springframework.util.CollectionUtils;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>功能描述：数据库中保存的规则类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: Rule
 * @Author: Zaregoto
 * @Date: 2021/6/24 17:49
 */
@Document(collection = "mobileUiBotRule")
@Data
@NoArgsConstructor
@AllArgsConstructor
public class Rule implements Serializable {
    private static final long serialVersionUID = -7984118157325999337L;

    @Field("_id")
    private String id;
    /**
     * 规则主分类
     *
     * @see RuleCategoryEnum
     */
    private String category;

    /**
     * 规则子分类
     *
     * @see RuleSubcategoryEnum
     */
    private String subcategory;

    /**
     * 当前规则作用的数据节点
     */
    private String path;

    /**
     * 当前规则作用的字段
     */
    private String schema;

    private String parameter;

    /**
     * 类型，针对不同category会有不同的值
     * 如果category是校验类型，则type值参考ValidateRuleTypeEnum类
     * 如果category是联动类型，目前暂无支持
     *
     * @see ValidateRuleTypeEnum,
     */
    private String type;
    private String errorMsgPrefix;

    /**
     * 规则运行时使用的条件
     */
    private String condition;

    /**
     * 规则运行时对条件的操作类型
     *
     * @see RuleConditionOperatorTypeEnum
     */
    private Integer conditionOperatorType;

    /**
     * 规则针对的目标字段
     */
    private String targetSchema;

    /**
     * 规则针对的目标字段是否可用
     */
    private String targetEnable;

    /**
     * 规则针对的目标字段是否必须
     */
    private String targetRequired;
    private String targetPlaceholder;
    /**
     * 规则针对的目标字段是否隐藏
     */
    private String targetHidden;

    /**
     * 引用当前规则的范围
     *
     * @see RuleScopeEnum
     */
    private String scope;

    /**
     * 规则是否生效
     */
    private Boolean valid;

    /**
     * 规则是否生效的判断条件集合。
     * 如果该属性为null或为空，则表示规则在任何条件下都生效。
     * 如果该属性不为空，则对该集合内的判断条件取“AND”逻辑运算并返回值。
     */
    private List<RuleValidCondition> validConditions;

    /**
     * 初始化的规则，需要执行的配置
     */
    private RuleInitConfig initConfig;

    /**
     * 规则的备注说明
     */
    private String remark;

    /**
     * mapping用于的组件名称，同组件Class.COMPONENT_TYPE
     */
    private String componentType;

    private RuleMappingConfig mappingConfig;

    /**
     * 通用联动规则
     * 为了可以将通用联动在配置表可配置
     */
    private CommonRule commonRule;

    /**
     * 校验型规则的错误信息配置，支持多语言。仅在category=validate时有用。key与value说明如下：
     * key-语言别。@see java.util.Locale
     * value-对应语言别下的错误信息文字
     */
    private Map<String, String> errorMessage;

    private String targetText;

    /**
     * 与校验参数比较的操作符号。
     * type为appoint时生效。若该字段没有或为空时，前端默认用“2-不等于”来比较。
     */
    private Integer operatorType;

    /**
     * 校验规则的排序。从0开始的自然数,正序
     */
    private Integer order;

    /**
     * 校验参数的数据类型。
     * 指定处理parameter时使用的数据类型。若该字段没有或为空时，前端需要把parameter默认视为string来操作。
     *
     * @see RuleDataTypeEnum
     */
    private String parameterDataType;

    /**
     * 租户
     * 2022年9月2日加这个字段是为了区别客制规则
     */
    private String tenantId;
    /**
     * 防止规则字段重复
     */
    private String pageId;

    /**
     * 根据业务数据的值，计算联动规则中配置的目标字段属性的动态表达式。<br>
     * 目前仅支持<b>一个</b>条件的动态表达式，且仅支持以下目标字段属性的设置：<br>
     * - targetRequired <br>
     * - targetEnable <br>
     * 表达式规范：${[rulePath].[bizDataKey]}。eg. ${project_info.is_attachment}, ${project_info.is_confirm_date}
     * <p>
     * eg. 如在数据库中，给targetRequired配置成这个值：${project_info.is_attachment},表示实际targetRequired
     * 的值会根据is_attachment的值而改变。另外前提，is_attachment的数据须是Boolean格式，若不是则默认设置为false
     *
     * @param bizData                  业务数据
     * @param targetPropertyExpression 配置目标字段属性表达式
     * @return true/false
     */
    public boolean calculateTargetPropertyResult(Map<String, Object> bizData, final String targetPropertyExpression) {
        boolean result = false;
        if (null == targetPropertyExpression || targetPropertyExpression.isEmpty()) {
            return result;
        }
        if (!RuleUtil.validExpression(targetPropertyExpression)) {
            if (!RuleUtil.validBooleanValue(targetPropertyExpression)) {
                return false;
            } else {
                return Boolean.parseBoolean(targetPropertyExpression);
            }
        } else {
            String expressionBizDataKey = RuleUtil.analyzeExpressionBizDataKey(targetPropertyExpression);
            Object bizDataValue = bizData.get(expressionBizDataKey);
            if (bizDataValue instanceof Boolean) {
                result = (boolean) bizDataValue;
            }
        }

        return result;
    }

    /**
     * 判断当前规则对象在结合业务数据后，是否有效。判断条件是：
     * 1.rule.valid==true
     * 2.rule.validConditions不为空&&它的所有condition==true
     *
     * @param bizData 业务数据
     * @return true-有效，false-无效
     */
    public boolean validWithBizData(Map<String, Object> bizData) {
        boolean result;
        if (!this.getValid()) {
            result = false;
        }
        if (CollectionUtils.isEmpty(this.getValidConditions())) {
            result = true;
        } else {
            result = this.getValidConditions().stream()
                    .allMatch(validCondition -> validCondition.conditionIsTrue(bizData));
        }

        return result;
    }

    //判断是否走客制规则
    public static boolean hasCustomMadeRule(ApiRequest apiRequest, List<Rule> ruleList) {
        //过滤出客制的规则
        List<Rule> customMadeRules = ruleList.stream().filter(
                r -> null != r.getTenantId()
        ).collect(Collectors.toList());
        //判断是否有客制规则
        if (CollectionUtils.isEmpty(customMadeRules)) {
            return false;
        } else {
            return true;
        }

    }

    //比较规则，将冲突规则由客制规则替代。不冲突的则保持。ruleType值有INIT、validate、relation
    public static List<Rule> compareRule(ApiRequest apiRequest, List<Rule> ruleList, String ruleType) {
        List<Rule> compareRule = new ArrayList<>();
        //过滤出当前租户客制的规则
        List<Rule> customMadeRules = customMadeRules(apiRequest.getTenantId(), ruleList);
        //根据不同规则分别进行交叉对比
        switch (ruleType) {
            case "INIT":
                compareRule = compareInitRule(ruleList, customMadeRules, apiRequest.getTenantId());
                break;
            case "validate":
                compareRule = compareValidateRule(ruleList, customMadeRules, apiRequest.getTenantId());
                break;
            case "relation":
                compareRule = compareRelationRule(ruleList, customMadeRules, apiRequest.getTenantId());
                break;
            default:
                compareRule = ruleList;
                break;
        }

        return compareRule;

    }

    public static List<Rule> compareInitRule(List<Rule> ruleList, List<Rule> customMadeRules, String tenantId) {
        //只获取公共部分的规则
        List<Rule> compareInitRule = ruleList.stream().filter(
                r -> null == r.getTenantId()
        ).collect(Collectors.toList());
        //公共部分+当前租户的规则
        compareInitRule.addAll(customMadeRules);
        for (Rule rule : ruleList) {
            //当客制规则有的时候，则去掉冲突的，增加定制的
            if (customMadeRules.size() > 0) {
                RuleInitConfig initConfig = rule.getInitConfig();
                //过滤出当前租户规则与公共部分规则冲突的规则
                Rule rule1 = customMadeRules.stream().filter(
                        r1 -> initConfig.getComponentType().equalsIgnoreCase(r1.getInitConfig().getComponentType())
                                && initConfig.getDataType().equalsIgnoreCase(r1.getInitConfig().getDataType())
                                && initConfig.getEnable() == r1.getInitConfig().getEnable()
                ).findFirst().orElse(null);
                if (null != rule1) {
                    //去掉冲突部分，保留
                    compareInitRule = compareInitRule.stream().filter(
                            r2 -> !(rule1.getInitConfig().getComponentType().equalsIgnoreCase(r2.getInitConfig().getComponentType())
                                    && rule1.getInitConfig().getDataType().equalsIgnoreCase(r2.getInitConfig().getDataType())
                                    && rule1.getInitConfig().getEnable() == r2.getInitConfig().getEnable()
                                    && !tenantId.equalsIgnoreCase(r2.getTenantId()))
                    ).collect(Collectors.toList());
                }
            }
        }
        return compareInitRule;
    }

    public static List<Rule> compareValidateRule(List<Rule> ruleList, List<Rule> customMadeRules, String tenantId) {
        //只获取公共部分的规则
        List<Rule> compareValidateRule = ruleList.stream().filter(
                r -> null == r.getTenantId()
        ).collect(Collectors.toList());
        //公共部分+当前租户的规则
        compareValidateRule.addAll(customMadeRules);
        for (Rule rule : ruleList) {
            //当客制规则有的时候，则去掉冲突的，增加定制的
            if (customMadeRules.size() > 0) {
                switch (ValidateRuleTypeEnum.getEnumByValue(rule.getType())) {
                    case PATTERN:
                        //TODO 待以后处理
                        break;
                    case REQUIRED:
                        //TODO 待以后处理
                        break;
                    case TIMESPACE:
                        //TODO 待以后处理
                        break;
                    case APPOINT:
                        //过滤出当前租户规则与公共部分规则冲突的规则
                        Rule ruleAppoint = customMadeRules.stream().filter(
                                r1 -> rule.getPath().equalsIgnoreCase(r1.getPath())
                                        && rule.getSchema().equalsIgnoreCase(r1.getSchema())
                                        && rule.getParameter().equalsIgnoreCase(r1.getParameter())
                                        && rule.getOrder() == r1.getOrder()
                        ).findFirst().orElse(null);
                        if (null != ruleAppoint) {
                            //去掉冲突部分，保留
                            compareValidateRule = compareValidateRule.stream().filter(
                                    r2 -> !(ruleAppoint.getPath().equalsIgnoreCase(r2.getPath())
                                            && ruleAppoint.getSchema().equalsIgnoreCase(r2.getSchema())
                                            && ruleAppoint.getParameter().equalsIgnoreCase(r2.getParameter())
                                            && ruleAppoint.getOrder() == r2.getOrder()
                                            && !tenantId.equalsIgnoreCase(r2.getTenantId()))
                            ).collect(Collectors.toList());
                        }
                        break;
                    default:
                        break;
                }
            }
        }
        return compareValidateRule;
    }

    public static List<Rule> compareRelationRule(List<Rule> ruleList, List<Rule> customMadeRules, String tenantId) {
        //只获取公共部分的规则
        List<Rule> compareRelationRule = ruleList.stream().filter(
                r -> null == r.getTenantId()
        ).collect(Collectors.toList());
        //公共部分+当前租户的规则
        compareRelationRule.addAll(customMadeRules);
        for (Rule rule : ruleList) {
            switch (RuleSubcategoryEnum.getEnumByValue(rule.getSubcategory())) {
                case DATE:
                    //TODO 待以后处理
                    break;
                case TEXT:
                    //目前用path、schema、condition、conditionOperatorType、targetSchema确定唯一
                    //过滤出当前租户规则与公共部分规则冲突的规则
                    Rule rule1 = customMadeRules.stream().filter(
                            r1 -> rule.getPath().equalsIgnoreCase(r1.getPath())
                                    && rule.getSchema().equalsIgnoreCase(r1.getSchema())
                                    && rule.getCondition().equalsIgnoreCase(r1.getCondition())
                                    && rule.getConditionOperatorType() == r1.getConditionOperatorType()
                                    && rule.getTargetSchema().equalsIgnoreCase(r1.getTargetSchema())
                    ).findFirst().orElse(null);
                    if (null != rule1) {
                        //去掉冲突部分，保留
                        compareRelationRule = compareRelationRule.stream().filter(
                                r2 -> !(rule1.getPath().equalsIgnoreCase(r2.getPath())
                                        && rule1.getSchema().equalsIgnoreCase(r2.getSchema())
                                        && rule1.getCondition().equalsIgnoreCase(r2.getCondition())
                                        && rule1.getConditionOperatorType() == r2.getConditionOperatorType()
                                        && rule1.getTargetSchema().equalsIgnoreCase(r2.getTargetSchema())
                                        && !tenantId.equalsIgnoreCase(r2.getTenantId()))
                        ).collect(Collectors.toList());
                    }
                    break;
                case COMMON:
                    Rule ruleCommon = customMadeRules.stream().filter(
                            r1 -> rule.getPath().equalsIgnoreCase(r1.getPath())
                                    && rule.getSchema().equalsIgnoreCase(r1.getSchema())
                                    && rule.getCommonRule() != null
                                    && r1.getCommonRule() != null
                                    && rule.getCommonRule().getCondition() == r1.getCommonRule().getCondition()
                                    && rule.getCommonRule().getTargetSchema().equalsIgnoreCase(r1.getCommonRule().getTargetSchema())
                    ).findFirst().orElse(null);
                    if (null != ruleCommon) {
                        //去掉冲突部分，保留
                        compareRelationRule = compareRelationRule.stream().filter(
                                r2 -> !(ruleCommon.getPath().equalsIgnoreCase(r2.getPath())
                                        && ruleCommon.getSchema().equalsIgnoreCase(r2.getSchema())
                                        && ruleCommon.getCommonRule().getCondition() == r2.getCommonRule().getCondition()
                                        && ruleCommon.getCommonRule().getTargetSchema().equalsIgnoreCase(r2.getCommonRule().getTargetSchema())
                                        && !tenantId.equalsIgnoreCase(r2.getTenantId()))
                        ).collect(Collectors.toList());
                    }
                    break;
                default:
                    break;
            }
        }
        return compareRelationRule;
    }

    //过滤出客制的规则
    public static List<Rule> customMadeRules(String pcTenantId, List<Rule> ruleList) {
        List<Rule> customMadeRules = ruleList.stream().filter(
                r -> pcTenantId.equalsIgnoreCase(r.getTenantId())
        ).collect(Collectors.toList());
        return customMadeRules;
    }

    public String errorMessageInfo(String locale) {
        String errorMessage = this.errorMessage.get(locale);
        return errorMessage;
    }
}
