package com.digiwin.mobile.mobileuibot.core.rule.validate;

import com.digiwin.mobile.mobileuibot.common.math.MathUtil;
import com.digiwin.mobile.mobileuibot.core.rule.Rule;
import com.digiwin.mobile.mobileuibot.core.rule.RuleCategoryEnum;
import com.digiwin.mobile.mobileuibot.core.rule.RuleUtil;
import com.digiwin.mobile.mobileuibot.core.rule.ValidateRuleLevelEnum;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.springframework.util.Assert;

import java.io.Serializable;
import java.util.Map;

/**
 * <p>功能描述：渲染时使用的校验规则</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: ValidateRule
 * @Author: Zaregoto
 * @Date: 2021/6/16 14:47
 */

@Data
@NoArgsConstructor
@AllArgsConstructor
public class ValidateRule implements Serializable {
    private static final long serialVersionUID = 5886119652387271420L;

    /**
     * 校验参数。type为pattern时，参数为正则表达式；type为appoint时，参数为指定值
     */
    private String parameter;
    /**
     * 被校验的字段
     */
    private String schema;
    /**
     * 校验范围
     *
     * @see ValidateRuleScopeEnum
     */
    private String scope;
    /**
     * 校验类型
     *
     * @see ValidateRuleTypeEnum
     */
    private String type;

    /**
     * 校验错误信息
     */
    private String errorMessage;

    /**
     * 与校验参数比较的操作符号。
     * type为appoint时生效。若该字段没有或为空时，前端默认用“2-不等于”来比较。
     */
    private Integer operatorType;

    /**
     * 校验规则的排序。从0开始的自然数,正序
     */
    private Integer order;

    /**
     * 校验参数的数据类型。
     * 指定处理parameter时使用的数据类型。若该字段没有或为空时，前端需要把parameter默认视为string来操作。
     *
     * @see com.digiwin.mobile.mobileuibot.core.rule.RuleDataTypeEnum
     */
    private String parameterDataType;

    /**
     * 校验日期的值，传年月日
     */
    private Map<String, Object> parameterDate;

    /**
     * 校验警示信息
     */
    private String warningMessage;

    /**
     * 校验级别
     *
     * @see ValidateRuleLevelEnum
     */
    private Integer level;

    public ValidateRule(String schema) {
        this.schema = schema;
    }

    public static ValidateRule create(Rule rule, String locale, Map<String, Object> bizData) {
        Assert.notNull(rule, "rule must not null!");
        Assert.isTrue(rule.getCategory().equalsIgnoreCase(
                RuleCategoryEnum.VALIDATE.getCategory()), "rule must be validate category!");

        ValidateRule validateRule = new ValidateRule();

        if (RuleUtil.validExpression(rule.getParameter())) {
            //例：值格式是"${a.complete_rate}"字符串，则这这层赋值
            String expressionBizDataKey = RuleUtil.analyzeExpressionBizDataKey(rule.getParameter());
            Object bizDataValue = bizData.get(expressionBizDataKey);
            if ("complete_rate".equalsIgnoreCase(expressionBizDataKey)) {
                //页面上获取的完成率是小于1的。需要转换成百分制。故这边过滤重新赋值
                Double completeRateValue = null != bizDataValue ? Double.parseDouble(String.valueOf(bizDataValue)) : 0.0;
                //double计算失去精度。
//                completeRateValue = MathUtil.getDecimalPoint(completeRateValue * 100.0, 2, true);
                validateRule.setParameter(String.valueOf(MathUtil.doubleMathCalculation(completeRateValue, 100, "*")));
            } else {
                validateRule.setParameter(String.valueOf(bizDataValue));
            }
        } else {
            validateRule.setParameter(rule.getParameter());
        }
        validateRule.setSchema(rule.getSchema());
        validateRule.setScope(rule.getScope());
        validateRule.setType(rule.getType());
        //当配置文件中配置了operatorType值,则走配置文件进行赋值。否则走原来逻辑，避免以前配置进来报错
        if (null != rule.getOperatorType()) {
            validateRule.setOperatorType(rule.getOperatorType());
        }
        //当配置文件中配置了parameterDataType值,则走配置文件进行赋值。否则走原来逻辑，避免以前配置进来报错
        if (null != rule.getParameterDataType()) {
            validateRule.setParameterDataType(rule.getParameterDataType());
        }
        //当配置文件中配置了errorMessage值，则需要把locale传入，读取对应语言别的错误信息
        if (null != rule.getErrorMessage()) {
            validateRule.setErrorMessage(rule.errorMessageInfo(locale));
        }
        //当配置文件中配置了order值,则走配置文件进行赋值。否则走原来逻辑，避免以前配置进来报错
        if (null != rule.getOrder()) {
            validateRule.setOrder(rule.getOrder());
        }

        return validateRule;
    }

    public boolean ruleIsEmpty() {
        return this.type == null && this.parameter == null;
    }

}
