package com.digiwin.mobile.mobileuibot.core.strategy.compdatabuild;

import com.digiwin.mobile.mobileuibot.api.AgileDataApiRequest;
import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.api.ApiRequestAgileDataInAiChat;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.file.FileUtil;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.core.component.MobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.basic.Empty;
import com.digiwin.mobile.mobileuibot.core.componentsetting.ComponentSettingRequestIdEnum;
import com.digiwin.mobile.mobileuibot.core.layout.doublepattern.util.ModuleUtils;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.proxy.DigiwinAthenaApiResponse;
import com.digiwin.mobile.mobileuibot.proxy.adt.service.DigiwinAdtProxyService;
import com.digiwin.mobile.mobileuibot.proxy.agiledataecho.model.EchoConvertChartResultData;
import com.digiwin.mobile.mobileuibot.proxy.agiledataecho.model.EchoFilterData;
import com.digiwin.mobile.mobileuibot.proxy.agiledataecho.service.DigiwinAgileDataEchoProxyService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.AgileDataConstStringEnum;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.AgileDataIntermediateDsl;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <p>功能描述：敏捷问数图表展示策略类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: UiBotCompDataBuildAgileDataContentForAiAssistantStrategy
 * @Author: Zaregoto
 * @Date: 2023/12/14 18:48
 */
@Slf4j
@Component("UiBotCompDataBuildAgileDataContentForAiAssistantStrategy")
public class UiBotCompDataBuildAgileDataContentForAiAssistantStrategy implements UiBotCompDataBuildStrategy<MobileComponent> {

    @Autowired
    private LocaleService localeService;

    @Autowired
    private DigiwinAdtProxyService digiwinAdtProxyService;

    @Resource(name = "defaultThreadPool")
    private ThreadPoolTaskExecutor threadPoolTaskExecutor;

    @Autowired
    private DigiwinAgileDataEchoProxyService echoProxyService;

    @Override
    public String getTypeByRequestId() {
        return ComponentSettingRequestIdEnum.AGILE_DATA_FOR_AI_ASSISTANT.getReuqestId();
    }

    @Override
    public MobileComponent buildComponentData(ApiRequest apiRequest)
            throws IOException {
        StopWatch stopWatch = new StopWatch("BuildAgileDataForAiAssistant");
        Boolean useMockData = AppContext.getUseMockData();
        Boolean modelTypeShow = AppRequestContext.requestNeedShowMockData();
        MobileComponent mobileComponent;
        if (!useMockData && !modelTypeShow) {
            mobileComponent = this.doActually(apiRequest, stopWatch);
        } else {
            mobileComponent = this.doMockData(apiRequest);
        }
        log.debug(stopWatch.prettyPrint());

        return mobileComponent;
    }

    private MobileComponent doActually(ApiRequest apiRequest, StopWatch stopWatch) {
        String locale = apiRequest.getLocale();

        MobileComponent mobileComponent;
        stopWatch.start("GetData From Echo AgileReport API");

        AgileDataIntermediateDsl intermediateDsl;

        AgileDataApiRequest agileData = AgileDataApiRequest.builder()
                .dataId(apiRequest.getDataId())
                .iamUserToken(apiRequest.getIamUserToken())
                .locale(locale)
                .tenantId(apiRequest.getTenantId())
                .build();
        /**
         * 除了筛选条件接口(/api/ai/v1/agile/show/filter)入参永远传原始快照Id：orgSnapshotId
         * 别的情况(如切换图表/收藏/订阅)全部传当前最新的快照Id：snapshotId
         */
        if (ObjectUtils.isNotEmpty(apiRequest.getFilterParams())) {
            EchoFilterData extFilterData = JsonUtil.jsonStringToObject(JsonUtil.javaObjectToJsonString(apiRequest.getFilterParams()), EchoFilterData.class);
            intermediateDsl = JsonUtil.jsonStringToObject(JsonUtil.javaObjectToJsonString(extFilterData.getDsl()), AgileDataIntermediateDsl.class);
            intermediateDsl.setDataTipMessage(extFilterData.getDataTipMessage());
        } else if (apiRequest.getIfChartConvertIf()) {
            agileData.setTransId(apiRequest.getTransId());
            EchoConvertChartResultData echoConvertChartResultData = echoProxyService.convertChartDataset(agileData);
            intermediateDsl = echoConvertChartResultData.getDsl();
            intermediateDsl.setDataTipMessage(echoConvertChartResultData.getDataTipMessage());
        } else {
            intermediateDsl = echoProxyService.getAgileDataDsl(agileData);
        }
        stopWatch.stop();
        if (StringUtils.hasLength(intermediateDsl.getErrorMessage())) {
            String emptyText = intermediateDsl.getErrorMessage();
            return Empty.create(emptyText, true, 80, 74, "IMAGE_EMPTY_WIDGET");
        }
        stopWatch.start("Start Generate Agile Data Detail...");
        // 构建页面的数据呈现组件清单
        List<MobileComponent> agileDataMobileComponents =
                ModuleUtils.generateComponentFromAgileDataIntermediateDsl(apiRequest, intermediateDsl, true);

        if (CollectionUtils.isEmpty(agileDataMobileComponents)) {
            String emptyText = this.localeService.getLanguageValue(locale, AgileDataConstStringEnum.AI_ASSISTANT_NO_DATA_REPLY.getValue());
            mobileComponent = Empty.create(emptyText, true, 80, 74, "IMAGE_EMPTY_WIDGET");
        } else {
            mobileComponent = agileDataMobileComponents.get(0);
        }
        stopWatch.stop();

        return mobileComponent;
    }

    private MobileComponent doMockData(ApiRequest apiRequest) {
        String locale = apiRequest.getLocale();

        String dslString = FileUtil.readText("./static/agiledata/aiassistant-dsl-response.json");
//        String dslString = FileUtil.readText("./static/agiledata/error-dsl-response.json");
        DigiwinAthenaApiResponse<AgileDataIntermediateDsl> response =
                JsonUtil.jsonStringToObject(dslString,
                        new TypeReference<DigiwinAthenaApiResponse<AgileDataIntermediateDsl>>() {
                        });
        AgileDataIntermediateDsl intermediateDsl = response.getResponse();

        // 提前准备空组件
        String emptyText = this.localeService.getLanguageValue(locale, "暂无数据");
        Empty emptyMobileComponent = Empty.create(emptyText, true, 80, 74, "IMAGE_EMPTY_WIDGET");

        // 查不到详情，给一个空组件
        if (intermediateDsl == null || intermediateDsl.isEmpty()) {
            return emptyMobileComponent;
        }

        // 构建页面的数据呈现组件清单
        List<MobileComponent> agileDataMobileComponents =
                ModuleUtils.generateComponentFromAgileDataIntermediateDsl(apiRequest, intermediateDsl, true);

        // 随机排序一下
//        Collections.shuffle(agileDataMobileComponents);

        LocalDateTime localDateTime = LocalDateTime.now();
        // 获取其中的一个给前端
        return localDateTime.getSecond() % 2 == 0 ? agileDataMobileComponents.get(0) :
                agileDataMobileComponents.get(1);
    }

    /**
     * 上报uibot获取ADT数据的异常响应到ADT
     *
     * @param apiRequest
     * @param errorMsg
     */
    private void saveAbnormalResponseToAdtService(ApiRequest apiRequest, String errorMsg) {
        this.threadPoolTaskExecutor.execute(() -> {
            Map<String, Object> params = new HashMap<>(4);
            if (apiRequest instanceof ApiRequestAgileDataInAiChat) {
                params.put("generateSerialNo", ((ApiRequestAgileDataInAiChat) apiRequest).getGenerateSerialNo());
                params.put("question", ((ApiRequestAgileDataInAiChat) apiRequest).getQu());
            } else {
                params.put("generateSerialNo", null);
                params.put("question", "");
            }
            params.put("snapshotId", apiRequest.getDataId());
            params.put("errorMsg", errorMsg);
            this.digiwinAdtProxyService.saveAbnormal(apiRequest.getIamUserToken(), apiRequest.getTenantId(), apiRequest.getLocale(), params);
        });
    }
}