package com.digiwin.mobile.mobileuibot.core.strategy.itemdetail;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.common.file.FileUtil;
import com.digiwin.mobile.mobileuibot.common.http.HttpService;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.core.UiBotModelBuildService;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSetting;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSettingIdPresetEnum;
import com.digiwin.mobile.mobileuibot.core.strategy.modelbuild.UiBotModelBuildStrategy;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcBacklogDetail;
import com.digiwin.mobile.mobileuibot.proxy.atmc.model.DigiwinAtmcBacklogReadCount;
import com.digiwin.mobile.mobileuibot.proxy.atmc.service.DigiwinAtmcProxyService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.digiwin.mobile.mobileuibot.proxy.uibot.service.DigiwinPcUiBotProxyService;
import com.digiwin.mobile.mobileuibot.task.model.TaskDetail;
import com.digiwin.mobile.mobileuibot.task.strategy.TaskDetailBuildStrategyFactory;
import lombok.SneakyThrows;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.StopWatch;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

/**
 * <p>功能描述：构建明细条目项的详情时的策略。</p>
 * <p>比如签核页面下方的每个明细条目项，在点开后会看到更多的详情页面。本策略就是用来生成这个详情页面的。</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: UiBotModelBuildDetailItemDetailStrategy
 * @Author: Zaregoto
 * @Date: 2021/5/18 22:28
 * @see com.digiwin.mobile.mobileuibot.task.strategy.TaskDetailBuildBizReportStrategy
 */
@Component("uiBotModelBuildDetailItemDetailStrategy")
public class UiBotModelBuildDetailItemDetailStrategy implements UiBotModelBuildStrategy {

    private static final Logger logger =
            LoggerFactory.getLogger(UiBotModelBuildDetailItemDetailStrategy.class);

    @Resource(name = "defaultThreadPool")
    private ThreadPoolTaskExecutor threadPoolTaskExecutor;

    @Autowired
    private DigiwinAtmcProxyService digiwinAtmcProxyService;

    @Autowired
    private DigiwinPcUiBotProxyService digiwinPcUiBotProxyService;

    @Autowired
    private UiBotModelBuildService uiBotModelBuildService;

    @Autowired
    private TaskDetailBuildStrategyFactory taskDetailBuildStrategyFactory;

    @Autowired
    private HttpService httpService;

    @Override
    public UiBotModel buildModel(PageSetting pageSetting, ApiRequest apiRequest)
            throws IOException {
        StopWatch stopWatch = new StopWatch("BuildDetailItemDetail");
        Boolean useMockData = AppContext.getUseMockData();
        Boolean modelTypeShow =
                AppRequestContext.requestNeedShowMockData();
        UiBotModel uiBotModel;
        if (!useMockData && !modelTypeShow) {
            uiBotModel = this.doActually(pageSetting, apiRequest, stopWatch);
        } else {
            uiBotModel = this.doMockData(pageSetting, apiRequest);
        }
        logger.debug(stopWatch.prettyPrint());

        return uiBotModel;
    }

    @SneakyThrows
    private UiBotModel doActually(PageSetting pageSetting, ApiRequest apiRequest,
                                  StopWatch stopWatch) {
        stopWatch.start("GetData From Athena API");
        //使用Future方式执行多任务
        List<Future<?>> futureList = new ArrayList<>();
        Future<DigiwinAtmcBacklogReadCount> futureReadCount = this.threadPoolTaskExecutor.submit(
                () -> this.digiwinAtmcProxyService.updateDigiwinAtmcBacklogReadCount(
                        apiRequest.getLocale(), apiRequest.getIamUserToken(), apiRequest.getTenantId(), apiRequest.getDataId()));
        futureList.add(futureReadCount);

        Future<DigiwinAtmcBacklogDetail> futureBacklogDetail = this.threadPoolTaskExecutor.submit(
                () -> this.digiwinAtmcProxyService.getDigiwinAtmcBacklogDetailByShare(apiRequest.getLocale(),
                        apiRequest.getIamUserToken(), apiRequest.getDataId()));
        futureList.add(futureBacklogDetail);

        String tmTaskId = "";
        String tmActivityId = "";
        if (null != apiRequest.getExtraParameter()) {
            tmTaskId = apiRequest.getExtraParameter().getTmTaskId();
            tmActivityId = apiRequest.getExtraParameter().getTmActivityId();
        } else {
            JSONObject executeCtxJsonObj = apiRequest.getRawData().getJSONObject("executeContext");
            tmTaskId = executeCtxJsonObj.getString("tmProjectId");
            tmActivityId = executeCtxJsonObj.getString("tmActivityId");
        }

        String finalTmTaskId = tmTaskId;
        String finalTmActivityId = tmActivityId;
        Future<UiBotModel> futureUiBotModel = this.threadPoolTaskExecutor.submit(
                () -> this.digiwinPcUiBotProxyService.getTaskTemplateShowByBacklogId(
                        apiRequest.getLocale(), apiRequest.getIamUserToken(), apiRequest.getTenantId(), apiRequest.getDataId(),
                        finalTmTaskId, finalTmActivityId, apiRequest.getRawData()));
        futureList.add(futureUiBotModel);

        DigiwinAtmcBacklogDetail backlogDetail = null;
        UiBotModel pcUiBotModel = null;
        //查询任务执行的结果
        try {
            for (Future<?> future : futureList) {
                // CPU高速轮询：每个future都并发轮循，判断完成状态然后获取结果，
                // 这一行，是本实现方案的精髓所在。即有10个future在高速轮询，完成一个future的获取结果，就关闭一个轮询
                while (true) {
                    // 获取future成功完成状态，如果想要限制每个任务的超时时间，取消本行的状态判断+future.get(1000*1, TimeUnit.MILLISECONDS)+catch超时异常使用即可。
                    if (future.isDone() && !future.isCancelled()) {
                        //获取结果
                        Object obj = future.get();
                        if (obj instanceof UiBotModel) {
                            pcUiBotModel = (UiBotModel) obj;
                        }
                        if (obj instanceof DigiwinAtmcBacklogDetail) {
                            backlogDetail = (DigiwinAtmcBacklogDetail) obj;
                        }
                        break;//当前future获取结果完毕，跳出while
                    } else {
                        //每次轮询休息1毫秒（CPU纳秒级），避免CPU高速轮循耗空CPU---》新手别忘记这个
                        Thread.sleep(1);
                    }
                }
            }
        } catch (InterruptedException | ExecutionException e) {
            logger.error(e.getLocalizedMessage());
            e.printStackTrace();
            String message = e.getMessage();
            if (null != e.getCause()) {
                message = e.getCause().getMessage();
            }
            throw new ServiceException(message);
        } finally {

        }
        stopWatch.stop();

        // 针对API返回的PC端UiBotModel进行解析，并转换为移动端所需接口格式
        Assert.notNull(backlogDetail, "task detail must have detail!");
        Assert.notNull(backlogDetail.getId(), "task detail must have detail!");
        Assert.notNull(pcUiBotModel, "task detail must have uibotmodel data!");

        stopWatch.start("Build Detail-Item-Detail Page Data");
        TaskDetail taskDetailItemDetail = null;
        // 根据pcUiBotModel的内容，组出taskDetail内的组件信息。注意这里传了一个TaskDetailItemDetail对象
        taskDetailItemDetail = this.taskDetailBuildStrategyFactory.get(apiRequest.getPageId())
                .build(apiRequest, pageSetting, pcUiBotModel, backlogDetail);
        stopWatch.stop();

        stopWatch.start("Build Common Page Model");
        UiBotModel uiBotModel =
                this.uiBotModelBuildService.buildModel(pageSetting, taskDetailItemDetail, true);
        stopWatch.stop();

        return uiBotModel;
    }

    private UiBotModel doMockData(PageSetting pageSetting, ApiRequest apiRequest) {
        try {
            Thread.sleep(100L);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        String data;
        UiBotModel uiBotModel = null;
        if (PageSettingIdPresetEnum.MOBILE_ATHENA_DETAIL_ITEM_DETAIL.toString()
                .equalsIgnoreCase(pageSetting.getPageId())) {
            data = FileUtil.readText(
                    "static/appstore-use/detail-item-detail-" + apiRequest.getDataId() + ".json");
        } else {
            data = "";
        }
        if (null != data && !data.isEmpty()) {
            uiBotModel = JsonUtil.jsonStringToObject(data, UiBotModel.class);
        }

        return uiBotModel;
    }
}
