package com.digiwin.mobile.mobileuibot.core.strategy.modelbuild;

import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.common.calculate.UUIDUtil;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.file.FileUtil;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.core.appec.SlipTitle;
import com.digiwin.mobile.mobileuibot.core.component.MobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.basic.Empty;
import com.digiwin.mobile.mobileuibot.core.component.title.titlebody.TitleBody;
import com.digiwin.mobile.mobileuibot.core.layout.doublepattern.util.ModuleUtils;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSetting;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.proxy.DigiwinAthenaApiResponse;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotExecuteContext;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotPageData;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.AgileDataConstStringEnum;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.agiledata.AgileDataIntermediateDsl;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.service.DigiwinPcUiBotProxyService;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * <p>功能描述：UiBot数据模型构建策略接口--实现敏捷数据详情页面构建</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: UiBotModelBuildAgileDataDetailStrategy
 * @Author: Zaregoto
 * @Date: 2023/11/23 19:09
 */
@Slf4j
@Component("uiBotModelBuildAgileDataDetailStrategy")
public class UiBotModelBuildAgileDataDetailStrategy implements UiBotModelBuildStrategy {

    private final DigiwinPcUiBotProxyService digiwinPcUiBotProxyService;

    private final LocaleService localeService;

    public UiBotModelBuildAgileDataDetailStrategy(DigiwinPcUiBotProxyService digiwinPcUiBotProxyService, LocaleService localeService) {
        this.digiwinPcUiBotProxyService = digiwinPcUiBotProxyService;
        this.localeService = localeService;
    }

    @Override
    public UiBotModel buildModel(PageSetting pageSetting, ApiRequest apiRequest)
            throws IOException {
        StopWatch stopWatch = new StopWatch("Build Agile Data Detail Page");
        Boolean useMockData = AppContext.getUseMockData();
        Boolean modelTypeShow = AppRequestContext.requestNeedShowMockData();

        UiBotModel uiBotModel;
        if (useMockData || modelTypeShow) {
            uiBotModel = this.doMockData(pageSetting, apiRequest);
        } else {
            uiBotModel = this.doActually(pageSetting, apiRequest, stopWatch);
        }
//        uiBotModel = this.doMockData(pageSetting, apiRequest);

        log.debug(stopWatch.prettyPrint());
        return uiBotModel;
    }

    @SneakyThrows
    public UiBotModel doActually(PageSetting pageSetting, ApiRequest apiRequest,
                                 StopWatch stopWatch) {
        String locale = apiRequest.getLocale();

        UiBotModel resultUiBotModel = pageSetting.getPageModel();
        stopWatch.start("GetData From Web UIBot API");
        AgileDataIntermediateDsl intermediateDsl = this.digiwinPcUiBotProxyService.getAgileReportShow(
                apiRequest.getDataId(), Collections.emptyMap(), apiRequest.getIamUserToken(), apiRequest.getTenantId(), locale);
        stopWatch.stop();

        // 查不到详情，给一个空组件
        if (StringUtils.hasLength(intermediateDsl.getErrorMessage())) {
            String errorMessage = intermediateDsl.getErrorMessage();
            resultUiBotModel = UiBotModel.emptyImageUibotModel(locale, errorMessage, 80, 74, "IMAGE_EMPTY_WIDGET");
            return resultUiBotModel;
        }

        stopWatch.start("Start Generate Agile Data Detail...");
        // 构建页面的标题组件
        TitleBody titleBody = this.createTitleBodyFromDslExecuteContext(intermediateDsl);

        // 构建页面的数据呈现组件清单
        List<MobileComponent> agileDataMobileComponents =
                ModuleUtils.generateComponentFromAgileDataIntermediateDsl(apiRequest, intermediateDsl, false);

        if (CollectionUtils.isEmpty(agileDataMobileComponents)) {
            String emptyText = this.localeService.getLanguageValue(locale, AgileDataConstStringEnum.DETAIL_PAGE_NO_DATA_REPLY.getValue());
            Empty emptyMobileComponent = Empty.create(emptyText, true, 80, 74, "IMAGE_EMPTY_WIDGET");

            agileDataMobileComponents.add(emptyMobileComponent);
        }

        // 整合页面标题和呈现数据组件，放入最终页面模型中
        List<MobileComponent> resultMobileComponents = new ArrayList<>(agileDataMobileComponents.size() + 1);
        resultMobileComponents.add(titleBody);
        resultMobileComponents.addAll(agileDataMobileComponents);

        resultUiBotModel = this.addMobileComponentsToUiBotModel(resultUiBotModel, resultMobileComponents);

        String pageTitle = this.localeService.getLanguageValue(locale, titleBody.getContent().getText());
        if(resultUiBotModel != null && resultUiBotModel.getSlip() != null){
            //右滑摇人标题
            if (resultUiBotModel.getSlip().getSlipConfig() != null && resultUiBotModel.getSlip().getSlipConfig().getLeft() != null) {
                SlipTitle slipTitle = resultUiBotModel.getSlip().getSlipConfig().getLeft().getSubTitle();
                slipTitle.setContent(pageTitle + " " + this.localeService.getLanguageValue(locale, "当前沟通"));
            }
        }
        stopWatch.stop();

        return resultUiBotModel;
    }

    /**
     * 根据中间态DSL数据生成页面标题组件
     *
     * @param intermediateDsl
     * @return
     */
    private TitleBody createTitleBodyFromDslExecuteContext(AgileDataIntermediateDsl intermediateDsl) {
        UiBotExecuteContext executeContext = intermediateDsl.getExecuteContext();
        String title = executeContext.getString("title");
        TitleBody titleBody = TitleBody.create(title, null, Collections.emptyList());

        return titleBody;
    }

    private UiBotModel addMobileComponentsToUiBotModel(final UiBotModel uiBotModel, final List<MobileComponent> mobileComponents) {
        if (null == uiBotModel || CollectionUtils.isEmpty(mobileComponents)) {
            return null;
        }
        mobileComponents.forEach(mc -> {
            String schema = UUIDUtil.getUuid();
            UiBotLayout mcLayout = new UiBotLayout(schema, mc.returnComponentType());
            UiBotPageData mcAddedPageData = uiBotModel.getPageData();
            mcAddedPageData.put(schema, mc);

            uiBotModel.addLayout(mcLayout);
            uiBotModel.setPageData(mcAddedPageData);
        });

        return uiBotModel;
    }

    private UiBotModel doMockData(PageSetting pageSetting, ApiRequest apiRequest) {
        String locale = apiRequest.getLocale();
        String dslString = FileUtil.readText("./static/agiledata/dsl-response.json");
//        String dslString = FileUtil.readText("./static/agiledata/error-dsl-response.json");
        DigiwinAthenaApiResponse<AgileDataIntermediateDsl> response =
                JsonUtil.jsonStringToObject(dslString,
                        new TypeReference<DigiwinAthenaApiResponse<AgileDataIntermediateDsl>>() {
                        });
        AgileDataIntermediateDsl intermediateDsl = response.getResponse();

        UiBotModel resultUiBotModel = pageSetting.getPageModel();

        // 查不到详情，给一个空组件
        if (intermediateDsl == null || intermediateDsl.isEmpty()) {
            resultUiBotModel = UiBotModel.emptyImageUibotModel(locale, AgileDataConstStringEnum.DETAIL_PAGE_NO_DATA_REPLY.getValue(), 80, 74, "IMAGE_EMPTY_WIDGET");
            return resultUiBotModel;
        }

        // 构建页面的标题组件
        TitleBody titleBody = this.createTitleBodyFromDslExecuteContext(intermediateDsl);

        // 构建页面的数据呈现组件清单
        List<MobileComponent> agileDataMobileComponents =
                ModuleUtils.generateComponentFromAgileDataIntermediateDsl(apiRequest, intermediateDsl, false);

        // 整合页面标题和呈现数据组件，放入最终页面模型中
        List<MobileComponent> resultMobileComponents = new ArrayList<>(agileDataMobileComponents.size() + 1);
        resultMobileComponents.add(titleBody);
        resultMobileComponents.addAll(agileDataMobileComponents);

        resultUiBotModel = this.addMobileComponentsToUiBotModel(resultUiBotModel, resultMobileComponents);

        String pageTitle = this.localeService.getLanguageValue(locale, titleBody.getContent().getText());
        if(resultUiBotModel != null && resultUiBotModel.getSlip() != null){
            //右滑摇人标题
            if (resultUiBotModel.getSlip().getSlipConfig() != null && resultUiBotModel.getSlip().getSlipConfig().getLeft() != null) {
                SlipTitle slipTitle = resultUiBotModel.getSlip().getSlipConfig().getLeft().getSubTitle();
                slipTitle.setContent(pageTitle + " " + this.localeService.getLanguageValue(locale, "当前沟通"));
            }
        }
        return resultUiBotModel;
    }
}
