package com.digiwin.mobile.mobileuibot.core.strategy.modelbuild;

import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.datetime.DateTimeUtil;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponentPerson;
import com.digiwin.mobile.mobileuibot.core.component.action.Action;
import com.digiwin.mobile.mobileuibot.core.component.action.ActionTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.basic.*;
import com.digiwin.mobile.mobileuibot.core.component.business.bizapproval.BizApprovalTabBuilder;
import com.digiwin.mobile.mobileuibot.core.component.business.bizapproval.BizApprovalTabList;
import com.digiwin.mobile.mobileuibot.core.component.button.BottomButtonDigiwinAthena;
import com.digiwin.mobile.mobileuibot.core.component.button.BottomButtonList;
import com.digiwin.mobile.mobileuibot.core.component.divider.Divider;
import com.digiwin.mobile.mobileuibot.core.component.divider.DividerTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.group.CustomGroup;
import com.digiwin.mobile.mobileuibot.core.component.group.CustomGroupContent;
import com.digiwin.mobile.mobileuibot.core.component.group.CustomGroupStyleTypeEnum;
import com.digiwin.mobile.mobileuibot.core.component.input.attachment.*;
import com.digiwin.mobile.mobileuibot.core.component.tab.customtab.CustomTabScrollModeEnum;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSetting;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSettingIdPresetEnum;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotPageData;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.parser.ParseFieldData;
import com.fasterxml.jackson.core.type.TypeReference;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * 功能: APP我的签核型任务详情页下 条目详情
 *
 * @author: wjw
 * @date: created at 2023/5/19
 */
@Component("uiBotModelBuildMyApprovalTaskItemDetailStrategy")
public class UiBotModelBuildMyApprovalTaskItemDetailStrategy implements UiBotModelBuildStrategy {

    /**
     * 控制显示隐藏的字段数量
     */
    private static final Integer FILED_MORE_SIZE = 5;

    @Autowired
    private BizApprovalTabBuilder bizApprovalTabBuilder;

    @Override
    public UiBotModel buildModel(PageSetting pageSetting, ApiRequest apiRequest) throws IOException {
        UiBotModel uiBotModel = null;
        Boolean useMockData = AppContext.getUseMockData();
        if (!useMockData) {
            uiBotModel = this.doActually(pageSetting, apiRequest);
        } else {
            uiBotModel = this.doMockData(apiRequest);
        }
        return uiBotModel;
    }

    private UiBotModel doActually(PageSetting pageSetting, ApiRequest apiRequest) {
        UiBotModel uiBotModel = pageSetting.getPageModel();
        UiBotPageData pageData = uiBotModel.getPageData();

        Map<String, Object> rawData = apiRequest.getRawData();
        List<ParseFieldData> fieldDataList = new ArrayList<>();
        List<UiBotLayout> tableLayout = new ArrayList<>();
        UiBotPageData tablePageData = new UiBotPageData();
        if (rawData.get("data") != null) {
            fieldDataList = JsonUtil.objectToJavaObject(rawData.get("data"), new TypeReference<List<ParseFieldData>>() {
            });
        }
        if (rawData.get("layout") != null) {
            tableLayout = JsonUtil.objectToJavaObject(rawData.get("layout"), new TypeReference<List<UiBotLayout>>() {
            });
        }
        if (rawData.get("pageData") != null) {
            tablePageData = JsonUtil.objectToJavaObject(rawData.get("pageData"), UiBotPageData.class);
        }
        for (UiBotLayout layout : uiBotModel.getLayout()) {
            switch (layout.getType()) {
                case CustomGroup.COMPONENT_TYPE:
                    // 单身详情数据构建
                    pageData.put(layout.getSchema(), this.buildCustomGroup(apiRequest, fieldDataList));
                    break;
                case "TABS":
                    // 子单身tabs 数据构建
                    UiBotModel tabUiBotModel = UiBotModel.emptyUiBotModel();
                    tabUiBotModel.setLayout(tableLayout);
                    tabUiBotModel.setPageData(tablePageData);
                    BizApprovalTabList tabs = this.bizApprovalTabBuilder.buildAllTabs(apiRequest, tabUiBotModel);
                    tabs.forEach(tab -> tab.setMScrollMode(CustomTabScrollModeEnum.NO_SCROLL.getMode()));
                    pageData.put(layout.getSchema(), tabs);
                    break;
                case BottomButtonDigiwinAthena.COMPONENT_TYPE:
                    BottomButtonList bottomButtonList =
                            JsonUtil.objectToJavaObject(pageData.get(layout.getSchema()), BottomButtonList.class);
                    bottomButtonList.forEach(bottomButtonDigiwinAthena -> {
                        String name = bottomButtonDigiwinAthena.getName();
                        bottomButtonDigiwinAthena.setName(LocaleUtil.getMobileTextByDatabaseKey(apiRequest.getLocale(), name));
                    });
                    pageData.put(layout.getSchema(), bottomButtonList);
                    break;
                default:
                    // 什么都不做
                    break;
            }
        }
        return uiBotModel;
    }

    /**
     * 构建单身详情数据 CustomGroup
     *
     * @param apiRequest    请求入参
     * @param fieldDataList 详情字段列表信息
     * @return CustomGroup
     */
    private CustomGroup buildCustomGroup(ApiRequest apiRequest, List<ParseFieldData> fieldDataList) {
        String locale = apiRequest.getLocale();
        CustomGroup customGroup = new CustomGroup();
        customGroup.setStyleType(CustomGroupStyleTypeEnum.STYLE_APPROVAL_ITEM_DETAIL.getValue());
        CustomGroupContent customGroupContent = this.buildCustomGroupContent(apiRequest, fieldDataList);
        customGroup.setContent(customGroupContent);
        /* 字段的收起展开 */
        if (customGroupContent.getLayout().size() > FILED_MORE_SIZE) {
            customGroup.setUseExpand(true);
            HiddenConfig hiddenConfig = new HiddenConfig();
            hiddenConfig.setHiddenType(HiddenConfigTypeEnum.FIELD.getValue());
            hiddenConfig.setHiddenExpandContent(LocaleUtil.getMobileTextByKey(locale, "查看更多"));
            hiddenConfig.setHiddenUnexpandContent(LocaleUtil.getMobileTextByKey(locale, "收起"));
            hiddenConfig.setHiddenCount(FILED_MORE_SIZE);
            customGroup.setHiddenConfig(hiddenConfig);
        }
        return customGroup;
    }

    /**
     * 构建单身详情数据 CustomGroup 下CustomGroupContent 数据
     *
     * @param apiRequest    请求入参
     * @param fieldDataList 详情字段列表信息
     * @return CustomGroupContent
     */
    private CustomGroupContent buildCustomGroupContent(ApiRequest apiRequest, List<ParseFieldData> fieldDataList) {
        CustomGroupContent customGroupContent = new CustomGroupContent();
        List<UiBotLayout> layoutList = new ArrayList<>(fieldDataList.size());
        UiBotPageData pageData = new UiBotPageData();
        customGroupContent.setLayout(layoutList);
        customGroupContent.setPageData(pageData);

        //找出执行人、任务进展状态、计划完成日期、签核状态
        //activity__performerName、activity__backLog__data、activity__planEndTime
        ParseFieldData performerNameFieldData = null;
        ParseFieldData backLogDataNameFieldData = null;
        ParseFieldData planEndTimeFieldData = null;
        ParseFieldData approvalStateFieldData = null;
        for (ParseFieldData fieldData : fieldDataList) {
            if ("activity__performerName".equalsIgnoreCase(fieldData.getSchema())) {
                performerNameFieldData = fieldData;
            }
            if ("activity__backLog__data".equalsIgnoreCase(fieldData.getSchema())) {
                backLogDataNameFieldData = fieldData;
            }
            if ("activity__planEndTime".equalsIgnoreCase(fieldData.getSchema())) {
                planEndTimeFieldData = fieldData;
            }
            if ("activity__approval__state".equalsIgnoreCase(fieldData.getSchema())) {
                approvalStateFieldData = fieldData;
            }
        }
        if (performerNameFieldData != null) {
            fieldDataList.remove(performerNameFieldData);
        }
        if (backLogDataNameFieldData != null) {
            fieldDataList.remove(backLogDataNameFieldData);
            Field field = calculateTaskStatus(backLogDataNameFieldData, apiRequest.getLocale());
            layoutList.add(new UiBotLayout(field.getSchema(), Field.COMPONENT_TYPE));
            pageData.put(field.getSchema(), field);
        }
        if (planEndTimeFieldData != null) {
            fieldDataList.remove(planEndTimeFieldData);
            Field field = convertTabItemCardContent(planEndTimeFieldData);
            layoutList.add(new UiBotLayout(field.getSchema(), Field.COMPONENT_TYPE));
            pageData.put(field.getSchema(), field);
        }
        if (approvalStateFieldData != null) {
            fieldDataList.remove(approvalStateFieldData);
            Field field = approvalStateField(approvalStateFieldData, apiRequest.getLocale(), apiRequest.getDataId());
            layoutList.add(new UiBotLayout(field.getSchema(), Field.COMPONENT_TYPE));
            pageData.put(field.getSchema(), field);
        }
        if (!layoutList.isEmpty()) {
            //有任务进展状态或计划完成日期，加分割线
            Divider divider = new Divider();
            divider.setType(DividerTypeEnum.ELEMENT_DIVIDER.getValue());
            String schema = divider.returnComponentType() + "_" + divider.getCmptUuid();
            layoutList.add(new UiBotLayout(schema, Divider.COMPONENT_TYPE));
            pageData.put(schema, divider);
        }
        Boolean needTitle = (Boolean) Optional.ofNullable(apiRequest.getRawData().get("needTitle")).orElse(false);
        //标题
        if (needTitle) {
            ParseFieldData fieldData = fieldDataList.get(0);
            fieldDataList.remove(fieldData);
            Field field = Field.create(fieldData, apiRequest.getLocale());
            field.setValueStyleType(FieldValueStyleTypeEnum.VALUE_BIG_TITLE.getValue());
            field.setAlignmentType(FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_LEFT.getValue());
            field.setLabel("");
            field.setLabelVisible(false);
            layoutList.add(new UiBotLayout(field.getSchema(), Field.COMPONENT_TYPE));
            pageData.put(field.getSchema(), field);
        }
        for (ParseFieldData fieldData : fieldDataList) {
            if ("DYNAMIC_GRAPH_VIEWER".equalsIgnoreCase(fieldData.getType())) {
                //暂不支持产品图号信息类型展示
                continue;
            }
            //签核类型数据
            if ("SIGN_OFF_PROGRESS".equalsIgnoreCase(fieldData.getType())) {
                Field field = approvalStateField(approvalStateFieldData, apiRequest.getLocale(), apiRequest.getDataId());
                layoutList.add(new UiBotLayout(field.getSchema(), Field.COMPONENT_TYPE));
                pageData.put(field.getSchema(), field);
                continue;
            }
            //附件类型
            if ("FILE_UPLOAD".equalsIgnoreCase(fieldData.getType())
                    || "TASK_FILE_UPLOAD".equalsIgnoreCase(fieldData.getType())) {
                Attachment attachment = new Attachment();
                attachment.setTitle(fieldData.getLabel());
                attachment.setType(AttachmentTypeEnum.READ_ONLY.getValue());
                attachment.setStyleType(AttachmentStyleTypeEnum.TITLE_WEAKENED.getValue());
                if (fieldData.getData() != null) {
                    Map dataMap = (Map) fieldData.getData();
                    if (dataMap.get("data") != null) {
                        List<Map<String, Object>> dataList = (ArrayList) dataMap.get("data");
                        if (!CollectionUtils.isEmpty(dataList)) {
                            List<AttachmentFile> fileList = dataList.stream().map(data -> {
                                AttachmentFile file = new AttachmentFile();
                                file.setFileId((String) data.get("id"));
                                if (data.get("size") instanceof Double) {
                                    Double size = (Double) data.get("size");
                                    file.setFileSize(size.longValue());
                                } else if (data.get("size") instanceof Long) {
                                    file.setFileSize((Long) data.get("size"));
                                } else if (data.get("size") instanceof String) {
                                    file.setFileSize(Long.parseLong((String) data.get("size")));
                                }
                                file.setUploader((String) data.get("upload_user_name"));
                                file.setFileName((String) data.get("name"));
                                String createDate = data.get("create_date") == null ? "" : String.valueOf(data.get("create_date"));
                                if (StringUtils.hasLength(createDate)) {
                                    createDate = createDate.replaceAll("/", "-");
                                }
                                file.setUploadTime(createDate);
                                file.setFileExtension(AttachmentUtil.getAttachmentFileSuffix((String) data.get("name")));
                                return file;
                            }).collect(Collectors.toList());
                            attachment.setFileList(fileList);
                        }
                    }

                }
                layoutList.add(new UiBotLayout(fieldData.getSchema(), Attachment.COMPONENT_TYPE));
                pageData.put(fieldData.getSchema(), attachment);
            } else {
                //普通文本类型
                Field field = Field.create(fieldData, apiRequest.getLocale());
                if (field != null) {
                    field.setLabelStyleType(2);
                    field.setValueStyleType(1);
                    field.setAlignmentType(FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_LEFT_NEWLINE.getValue());
                    layoutList.add(new UiBotLayout(field.getSchema(), Field.COMPONENT_TYPE));
                    pageData.put(field.getSchema(), field);
                }
            }
        }

        return customGroupContent;
    }

    private Field convertTabItemCardContent(ParseFieldData fieldData) {
        if (fieldData == null || fieldData.getData() == null) {
            return null;
        }
        Field field = new Field();
        field.setLabel(fieldData.getLabel());
        field.setLabelStyleType(2);
        field.setValueStyleType(1);
        field.setSchema(fieldData.getSchema());
        field.setAlignmentType(FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_LEFT.getValue());
        AtomicReference<Object> value = new AtomicReference<>(fieldData.getData());
        if (fieldData.getOptions() != null) {
            fieldData.getOptions().forEach(uiBotTableColumnEditorOption -> {
                if (value.get().equals(uiBotTableColumnEditorOption.getValue())) {
                    value.set(uiBotTableColumnEditorOption.getTitle());
                }
            });
        }
        field.setValue(value.get().toString());
        return field;
    }

    /**
     * 转化签核类型数据
     *
     * @param fieldData
     * @param locale
     * @return
     */
    private Field approvalStateField(ParseFieldData fieldData, String locale, String dataId) {
        Field field = new Field();
        field.setLabel(fieldData.getLabel());
        field.setSchema(fieldData.getSchema());
        field.setLabelStyleType(2);
        field.setValueStyleType(1);
        Map valueMap = (Map) fieldData.getData();
        String approve = valueMap.get("approve").toString();
        switch (approve) {
            case "3":
                field.setValue(LocaleUtil.getMobileTextByKey(locale, "已签核"));
                break;
            default:
                field.setValue(LocaleUtil.getMobileTextByKey(locale, "签核中"));
                break;
        }
        Action action = new Action();
        action.setType(ActionTypeEnum.OPEN_NEW_PAGE_FROM_BOTTOM.getValue());
        action.setJumpPageTitle(fieldData.getLabel());
        action.setJumpPageId(PageSettingIdPresetEnum.MOBILE_APPROVAL_TASK_PROGRESS.toString());
        action.setRawData(valueMap);
        action.setDataId(dataId);
        field.setAction(action);
        field.setAlignmentType(FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_LEFT.getValue());
        return field;
    }

    /**
     * 处理任务进展状态、执行人
     *
     * @param fieldData
     * @return
     */
    private Field calculateTaskStatus(ParseFieldData fieldData, String locale) {
        Map<String, Object> dataMap = (Map<String, Object>) fieldData.getData();
        //计划完成时间
        String planEndTime = (String) dataMap.get("planEndTime");
        //是否关闭（完成）
        Boolean close = (Boolean) dataMap.get("closed");
        //实际完成日期
        String endTime = (String) dataMap.get("endTime");
        //执行人id
        String performerId = (String) dataMap.get("performerId");
        //执行人name
        String performerName = (String) dataMap.get("performerName");

        Field field = new Field();
        field.setSchema(fieldData.getSchema());
        field.setLabelStyleType(2);
        field.setAlignmentType(FieldAlignmentTypeEnum.LABEL_LEFT_VALUE_LEFT.getValue());
        field.setLabel(fieldData.getLabel());
        if (close) {
            //已完成--正常完成、逾时完成
            if (DateTimeUtil.compareTime(endTime, planEndTime)) {
                //实际完成日期早于计划完成日期--正常完成
                field.setValue(LocaleUtil.getMobileTextByKey(locale, "已完成"));
                field.setType("DONE");
            } else {
                //实际完成日期晚于计划完成日期--逾时完成
                //逾期天数
                int days = DateTimeUtil.getDateTimePeriod(planEndTime, endTime);
                field.setValue(String.format(LocaleUtil.getMobileTextByKey(locale, "逾期后完成"), days));
                field.setType("OVERDUE_DONE");
            }
        } else {
            //未完成
            if (!DateTimeUtil.isBeforeNowDateTime(planEndTime)) {
                //未到计划完成日期--正在进行
                field.setValue(LocaleUtil.getMobileTextByKey(locale, "正在进行"));
                field.setType("LOADING");
            } else {
                //已到计划完成日期--逾时xx天
                int days = DateTimeUtil.getDateTimePeriod(planEndTime, DateTimeUtil.getTodayTimeUseDefaultPattern());
                field.setValue(String.format(LocaleUtil.getMobileTextByKey(locale, "逾期天数"), days));
                field.setType("OVERDUE");
            }
        }
        if (StringUtils.hasLength(performerName)) {
            BaseMobileComponentPerson person = new BaseMobileComponentPerson();
            person.setName(performerName);
            person.setUserId(performerId);
            field.setPerson(person);
        }
        return field;
    }


    private UiBotModel doMockData(ApiRequest apiRequest) {
        return null;
    }
}
