package com.digiwin.mobile.mobileuibot.designer;

import com.digiwin.mobile.mobileuibot.api.ApiRequest;
import com.digiwin.mobile.mobileuibot.common.http.model.AthenaBasicHttpRequestHead;
import com.digiwin.mobile.mobileuibot.core.strategy.modelbuild.pattern.CustomPatternTaskTypeEnum;
import com.digiwin.mobile.mobileuibot.designer.uibot.UiBotDesignerConfig;
import com.digiwin.mobile.mobileuibot.designer.uibot.UiBotDesignerDataSource;
import com.digiwin.mobile.mobileuibot.designer.uibot.service.UiBotDesignerService;
import com.digiwin.mobile.mobileuibot.mongodb.designerconfig.common.MobileDesignerConfigTypeEnum;
import com.digiwin.mobile.mobileuibot.openapi.service.ActivityFilterService;
import com.digiwin.mobile.mobileuibot.proxy.DigiwinAthenaApiRequest;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.model.response.KnowledgeMapsActivityDefinition;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.model.response.KnowledgeMapsRuleQueryResult;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.model.response.KnowledgeMapsTagQueryResult;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.model.response.metadata.KnowledgeMapsActionMetadataQueryResult;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.service.DigiwinKnowledgeMapsProxyService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.digiwin.mobile.mobileuibot.sso.enums.SsoSourceTypeEnum;
import com.digiwin.mobile.mobileuibot.task.common.TaskStringUtil;
import com.digiwin.mobile.mobileuibot.test.TestV1ToV2Util;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Future;

/**
 * <p>功能描述：鼎捷雅典娜设计器服务实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: DesignerServiceImpl
 * @Author: zaregoto
 * @Date: 2023/2/16 15:24
 */
@Slf4j
@Service("designerService")
public class DesignerServiceImpl implements DesignerService {

    @Resource(name = "defaultThreadPool")
    private ThreadPoolTaskExecutor threadPoolTaskExecutor;

    private final DigiwinKnowledgeMapsProxyService digiwinKnowledgeMapsProxyService;

    private final UiBotDesignerService uiBotDesignerService;

    private final ActivityFilterService activityFilterService;

    public DesignerServiceImpl(DigiwinKnowledgeMapsProxyService digiwinKnowledgeMapsProxyService,
                               UiBotDesignerService uiBotDesignerService,
                               ActivityFilterService activityFilterService) {
        this.digiwinKnowledgeMapsProxyService = digiwinKnowledgeMapsProxyService;
        this.uiBotDesignerService = uiBotDesignerService;
        this.activityFilterService = activityFilterService;
    }

    /**
     * 任务详情数据与设计器集成
     *
     * @param apiRequest
     * @param unprocessedUiBotModel
     * @return
     */
    @Override
    public UiBotModel integrateTaskDetailWithUiBotDesigner(
            ApiRequest apiRequest,
            UiBotModel unprocessedUiBotModel) {
        UiBotModel result = unprocessedUiBotModel;

        // 获取任务详情的UiBot设计器配置 并 组装 apiRequest中的PatternTaskInfo
        UiBotDesignerConfig uiBotDesignerConfig = apiRequest.getPatternTaskInfo().getUiBotDesignerConfig();
        if (uiBotDesignerConfig == null) {
            return result;
        }
        // 未初始化完成，则不往下进行处理
        if (!uiBotDesignerConfig.getInitialized()) {
            return result;
        }

        // TODO unprocessedUiBotModel应该在一开始就解析出来是表格还是表单布局，然后把字段（UiBotTableColumn或UiBotField）提取出来，并交给下方的方法来处理，提高性能
        // 业务字段范围和顺序的配置
        this.uiBotDesignerService.applyFieldAndOrder(result, uiBotDesignerConfig);

        // 业务字段Tag组件的配置解析
        this.uiBotDesignerService.applyFieldTag(result, uiBotDesignerConfig);

        //业务字段规则依设计器配置增加
        this.uiBotDesignerService.applyFieldRule(result, uiBotDesignerConfig);

        // 业务字段操作的配置解析与应用
        this.uiBotDesignerService.applyFieldOperation(result, uiBotDesignerConfig);

        return result;
    }

    @Override
    public UiBotDesignerConfig initPatternTaskInfoWithUiBotDesigner(
            ApiRequest apiRequest, String tmTaskId, String tmActivityId,
            Boolean systemDefault, String appCode) {

        UiBotDesignerConfig uiBotDesignerConfig = apiRequest.getPatternTaskInfo().getUiBotDesignerConfig();
        if (null != uiBotDesignerConfig) {
            return uiBotDesignerConfig;
        }
        String tenantVersion = activityFilterService.tenantVersion(apiRequest.getIamUserToken(), apiRequest.getTenantId(), apiRequest.getLocale());
        // 从设计器获取对应的页面配置
        uiBotDesignerConfig =
                this.getTaskDetailUiBotDesignerConfig(
                        apiRequest,
                        apiRequest.getLocale(),
                        apiRequest.getIamUserToken(),
                        apiRequest.getTenantId(),
                        tmTaskId, tmActivityId, tenantVersion, appCode);

        // 移动端任务卡展示逻辑 任务卡类型
        apiRequest.getPatternTaskInfo().setUiBotDesignerConfig(uiBotDesignerConfig);
        // true表示是平台页面(受设计器配置影响)，false表示是给应用定制开发的页面（不受设计器配置影响）
        if (Boolean.TRUE.equals(systemDefault)) {
            if (uiBotDesignerConfig.getPageInitialized()) {
                // 设计过APP界面
                if (uiBotDesignerConfig.isCustomize()) {
                    // 设计过APP界面，并且APP的isCustomize=true
                    apiRequest.getPatternTaskInfo().setCustomPatternTaskType(CustomPatternTaskTypeEnum.EXIST_MOBILE_PAGE_AND_ISCUSTOMIZE_TRUE);
                } else {
                    // 设计过APP界面，并且APP的isCustomize=false
                    apiRequest.getPatternTaskInfo().setCustomPatternTaskType(CustomPatternTaskTypeEnum.EXIST_MOBILE_PAGE_AND_ISCUSTOMIZE_FALSE);
                }
            } else {
                // 未设计过APP界面
                if (uiBotDesignerConfig.isPcCustomize()) {
                    // 未设计过APP界面，并且PC的isCustomize=true
                    apiRequest.getPatternTaskInfo().setCustomPatternTaskType(CustomPatternTaskTypeEnum.NOT_EXIST_MOBILE_PAGE_AND_PC_ISCUSTOMIZE_TRUE);
                } else {
                    // 未设计过APP界面，并且PC的isCustomize=false
                    apiRequest.getPatternTaskInfo().setCustomPatternTaskType(CustomPatternTaskTypeEnum.NOT_EXIST_MOBILE_PAGE_AND_PC_ISCUSTOMIZE_FALSE);
                }
            }
        }
        /**
         *  wjw FIXME: 2024/1/10
         *  数据录入模型驱动-流程设计生成的任务卡，需要根据绑定的表单信息下的基础数据录入的code，来判断是否存在移动界面设计
         */
        String type = MobileDesignerConfigTypeEnum.TASK.getValue();
        String tempActivityId = apiRequest.getExtraParameter().getTmActivityId();
        String bindFormFormCode = uiBotDesignerConfig.getBindFormFormCode();
        if (StringUtils.hasLength(bindFormFormCode)) {
            tempActivityId = bindFormFormCode;
            type = MobileDesignerConfigTypeEnum.DATA_ENTRY.getValue();
        }
        appCode = uiBotDesignerConfig.getAppCode();
        /**
         * wjw FIXME: 2023/8/23
         * 移动运行态判断 任务卡类型 临时处理方案，因移动设计页面配置存储在移动MongoDB中。
         * 后面统一从KM任务定义中判断是否存在移动layout节点，去除MongoDB的判断
         */
        if ((StringUtils.hasLength(appCode)
                && BooleanUtils.isTrue(activityFilterService.isExsitTmActivityIdOfMobileDesignerConfig(appCode, tempActivityId, type, apiRequest.getTenantId(), tenantVersion))
        ) || checkNaNa(apiRequest)
            // wjw TODO: 2023/11/24 去除KM中是否存在layout节点控制，只判断存在移动MongoDB中任务卡走V2。因历史任务卡中存在DSL节点layout，但是需要走V1版本渲染
            /*|| uiBotDesignerConfig.isExistMobileLayOut()*/) {
            // 新移动设计任务卡：存在mobileDesignerConfig表中 或 存在移动设计DSL节点layout， 设置走V2版策略处理
            apiRequest.getPatternTaskInfo().setCustomPatternTaskType(CustomPatternTaskTypeEnum.EXIST_MOBILE_PAGE_AND_ISCUSTOMIZE_FALSE);
            if (TaskStringUtil.isBizSmartAdjust(tempActivityId)) {
                if (!uiBotDesignerConfig.isExistMobileLayOut()) {
                    // 随心控(人工型：code+_task_manual;审核型：code+_task_approve) 任务，不存在layout 处理
                    apiRequest.getPatternTaskInfo().setCustomPatternTaskType(CustomPatternTaskTypeEnum.NOT_EXIST_MOBILE_PAGE_AND_PC_ISCUSTOMIZE_FALSE);
                }
            }
        } else {
            // 历史任务卡：不在mobileDesignerConfig表中 并且 不存在移动设计DSL节点layout
            if (CustomPatternTaskTypeEnum.EXIST_MOBILE_PAGE_AND_ISCUSTOMIZE_FALSE.equals(apiRequest.getPatternTaskInfo().getCustomPatternTaskType())) {
                // 如果历史任务卡存在移动设计页面配置，则需要根据PC的配置走相应逻辑
                if (apiRequest.getPatternTaskInfo().getUiBotDesignerConfig().isPcCustomize()) {
                    apiRequest.getPatternTaskInfo().setCustomPatternTaskType(CustomPatternTaskTypeEnum.NOT_EXIST_MOBILE_PAGE_AND_PC_ISCUSTOMIZE_TRUE);
                } else {
                    apiRequest.getPatternTaskInfo().setCustomPatternTaskType(CustomPatternTaskTypeEnum.DEFAULT);
                }
            }
        }
        // wjw TODO: 2024/8/21 临时处理，后续V1任务卡全部转V2后，移除此判断
        TestV1ToV2Util.handleV1ToV2(apiRequest, tempActivityId);
        return uiBotDesignerConfig;
    }

    private Boolean checkNaNa(ApiRequest apiRequest) {
        return SsoSourceTypeEnum.NANA_MESSAGE.getSource().equals(apiRequest.getSourceType());
    }

    /**
     * 获取任务详情的设计器配置
     *
     * @param locale
     * @param iamUserToken
     * @param tenantId
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    private UiBotDesignerConfig getTaskDetailUiBotDesignerConfig(ApiRequest apiRequest, String locale, String iamUserToken, String tenantId,
                                                                 String tmTaskId, String tmActivityId, String tenantVersion, String appCode) {
        UiBotDesignerConfig result = new UiBotDesignerConfig();

        // 准备并发调用
        CountDownLatch countDownLatch = new CountDownLatch(4);
        List<Future<?>> futureList = new ArrayList<>();

        // 调用知识图谱的API，获取任务详情页面定义和数据源配置
        CompletableFuture<Void> mobileCompletableFuture = CompletableFuture.runAsync(() -> {
            Map<String, String> uriVariables = new HashMap<>();
            uriVariables.put("taskId", tmTaskId);
            uriVariables.put("activityId", tmActivityId);
            uriVariables.put("pageCode", UiBotDesignerService.PAGECODE_TASK_DETAIL);
            DigiwinAthenaApiRequest athenaApiRequest =
                    DigiwinAthenaApiRequest.builder()
                            .requestHead(new AthenaBasicHttpRequestHead("", locale, iamUserToken, tenantId))
                            .uriVariables(uriVariables)
                            .build();
            KnowledgeMapsActivityDefinition rawActivityDefinition =
                    this.digiwinKnowledgeMapsProxyService.getActivityDefinition(athenaApiRequest);
            if (null != rawActivityDefinition) {
                // 保存原始响应的任务定义
                result.setRawTaskActivityDefinition(rawActivityDefinition);
                // 初始化页面配置
                result.initPage();
                // 初始化提取页面与数据源
                result.initDataSource();
            }
        }, this.threadPoolTaskExecutor);

        if ((StringUtils.hasLength(appCode)
                && BooleanUtils.isTrue(activityFilterService.isExsitTmActivityIdOfMobileDesignerConfig(appCode, tmActivityId, MobileDesignerConfigTypeEnum.TASK.getValue(), tenantId, tenantVersion))
        ) || checkNaNa(apiRequest)) {
            // 优化处理：走V2版策略处理时，不调用PC接口和后续规则、tag ---- 模型驱动生成的任务卡不会走此逻辑
            mobileCompletableFuture.join();
            return result;
        }
        // 调用知识图谱的API，获取任务详情页面定义和数据源配置
        CompletableFuture<Void> pcCompletableFuture = CompletableFuture.runAsync(() -> {
            Map<String, String> uriVariables = new HashMap<>();
            uriVariables.put("taskId", tmTaskId);
            uriVariables.put("activityId", tmActivityId);
            uriVariables.put("pageCode", UiBotDesignerService.PC_PAGECODE_TASK_DETAIL);
            DigiwinAthenaApiRequest athenaApiRequest =
                    DigiwinAthenaApiRequest.builder()
                            .requestHead(new AthenaBasicHttpRequestHead("", locale, iamUserToken, tenantId))
                            .uriVariables(uriVariables)
                            .build();
            KnowledgeMapsActivityDefinition rawActivityDefinition =
                    this.digiwinKnowledgeMapsProxyService.getActivityDefinition(athenaApiRequest);
            if (null != rawActivityDefinition) {
                // 保存原始响应的任务定义
                result.setPcRawTaskActivityDefinition(rawActivityDefinition);
                // 初始化页面配置
                result.initPcPage();
            }
        }, this.threadPoolTaskExecutor);
        CompletableFuture.allOf(mobileCompletableFuture, pcCompletableFuture).join();

        // 如果页面定义初始化未成功，表示未从设计器中拿到页面配置，则直接结束
        if (Boolean.FALSE.equals(result.getPageInitialized())) {
            return result;
        }

        // resourceId即为actionId
        String resourceId = Optional.ofNullable(result.getDataSource())
                .map(UiBotDesignerDataSource::getActionId).orElse("");
        CompletableFuture.runAsync(() -> {
                    // 调用知识图谱的API，获取数据源对应行动（Action）的元数据
                    DigiwinAthenaApiRequest athenaApiRequest =
                            DigiwinAthenaApiRequest.builder()
                                    .requestHead(new AthenaBasicHttpRequestHead("", locale, iamUserToken, tenantId))
                                    .uriVariables("actionId", resourceId)
                                    .build();
                    KnowledgeMapsActionMetadataQueryResult rawMetadataQueryResult =
                            this.digiwinKnowledgeMapsProxyService.actionMetadata(athenaApiRequest);
                    if (null != rawMetadataQueryResult) {
                        // 保存原始响应的元数据
                        result.setRawActionMetadata(rawMetadataQueryResult);
                        // 进行初始化动作
                        result.initActionResponseMetadata();
                    }
                }, this.threadPoolTaskExecutor)
                .thenRunAsync(() -> {
                    // 调用知识图谱的API，获取任务详情字段与tag配置
                    DigiwinAthenaApiRequest athenaApiRequest = this.buildRequestForGetTagQueryResult(
                            locale, iamUserToken, tenantId, tmTaskId, tmActivityId,
                            resourceId, UiBotDesignerService.TAG_TYPE_TASK);
                    List<KnowledgeMapsTagQueryResult> rawTagQueryResults =
                            this.digiwinKnowledgeMapsProxyService.getTagQueryResult(athenaApiRequest);
                    if (null != rawTagQueryResults) {
                        // 保存原始响应的字段Tag
                        result.setRawFieldTags(rawTagQueryResults);
                        // 进行初始化动作
                        result.initFieldTag();
                    }
                }, this.threadPoolTaskExecutor)
                .thenRunAsync(() -> {
                    // 调用知识图谱的API，获取任务规则配置
                    Map<String, Object> map = new HashMap<>();
                    map.put("include", Lists.newArrayList("contentMobile"));
                    DigiwinAthenaApiRequest athenaApiRequest =
                            DigiwinAthenaApiRequest.builder()
                                    .requestHead(new AthenaBasicHttpRequestHead("", locale, iamUserToken, tenantId))
                                    .bodyPayloadMap("activityId", tmActivityId)
                                    .bodyPayloadMap("count", -1)
                                    .bodyPayloadMap("features", map)
                                    .build();
                    KnowledgeMapsRuleQueryResult ruleQueryResult =
                            this.digiwinKnowledgeMapsProxyService.getRuleQueryResult(athenaApiRequest);
                    if (null != ruleQueryResult) {
                        // 保存原始响应的字段规则
                        result.setRawRule(ruleQueryResult);
                        // 进行初始化动作
                    }
                }, this.threadPoolTaskExecutor)
                .whenComplete((a, throwable) -> {
                    // 进行其他初始化配置
                    result.initDesignerConfig();
                    log.debug("DesignerConfig inited!!!");
                }).join();

        return result;
    }

    /**
     * 获取项目详情的设计器配置
     *
     * @param locale
     * @param iamUserToken
     * @param tenantId
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    private UiBotDesignerConfig getProjectDetailUiBotDesignerConfig(String locale, String iamUserToken, String tenantId,
                                                                    String tmTaskId, String tmActivityId) {
        return null;
    }


    /**
     * 为getTagQueryResult测试方法构造请求参数
     *
     * @param locale
     * @param iamUserToken
     * @param tenantId
     * @param tmTaskId
     * @param tmActivityId
     * @param resourceId   设计器保存的ESP服务名称
     * @param detailType   详情类型，分为任务详情或项目详情。小写字符串。可选值：peformer，表示任务；charge，表示项目
     * @return
     */
    private DigiwinAthenaApiRequest buildRequestForGetTagQueryResult(
            String locale, String iamUserToken, String tenantId,
            String tmTaskId, String tmActivityId,
            String resourceId, String detailType) {
        List<String> relationIds = new ArrayList<>(2);
        relationIds.add(detailType);
        relationIds.add(tmActivityId);

        Map<String, String> features = new HashMap<>(1);
        features.put("deviceType", "mobile");

        return DigiwinAthenaApiRequest.builder()
                .requestHead(new AthenaBasicHttpRequestHead("", locale, iamUserToken, tenantId))
                .bodyPayloadMap("resourceId", resourceId)
                .bodyPayloadMap("relationIds", relationIds)
                .bodyPayloadMap("features", features)
                .build();
    }

}
