package com.digiwin.mobile.mobileuibot.designer.uibot;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.core.component.MobileLayoutTypeEnum;
import com.digiwin.mobile.mobileuibot.designer.DesignerConfig;
import com.digiwin.mobile.mobileuibot.designer.uibot.page.PageDataState;
import com.digiwin.mobile.mobileuibot.designer.uibot.page.UiBotDesignerPage;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.model.response.*;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.model.response.metadata.KnowledgeMapsActionMetadataElement;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.model.response.metadata.KnowledgeMapsActionMetadataQueryResult;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.model.response.metadata.KnowledgeMapsActionMetadataResponse;
import com.digiwin.mobile.mobileuibot.task.common.TaskStringUtil;
import com.google.common.collect.Lists;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;

/**
 * <p>功能描述：UiBot设计器配置类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: UiBotDesignerConfig
 * @Author: zaregoto
 * @Date: 2023/2/16 10:47
 */
@Slf4j
@Data
@NoArgsConstructor
public class UiBotDesignerConfig extends DesignerConfig {
    private static final long serialVersionUID = -8115122632007551011L;
    /**
     * 当前配置是否已初始化过.
     * 初始化的含义：从外部原始配置，提取并转换为移动定义的配置数据
     */
    private Boolean initialized = false;
    /**
     * 移动端布局类型：单层or多层，分别对应不同的app端单层容器组件or多层容器组件，会影响后续生成的渲染DSL
     */
    private MobileLayoutTypeEnum mobileLayoutType;

    /**
     * 从原始配置数据中转换得到的移动定义的配置数据
     */
    // 页面配置（包含不同数据状态等基础配置信息）
    private UiBotDesignerPage page;
    private Boolean pageInitialized = false;

    // PC 页面配置（包含不同数据状态等基础配置信息）
    private UiBotDesignerPage pcPage;
    private Boolean pcPageInitialized = false;

    // 页面实际被使用的数据源
    private UiBotDesignerDataSource dataSource;
    // 页面已配置的多个数据源的映射关系（使用不同的key进行区分，key为用户配置的）
    private UiBotDesignerDataSourceMap dataSourceMap;
    private Boolean dataSourceInitialized = false;

    // 页面配置的字段标签
    private List<UiBotDesignerFieldTagConfig> tagConfigs;
    private Boolean tagInitialized = false;

    // 页面配置的业务规则
    private List<UiBotDesignerRuleConfig> ruleConfigs;
    private Boolean ruleInitialized = false;

    // 页面配置的数据源对应Action的响应参数元数据
    private UiBotDesignerActionMetadataElement actionResponseMetadata;
    private Boolean actionResponseMetadataInitialized = false;

    /**
     * 从鼎捷雅典娜知识图中获取的界面原始配置数据
     */
    // KM任务定义原始配置数据
    private KnowledgeMapsActivityDefinition rawTaskActivityDefinition;
    private KnowledgeMapsActivityDefinition pcRawTaskActivityDefinition;

    // 字段标签
    private List<KnowledgeMapsTagQueryResult> rawFieldTags;

    // 字段业务规则
    private KnowledgeMapsRuleQueryResult rawRule;

    // 字段元数据
    private KnowledgeMapsActionMetadataQueryResult rawActionMetadata;


    /**
     * 设计器配置中使用的字段分割符号
     */
    public static final String DESIGNER_FIELD_SPLITTER = ".";

    /**
     * 使用外部原始配置数据初始化当前设计时配置对象
     */
    public void initDesignerConfig() {
        if (null == this.page) {
            this.initPage();
        }
        if (null == this.pcPage) {
            this.initPcPage();
        }
        if (null == this.dataSource) {
            this.initDataSource();
        }
        if (null == this.tagConfigs) {
            this.initFieldTag();
        }
        if (null == this.ruleConfigs) {
            this.initRule();
        }
        if (null == this.actionResponseMetadata) {
            this.initActionResponseMetadata();
        }
        this.initialized = this.pageInitialized && this.dataSourceInitialized
                && this.tagInitialized && this.ruleInitialized && this.actionResponseMetadataInitialized;
    }

    /**
     * 判断是否存在移动设计DSL节点 layout
     *
     * @return
     */
    public boolean isExistMobileLayOut() {
        if (Boolean.FALSE.equals(this.pageInitialized)) {
            return false;
        }
        List<PageDataState> dataStates = Optional.ofNullable(this.page).map(UiBotDesignerPage::getDataStates).orElse(Lists.newArrayList());
        if (CollectionUtils.isEmpty(dataStates)) {
            return false;
        }
        for (PageDataState dataState : dataStates) {
            if (!CollectionUtils.isEmpty(dataState.getLayout())) {
                // 表示存在移动设计DSL节点 layout
                return true;
            }
        }
        return false;
    }

    /**
     * 获取设计器流程设定的表单信息下的基础数据录入的code
     * 存在条件：tmPattern=BUSINESS、tmCategory=DATA_ENTRY_DRIVEN
     *
     * @return
     */
    public String getBindFormFormCode() {
        if (this.rawTaskActivityDefinition == null) {
            return null;
        }
        String pattern = this.rawTaskActivityDefinition.getString("pattern");
        String category = this.rawTaskActivityDefinition.getString("category");
        if (!StringUtils.hasLength(pattern) || !StringUtils.hasLength(category)) {
            return null;
        }
        if (!TaskStringUtil.isBizDataEntryDriven(pattern, category)) {
            return null;
        }
        JSONObject bindForm = this.rawTaskActivityDefinition.getJSONObject("bindForm");
        if (CollectionUtils.isEmpty(bindForm)) {
            return null;
        }
        return bindForm.getString("formCode");
    }

    public String getAppCode() {
        if (this.rawTaskActivityDefinition == null) {
            return null;
        }
        return this.rawTaskActivityDefinition.getString("application");
    }

    /**
     * 判断当前界面是否是定制界面
     *
     * @return
     */
    public Boolean isCustomize() {
        if (Boolean.FALSE.equals(this.pageInitialized)) {
//            throw new IllegalArgumentException("initDesignerConfig method must be called first, or check [page] property of UiBotDesignerPage has non-null value!");
            return false;
        }
        return Optional.ofNullable(this.page).map(UiBotDesignerPage::getIsCustomize).orElse(false);
    }

    /**
     * 判断 PC 当前界面是否是定制界面
     *
     * @return
     */
    public Boolean isPcCustomize() {
        if (Boolean.FALSE.equals(this.pcPageInitialized)) {
            return false;
        }
        return Optional.ofNullable(this.pcPage).map(UiBotDesignerPage::getIsCustomize).orElse(false);
    }

    private void selectMobileLayoutType() {
        // TODO 判断移动端的画面布局类型
    }

    /**
     * 初始化页面配置。将从API获得的原始配置，保存并转化到当前类
     */
    public void initPage() {
        if (null == this.rawTaskActivityDefinition) {
            return;
        }
        JSONObject rawPage = this.rawTaskActivityDefinition.getJSONObject("pages");

        if (null == rawPage) {
            log.error("Athena-KM returned NULL pages, PLZ CHECK THEM. No-More process will be done !");
            return;
        }

        UiBotDesignerPage uiBotDesignerPage = JsonUtil.objectToJavaObject(rawPage, UiBotDesignerPage.class);
        uiBotDesignerPage.initDesignerPageConfig();

        this.page = uiBotDesignerPage;
        this.pageInitialized = true;
    }

    /**
     * 初始化 PC 页面配置。将从API获得的原始配置，保存并转化到当前类
     */
    public void initPcPage() {
        if (null == this.pcRawTaskActivityDefinition) {
            return;
        }
        JSONObject rawPage = this.pcRawTaskActivityDefinition.getJSONObject("pages");

        if (null == rawPage) {
            log.error("Athena-KM returned NULL pages, PLZ CHECK THEM. No-More process will be done !");
            return;
        }

        UiBotDesignerPage uiBotDesignerPage = JsonUtil.objectToJavaObject(rawPage, UiBotDesignerPage.class);
        uiBotDesignerPage.initDesignerPageConfig();

        this.pcPage = uiBotDesignerPage;
        this.pcPageInitialized = true;
    }

    /**
     * 初始化数据源配置。将从API获得的原始配置，保存并转化到当前类
     */
    public void initDataSource() {
        if (null == this.rawTaskActivityDefinition) {
            return;
        }
        JSONObject rawDataSourceMap = this.rawTaskActivityDefinition.getJSONObject("dataSources");

        if (null == rawDataSourceMap) {
            log.error("Athena-KM returned NULL/EMPTY dataSources, PLZ CHECK THEM!");
            return;
        }

        UiBotDesignerDataSourceMap uiBotDesignerDataSourceMap = JsonUtil.objectToJavaObject(rawDataSourceMap, UiBotDesignerDataSourceMap.class);
        this.dataSourceMap = uiBotDesignerDataSourceMap;
        // 因数据源在任务详情的多个数据状态中也只会有一份，所以这里直接取第1个数据源的信息即可
        Set<Map.Entry<String, UiBotDesignerDataSource>> dataSourceEntry = uiBotDesignerDataSourceMap.entrySet();
        UiBotDesignerDataSource targetDataSource = null;
        String dataSourceDesignId = "";
        int i = 0;
        for (Map.Entry<String, UiBotDesignerDataSource> entry : dataSourceEntry) {
            if (i == 1) {
                break;
            }
            dataSourceDesignId = entry.getKey();
            targetDataSource = entry.getValue();
            i++;
        }
        if (null != targetDataSource) {
            targetDataSource.setDataSourceKeyInDesigner(dataSourceDesignId);
            this.dataSource = targetDataSource;
            this.dataSourceInitialized = true;
        }
    }

    /**
     * 初始化字段Tag配置。将从API获得的原始配置，保存并转化到当前类
     */
    public void initFieldTag() {
        if (null == this.rawFieldTags) {
            return;
        }
        if (null == this.tagConfigs) {
            this.tagConfigs = new ArrayList<>(rawFieldTags.size());
        }

        String actionId = this.dataSource.getActionId();
        String serviceName = this.dataSource.getServiceName();
        String key = this.dataSource.getDataSourceKeyInDesigner();
        // 待处理的字符串格式：esp_pdemo.user.get.response.user_data.name
        String prefix = actionId + DESIGNER_FIELD_SPLITTER + "response" + DESIGNER_FIELD_SPLITTER + key + DESIGNER_FIELD_SPLITTER;
        for (KnowledgeMapsTagQueryResult rawFieldTag : this.rawFieldTags) {
            String fieldId = Optional.ofNullable(rawFieldTag.getResourceContent())
                    .map(KnowledgeMapsTagQueryResourceContent::getResponse)
                    .orElse("").substring(prefix.length());
            UiBotDesignerFieldTagConfig fieldTagConfig = new UiBotDesignerFieldTagConfig();
            fieldTagConfig.setEspFieldId(fieldId);
            fieldTagConfig.setEspServiceName(serviceName);
            fieldTagConfig.setTags(rawFieldTag.getTags());

            this.tagConfigs.add(fieldTagConfig);
        }
        this.tagInitialized = true;
    }

    /**
     * 初始化规则配置。将从API获得的原始配置，保存并转化到当前类
     */
    public void initRule() {
        if (null == this.rawRule) {
            return;
        }

        if (CollectionUtils.isEmpty(this.rawRule.getRules())) {
            return;
        }

        if (null == this.ruleConfigs) {
            this.ruleConfigs = new ArrayList<>(this.rawRule.getRules().size());
        }

        for (KnowledgeMapsRuleContent rule : this.rawRule.getRules()) {
            UiBotDesignerRuleConfig ruleConfig = new UiBotDesignerRuleConfig();
            ruleConfig.setContentMobile(rule.getContentMobile());
            // TODO 规则的部分暂先这样写
            this.ruleConfigs.add(ruleConfig);
        }
        this.ruleInitialized = true;
    }

    public void initActionResponseMetadata() {
        if (null == this.rawActionMetadata) {
            return;
        }
        Optional<KnowledgeMapsActionMetadataElement> optResponseData
                = Optional.ofNullable(this.rawActionMetadata.getResponse())
                .map(KnowledgeMapsActionMetadataResponse::getData);
        if (optResponseData.isPresent()) {
            this.actionResponseMetadata = JsonUtil.objectToJavaObject(
                    optResponseData.get(), UiBotDesignerActionMetadataElement.class);
            this.actionResponseMetadataInitialized = true;
        }
    }

    /**
     * <p>根据响应字段元数据信息，获取目标字段元数据信息（如order_id字段的元数据信息）</p>
     * <p>如果没有响应字段元数据信息，则直接返回NULL</p>
     * <p>如果没有指定要查询的字段，则直接返回NULL</p>
     * <p>如果parentFieldId和targetFieldId都匹配到，则返回targetFieldId匹配元数据；反之则返回NULL</p>
     *
     * @param parentFieldId 响应参数中目标字段的父字段。可能会使用半角的『.』表示多层。如：car_data，或者car_data.view1
     * @param targetFieldId 响应参数中目标字段。可能会使用半角的『.』表示多层。如：car_number，或者view1.light_id
     * @return
     */
    public UiBotDesignerActionMetadataElement getFieldInResponseMetadata(final String parentFieldId, final String targetFieldId) {
        if (Boolean.FALSE.equals(this.actionResponseMetadataInitialized) ||
                Objects.isNull(this.actionResponseMetadata)) {
            log.error("No Field Metadata From Designer Database...");
            return null;
        }
        if (!StringUtils.hasLength(parentFieldId) || !StringUtils.hasLength(targetFieldId)) {
            log.error("parentFieldId and targetFieldId must has length...");
            return null;
        }
        String newParentFieldId = parentFieldId;
        String newTargetFieldId = targetFieldId;
        if (parentFieldId.contains(DESIGNER_FIELD_SPLITTER)) {
            String[] spiltedParentFieldAry = parentFieldId.split("\\" + DESIGNER_FIELD_SPLITTER, 2);
            newParentFieldId = spiltedParentFieldAry[0];
            newTargetFieldId = String.join(DESIGNER_FIELD_SPLITTER, spiltedParentFieldAry[1], targetFieldId);
            log.debug("parentFieldId: {} is not simple, splitting parentFieldId and getField recursively...newParentFieldId is: {}, newTargetFieldId is: {}", parentFieldId, newParentFieldId, newTargetFieldId);
            return this.getFieldInResponseMetadata(newParentFieldId, newTargetFieldId);
        }
        if (Objects.equals(newParentFieldId, this.actionResponseMetadata.getDataName())) {
            UiBotDesignerActionMetadataElement parentFieldElement = this.actionResponseMetadata;
            return this.searchTargetElement(parentFieldElement, newTargetFieldId);
        } else {
            log.error("parentFieldId: {} is NOT matched with Metadata...", parentFieldId);
            return null;
        }

    }

    /**
     * 递归获取下层字段元数据
     *
     * @param parentFieldElement
     * @param targetFieldId
     * @return
     */
    private UiBotDesignerActionMetadataElement searchTargetElement(final UiBotDesignerActionMetadataElement parentFieldElement, final String targetFieldId) {
        if (Objects.isNull(parentFieldElement)) {
            log.error("parentFieldElement is NULL, searchTargetElement STOPPED...");
            return null;
        }
        String newParentFieldId;
        String newTargetFieldId;
        if (targetFieldId.contains(DESIGNER_FIELD_SPLITTER)) {
            String[] splitedTargetFieldAry = targetFieldId.split("\\.", 2);
            newParentFieldId = splitedTargetFieldAry[0];
            newTargetFieldId = splitedTargetFieldAry[1];
            UiBotDesignerActionMetadataElement newParentFieldElement = parentFieldElement.getField().stream()
                    .filter(e -> Objects.equals(e.getDataName(), newParentFieldId))
                    .findFirst()
                    .orElse(null);
            return this.searchTargetElement(newParentFieldElement, newTargetFieldId);
        } else {
            return parentFieldElement.getField().stream()
                    .filter(e -> Objects.equals(e.getDataName(), targetFieldId))
                    .findFirst()
                    .orElse(null);
        }
    }

}