package com.digiwin.mobile.mobileuibot.designer.uibot.service;

import com.digiwin.mobile.mobileuibot.api.*;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponent;
import com.digiwin.mobile.mobileuibot.core.component.BaseMobileComponentWrapper;
import com.digiwin.mobile.mobileuibot.core.component.list.editcardlist.EditCardList;
import com.digiwin.mobile.mobileuibot.core.component.list.ztbviewlist.ZtbViewList;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.proxy.atdm.model.DigiwinAtdmQueryPageDataReq;
import com.digiwin.mobile.mobileuibot.proxy.atdm.model.DigiwinAtdmQueryPageDataResult;
import com.digiwin.mobile.mobileuibot.proxy.atdm.service.DigiwinAtdmProxyService;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotActionSearchInfoMapping;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotDataSourceSetDTO;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotExecuteContext;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotPageInfo;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>功能描述: UiBot组件数据查询服务</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: UiBotComponentDataQueryService.java
 * @Author: xujinc
 * @Date: created at 2024/12/30 13:32
 */

@Slf4j
@Service
public class UiBotComponentDataQueryService {

    @Autowired
    private LocaleService localeService;

    @Resource
    private UiBotDesignerRenderService uiBotDesignerRenderService;

    @Autowired
    private DigiwinAtdmProxyService digiwinAtdmProxyService;

    public ApiResponse<ComponentDataQueryResult> componentDataQuery(ApiRequestComponentDataQuery query) {
        // 获取查询结果
        DigiwinAtdmQueryPageDataResult queryData = this.getQueryData(query);
        if (queryData == null) {
            return ApiResponse.buildError(localeService.getLanguageValue(query.getLocale(), "查询数据失败"));
        }
        // 获取当前操作rawData
        ApiRawData rawData = this.getCurrentRawData(query);
        if (rawData == null) {
            return ApiResponse.buildError(localeService.getLanguageValue(query.getLocale(), "当前操作rawData为空"));
        }
        // 渲染组件
        ApiResponse<BaseMobileComponentWrapper<BaseMobileComponent>> componentResponse = this.renderComponent(query, rawData, queryData.getPageData());
        if (componentResponse.isError()) {
            return ApiResponse.buildError(componentResponse.getMsg());
        }
        BaseMobileComponentWrapper<BaseMobileComponent> componentWrapper = componentResponse.getData();
        if (componentWrapper == null) {
            return ApiResponse.buildError(localeService.getLanguageValue(query.getLocale(), "组件渲染失败"));
        }

        // 构建返回数据
        ComponentDataQueryResult result = new ComponentDataQueryResult();
        if (componentWrapper.getData() instanceof ZtbViewList) {
            ZtbViewList ztbViewList = (ZtbViewList) componentWrapper.getData();
            result.setData(CollectionUtils.isEmpty(ztbViewList.getData()) ? new ArrayList<>() : new ArrayList<>(ztbViewList.getData()));
        } else if (componentWrapper.getData() instanceof EditCardList) {
            EditCardList editCardList = (EditCardList) componentWrapper.getData();
            result.setData(CollectionUtils.isEmpty(editCardList.getContentGroup()) ? new ArrayList<>() : new ArrayList<>(editCardList.getContentGroup()));
        } else {
            // 其他
            return ApiResponse.buildError(localeService.getLanguageValue(query.getLocale(), "不支持的组件类型"));
        }
        PcUiBotPageInfo pageInfo = queryData.getPageInfo();
        pageInfo.setPageCountSize(queryData.getPageCountSize());
        result.setPageInfo(pageInfo);
        return ApiResponse.buildOK().setData(result);
    }

    private DigiwinAtdmQueryPageDataResult getQueryData(ApiRequestComponentDataQuery query) {
        // 获取当前操作rawData
        ApiRawData rawData = this.getCurrentRawData(query);
        if (rawData == null) {
            return null;
        }
        PcUiBotDataSourceSetDTO dataSourceSet = JsonUtil.objectToJavaObject(rawData.get("dataSourceSet"), PcUiBotDataSourceSetDTO.class);
        PcUiBotExecuteContext executeContext = JsonUtil.objectToJavaObject(rawData.get("executeContext"), PcUiBotExecuteContext.class);
        // 查询数据
        DigiwinAtdmQueryPageDataReq bodyData = new DigiwinAtdmQueryPageDataReq()
                .setDataSourceSet(dataSourceSet)
                .setExecuteContext(executeContext)
                .setPageInfo(this.getPageInfo(query))
                .setParameter(new HashMap<>())
                .setSearchInfo(this.buildSearchInfo(query))
                .setSortInfo(new ArrayList<>());
        return digiwinAtdmProxyService.queryPageDataByDataset(query.getLocale(), query.getIamUserToken(), query.getTenantId(), bodyData);
    }

    private ApiRawData getCurrentRawData(ApiRequestComponentDataQuery query) {
        ApiRawData rawData = null;
        Integer queryType = query.getQueryType();
        if (ApiRequestComponentDataQuery.QueryType.PAGE.getValue().equals(queryType)) {
            // 分页
            rawData = query.getPaging().getRawData();
        } else if (ApiRequestComponentDataQuery.QueryType.SEARCH.getValue().equals(queryType)) {
            // 搜索
            rawData = query.getSearch().getRawData();
        } else if (ApiRequestComponentDataQuery.QueryType.FILTER.getValue().equals(queryType)) {
            // 过滤
            rawData = query.getFilter().getRawData();
        } else if (ApiRequestComponentDataQuery.QueryType.SORT.getValue().equals(queryType)) {
            // 排序
            ApiRequestComponentDataQuery.Sort sort = query.getSort();
        } else {
            // 其他
        }
        return rawData;
    }

    private PcUiBotPageInfo getPageInfo(ApiRequestComponentDataQuery query) {
        PcUiBotPageInfo defaltPageInfo = new PcUiBotPageInfo();
        defaltPageInfo.setPageNo(1);
        defaltPageInfo.setPageSize(50);

        ApiRequestComponentDataQuery.Paging paging = query.getPaging();
        if (paging == null) {
            // 没有配置分页信息，则分页信息。因为/api/atdm/v1/data/query/page/by/dataset接口pageInfo不可为空
            return defaltPageInfo;
        }
        PcUiBotPageInfo pageInfo = paging.getPageInfo();
        if (pageInfo == null) {
            // 没有配置分页信息，则分页信息。因为/api/atdm/v1/data/query/page/by/dataset接口pageInfo不可为空
            return defaltPageInfo;
        }
        return pageInfo;
    }

    private List<Map<String, Object>> buildSearchInfo(ApiRequestComponentDataQuery query) {
        List<Map<String, Object>> searchInfoResult = new ArrayList<>();

        // 搜索条件构建
        ApiRequestComponentDataQuery.Search search = query.getSearch();
        if (search != null) {
            List<PcUiBotActionSearchInfoMapping> searchInfo = Optional.ofNullable(search.getSearchInfo()).orElse(new ArrayList<>()).stream()
                    .filter(e -> StringUtils.hasLength(e.getSearchOperator()))
                    .collect(Collectors.toList());
            if (StringUtils.hasLength(search.getCondition()) && !CollectionUtils.isEmpty(searchInfo)) {
                // 构建搜索配置
                List<Map<String, Object>> searchList = new ArrayList<>();
                boolean multi = searchInfo.size() > 1;
                for (int i = 0; i < searchInfo.size(); i++) {
                    Map<String, Object> searchMap = new HashMap<>();
                    if (multi) {
                        if (i == 0) {
                            searchMap.put("bracket", "(");
                            searchMap.put("logic", "OR");
                        } else if (i == searchInfo.size() - 1) {
                            searchMap.put("bracket", ")");
                        } else {
                            searchMap.put("logic", "OR");
                        }
                    }
                    searchMap.put("order", i + 1);
                    searchMap.put("search_field", searchInfo.get(i).getSearchField());
                    searchMap.put("search_operator", searchInfo.get(i).getSearchOperator());
                    searchMap.put("search_value", Lists.newArrayList(search.getCondition()));
                    searchList.add(searchMap);
                }
                searchInfoResult.addAll(searchList);
            }
        }

        // 筛选条件构建
        // wjw TODO: 2025/4/1

        return searchInfoResult;
    }

    private ApiResponse<BaseMobileComponentWrapper<BaseMobileComponent>> renderComponent(ApiRequestComponentDataQuery query, ApiRawData rawData, Map<String, Object> pageData) {
        // 渲染组件时使用（/mobile/v1/proxy/getCardListDslData）
        ComponentDslDataRequest dslDataRequest = new ComponentDslDataRequest();
        dslDataRequest.setExecuteContext(rawData.get("executeContext"));
        dslDataRequest.setRawData(JsonUtil.objectToJavaObject(rawData.get("rawData"), ApiRawData.class));
        dslDataRequest.setPageId(rawData.getString("pageId"));
        dslDataRequest.setComponentType(rawData.getString("componentType"));
        dslDataRequest.setCmptUuid(rawData.getString("cmptUuid"));
        dslDataRequest.setData(pageData);
        dslDataRequest.setViewCode(rawData.getString("viewCode"));

        ApiRequestComponent apiRequestComponent = JsonUtil.objectToJavaObject(query, ApiRequestComponent.class);
        apiRequestComponent.setComponentData(dslDataRequest);
        return uiBotDesignerRenderService.buildComponentDsl(apiRequestComponent);
    }
}
