package com.digiwin.mobile.mobileuibot.designer.uibot.service;

import com.digiwin.mobile.mobileuibot.core.layout.doublepattern.bean.PcModuleEnum;
import com.digiwin.mobile.mobileuibot.designer.uibot.UiBotDesignerActionMetadataElement;
import com.digiwin.mobile.mobileuibot.designer.uibot.UiBotDesignerConfig;
import com.digiwin.mobile.mobileuibot.designer.uibot.UiBotDesignerFieldTagConfig;
import com.digiwin.mobile.mobileuibot.designer.uibot.page.PageFieldMetaData;
import com.digiwin.mobile.mobileuibot.designer.uibot.page.ScoredField;
import com.digiwin.mobile.mobileuibot.designer.uibot.page.ScoredFieldsSummary;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.model.KnowledgeMapsTag;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotTargetLayoutSearchResult;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumn;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumnDefinition;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>功能描述：UIBot设计器『字段范围与排序』配置之应用器实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: UiBotDesignerFieldAndOrderApplierServiceImpl
 * @Author: zaregoto
 * @Date: 2023/2/27 12:00
 */
@Slf4j
@Service("uiBotDesignerFieldAndOrderApplierService")
public class UiBotDesignerFieldAndOrderApplierServiceImpl implements UiBotDesignerFieldAndOrderApplierService {

    @Override
    public void applyFieldAndOrder(UiBotModel uiBotModel, UiBotDesignerConfig uiBotDesignerConfig) {
        // 从PC端UIBot数据中提取出的，待处理的layout
        UiBotTargetLayoutSearchResult layoutSearchResult = uiBotModel.searchPcTargetLayout();
        if (!layoutSearchResult.found()) {
            return;
        }

        // 获取排序后的字段ID
        List<String> orderedFieldIds = this.getOrderedFieldIds(uiBotDesignerConfig);

        // 将排序后的字段ID应用到原始PC UiBot中
        UiBotLayout pcDataLayout = layoutSearchResult.getTargetUiBotLayout();
        // 依据排序后的字段清单，调整字段内容，并设置默认为可编辑（后续会有『不可编辑』的tag来处理读写状态）
        if (pcDataLayout.isTypeOfTable()) {
            this.applyFieldAndOrderWithPcTable(uiBotDesignerConfig, uiBotModel, orderedFieldIds, pcDataLayout);
            uiBotModel.setAppliedDesignerFieldAndOrder(true);
        } else if (pcDataLayout.isTypeOfFormList()) {
            this.applyFieldAndOrderWithPcFormList(uiBotDesignerConfig, uiBotModel, orderedFieldIds, pcDataLayout);
            uiBotModel.setAppliedDesignerFieldAndOrder(true);
        } else {
            return;
        }
    }

    /**
     * 针对PC UIBot的表格布局调整字段与顺序
     *
     * @param uiBotDesignerConfig
     * @param uiBotModel
     * @param orderedFieldIds
     * @param pcDataLayout
     */
    private void applyFieldAndOrderWithPcTable(UiBotDesignerConfig uiBotDesignerConfig, UiBotModel uiBotModel, List<String> orderedFieldIds, UiBotLayout pcDataLayout) {
        List<UiBotTableColumn> tableColumns;
        String schema = pcDataLayout.getSchema();
        // 因为是TABLE类型的layout，所以数据一定是数组
        List<Map<String, Object>> bizDataList = (List<Map<String, Object>>) uiBotModel.getPageData().get(schema);
        Assert.notNull(bizDataList, "pagedata." + schema + " cannot be null");

        List<UiBotTableColumnDefinition> tableColumnDefinitionList = pcDataLayout.getColumnDefs();
        if (CollectionUtils.isEmpty(tableColumnDefinitionList)) {
            return;
        }
        // 分解出明细数据一行的字段列表
        tableColumns = UiBotTableColumnDefinition.decomposeTableColumnDefinitions(tableColumnDefinitionList);
        Map<String, UiBotTableColumn> tableColumnMap = tableColumns.stream()
                .collect(Collectors.toMap(UiBotTableColumn::getSchema, c -> c, (c1, c2) -> c1));

        int resultSize = orderedFieldIds.size();
        List<UiBotTableColumnDefinition> tableColumnDefsFromDesigner = new ArrayList<>(resultSize);
        for (String fieldId : orderedFieldIds) {
            UiBotTableColumn newUiBotTableColumn = tableColumnMap.get(fieldId);
            if (null == newUiBotTableColumn) {
                newUiBotTableColumn = new UiBotTableColumn();
                newUiBotTableColumn.setSchema(fieldId);
                UiBotDesignerActionMetadataElement metadataElement = uiBotDesignerConfig.getFieldInResponseMetadata(schema, fieldId);
                newUiBotTableColumn.setHeaderName(Optional.ofNullable(metadataElement)
                        .map(UiBotDesignerActionMetadataElement::getDescription).orElse(fieldId));
                newUiBotTableColumn.setDataType(Optional.ofNullable(metadataElement)
                        .map(UiBotDesignerActionMetadataElement::getDataType).orElse("string"));
                newUiBotTableColumn.setType(PcModuleEnum.INPUT.getValue());
//                continue;
            }
            newUiBotTableColumn.doSetCanEdit();

            // 重新包装成PC端UIBot的字段定义类，以便原有运行时任务生成的逻辑可以接上
            UiBotTableColumnDefinition tableColumnDefinition = new UiBotTableColumnDefinition();
            tableColumnDefinition.setSchema(fieldId);
            tableColumnDefinition.addUiBotTableColumn(newUiBotTableColumn);
            tableColumnDefsFromDesigner.add(tableColumnDefinition);
        }
        if (!CollectionUtils.isEmpty(tableColumnDefsFromDesigner)) {
            // 覆盖字段定义到原有PC端UIBot的数据
            pcDataLayout.setColumnDefs(tableColumnDefsFromDesigner);
        }
    }

    /**
     * 针对PC UIBot的表单布局调整字段与顺序
     *
     * @param uiBotDesignerConfig
     * @param uiBotModel
     * @param orderedFieldIds
     * @param pcDataLayout
     */
    private void applyFieldAndOrderWithPcFormList(UiBotDesignerConfig uiBotDesignerConfig, UiBotModel uiBotModel, List<String> orderedFieldIds, UiBotLayout pcDataLayout) {
        String schema = pcDataLayout.getSchema();
        // 因为是FORM_LIST类型的layout，所以数据一定是对象
        Map<String, Object> bizData = (Map<String, Object>) uiBotModel.getPageData().get(schema);
        Assert.notNull(bizData, "pagedata." + schema + " cannot be null");

        List<UiBotLayout> pcFieldLayoutList = pcDataLayout.decomposeFieldsWithPcFormList(Collections.emptyList());
        Map<String, UiBotLayout> pcFieldLayoutMap = pcFieldLayoutList.stream()
                .collect(Collectors.toMap(UiBotLayout::getSchema, c -> c, (c1, c2) -> c1));

        int resultSize = orderedFieldIds.size();
        List<UiBotLayout> fieldLayoutFromDesigner = new ArrayList<>(resultSize);
        for (String fieldId : orderedFieldIds) {
            UiBotLayout fieldLayout = pcFieldLayoutMap.get(fieldId);
            // 出现了移动端配置展示的字段，但pc端没有配置展示的字段
            if (null == fieldLayout) {
                fieldLayout = new UiBotLayout();
                fieldLayout.setSchema(fieldId);
                UiBotDesignerActionMetadataElement metadataElement = uiBotDesignerConfig.getFieldInResponseMetadata(schema, fieldId);
                fieldLayout.setHeaderName(Optional.ofNullable(metadataElement)
                        .map(UiBotDesignerActionMetadataElement::getDescription).orElse(fieldId));
                fieldLayout.setDataType(Optional.ofNullable(metadataElement)
                        .map(UiBotDesignerActionMetadataElement::getDataType).orElse("string"));
                fieldLayout.setType(PcModuleEnum.INPUT.getValue());
//                continue;
            }
            fieldLayout.doSetCanEdit();

            fieldLayoutFromDesigner.add(fieldLayout);
        }
        if (!CollectionUtils.isEmpty(fieldLayoutFromDesigner)) {
            // 覆盖界面layout到原有PC端UIBot的数据
            pcDataLayout.setGroup(fieldLayoutFromDesigner);
        }
    }

    /**
     * 根据设计器配置，获取排序过后的字段ID清单。靠前的优先展示。排序依据有两个，按优先级个别获取：
     * 1.依据知识图谱中字段库的评分；（使用开发平台设计器的新应用）
     * 2.依据知识图谱中配置的Order类型Tag；（未使用开发平台设计器的旧应用）
     *
     * @param uiBotDesignerConfig
     * @return
     */
    @NotNull
    private List<String> getOrderedFieldIds(UiBotDesignerConfig uiBotDesignerConfig) {
        List<String> orderedFieldIds = this.getOrderedFieldIdsByScoredWords(uiBotDesignerConfig);
        if (CollectionUtils.isEmpty(orderedFieldIds)) {
            orderedFieldIds = this.getOrderedFieldIdsByTag(uiBotDesignerConfig);
        }
        if (CollectionUtils.isEmpty(orderedFieldIds)) {
            orderedFieldIds = Collections.emptyList();
        }
        return orderedFieldIds;
    }

    /**
     * 根据词库打分情况，获得排序后的字段id清单
     *
     * @param uiBotDesignerConfig UiBot设计器配置
     * @return
     */
    private List<String> getOrderedFieldIdsByScoredWords(UiBotDesignerConfig uiBotDesignerConfig) {
        // 解析出配置的业务字段并按先后顺序排序
        Map<String, PageFieldMetaData> pageFieldMetaDataMap =
                uiBotDesignerConfig.getPage().getPageFieldMetadataMap();
        if (CollectionUtils.isEmpty(pageFieldMetaDataMap)) {
            return Collections.emptyList();
        }
        int i = 0;
        PageFieldMetaData fieldMetaData = null;
        for (Map.Entry<String, PageFieldMetaData> entry : pageFieldMetaDataMap.entrySet()) {
            /**
             * FIXME
             * 目前先取第一个数据状态页面的字段做排序。
             * 因设计器中任务的数据据状态可以有多个，对应多个页面定义；且现在又允许用户配置相互独立的数据状态，故当前会出现不同数据状态展现不同字段的情况。
             * 但开发者是否会因为不同数据状态就要配置不同的字段和顺序来增加复杂度还未知。
             * 至少目前在鼎捷内部的应用团队，都未看到。
             */
            if (i == 1) {
                break;
            }
            fieldMetaData = entry.getValue();
            i++;
        }
        if(ObjectUtils.isEmpty(fieldMetaData)){
            return Collections.emptyList();
        }
        List<ScoredFieldsSummary> scoredFieldSummaries = Collections.emptyList();
        if(fieldMetaData != null){
            scoredFieldSummaries = fieldMetaData.getNewScoreWords();
        }
        if (CollectionUtils.isEmpty(scoredFieldSummaries)) {
            return Collections.emptyList();
        }

        /**
         * FIXME
         * 这里只获取了第一组词库的排序结果。但词库针对业务字段的类型（M、D、SD等）做了不同的排序，移动还需有不同对应
         */
        List<ScoredField> scoredFields = scoredFieldSummaries.get(0).getFields();

        return scoredFields.stream()
                .sorted(Comparator.comparingLong(ScoredField::getScore).reversed())
                .map(ScoredField::getFieldName)
                .collect(Collectors.toList());
    }

    @Data
    @Builder
    @NoArgsConstructor
    @AllArgsConstructor
    private static class DesignerFieldOrder {
        private String fieldId;
        private Long order;
    }

    /**
     * 根据字段TAG中的排序TAG进行排序，获得排序后的字段id清单。
     *
     * @param uiBotDesignerConfig UiBot设计器配置
     * @return
     */
    private List<String> getOrderedFieldIdsByTag(UiBotDesignerConfig uiBotDesignerConfig) {
        final String ORDER_TAG_CATEGORY = "ORDER";
        final String TAG_CODE_SPLITTER = "__";
        final String ORDER_TAG_SPLITTER = "_";
        final Long ORDER_TAG_MAX_NUMBER = 999999999L;
        int fieldSize = uiBotDesignerConfig.getTagConfigs().size();


        // 取出ORDER类型的Tag类型，并解析成对应的数字顺序
        List<DesignerFieldOrder> designerFieldOrderList = new ArrayList<>(fieldSize);
        for (UiBotDesignerFieldTagConfig fieldTagConfig : uiBotDesignerConfig.getTagConfigs()) {
            String fieldId = fieldTagConfig.getEspFieldId();
            Optional<KnowledgeMapsTag> optKmTag = fieldTagConfig.getTags()
                    .stream().filter(tag -> ORDER_TAG_CATEGORY.equals(tag.getCategory()))
                    .findFirst();
            if (null == fieldId || !optKmTag.isPresent()) {
                continue;
            }
            /**
             * Tag对象的JSON示例
             *  {
             *      "code": "performer__HelloAthena_Demo_helloathena_task0001__ORDER_80",
             *      "name": "顺序80",
             *      "extendedInfo": {},
             *      "id": 680070273835204608,
             *      "category": "ORDER"
             *  }
             */
            String[] splitTagCode = optKmTag.get().getCode().split(TAG_CODE_SPLITTER);
            // TAG格式不规范的字段，序号最大，会排在最后展示
            if (splitTagCode.length < 3) {
                designerFieldOrderList.add(DesignerFieldOrder.builder()
                        .fieldId(fieldId)
                        .order(ORDER_TAG_MAX_NUMBER)
                        .build());
                continue;
            }
            String orderDesc = splitTagCode[2];
            String[] splitOrderDesc = orderDesc.split(ORDER_TAG_SPLITTER);
            if (splitOrderDesc.length < 2) {
                designerFieldOrderList.add(DesignerFieldOrder.builder()
                        .fieldId(fieldId)
                        .order(ORDER_TAG_MAX_NUMBER)
                        .build());
                continue;
            }
            // splitOrderDesc 长度可能有三，也有可能两，orderNumber都是在最后一个
            String orderNumber = splitOrderDesc[splitOrderDesc.length - 1];
            designerFieldOrderList.add(DesignerFieldOrder.builder()
                    .fieldId(fieldId)
                    .order(Long.parseLong(orderNumber))
                    .build());
        }

        // 正序Order字段后提取FieldId返回
        return designerFieldOrderList.stream()
                .sorted(Comparator.comparingLong(DesignerFieldOrder::getOrder))
                .map(DesignerFieldOrder::getFieldId)
                .collect(Collectors.toList());
    }
}
