package com.digiwin.mobile.mobileuibot.designer.uibot.service;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.designer.uibot.support.PcUiBotOpenWindowDefineAnalyzer;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotExecuteContext;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotOperationDTO;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotTargetLayoutSearchResult;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.field.UiBotFieldOption;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.operation.PcUiBotOpenWindowDefineDTO;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.operation.PcUiBotTmOperation;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumn;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumnDefinition;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumnEditor;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumnEditorOption;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>功能描述：UIBot设计器『操作』配置之应用器实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: UiBotDesignerFieldOperationApplierServiceImpl
 * @Author: zaregoto
 * @Date: 2023/2/27 12:07
 */
@Slf4j
@Service("uiBotDesignerFieldOperationApplierService")
public class UiBotDesignerFieldOperationApplierServiceImpl implements UiBotDesignerFieldOperationApplierService {

    @Autowired
    private PcUiBotOpenWindowDefineAnalyzer pcUiBotOpenWindowDefineAnalyzer;

    @Override
    public void applyOpenWindowOperation(UiBotModel uiBotModel, List<PcUiBotTmOperation> openWindowTmOperationList) {
        // 从PC端UIBot数据中提取出的，待处理的layout
        UiBotTargetLayoutSearchResult layoutSearchResult = uiBotModel.searchPcTargetLayout();
        if (!layoutSearchResult.found()) {
            return;
        }
        UiBotLayout pcDataLayout = layoutSearchResult.getTargetUiBotLayout();
        // 将设计器中配置的字段开窗操作应用到原始PC UiBot中的字段内
        if (pcDataLayout.isTypeOfTable()) {
            this.processFieldOpenWindowWithPcTable(uiBotModel, pcDataLayout, openWindowTmOperationList);
            uiBotModel.setAppliedDesignerFieldOperation(true);
        } else if (pcDataLayout.isTypeOfFormList()) {
            this.processFieldOpenWindowWithPcFormList(uiBotModel, pcDataLayout, openWindowTmOperationList);
            uiBotModel.setAppliedDesignerFieldOperation(true);
        } else {
            return;
        }
    }

    @Override
    public void applySingleSelectOperation(UiBotModel uiBotModel, List<PcUiBotTmOperation> singleSelectTmOperationList) {
        // 从PC端UIBot数据中提取出的，待处理的layout
        UiBotTargetLayoutSearchResult layoutSearchResult = uiBotModel.searchPcTargetLayout();
        if (!layoutSearchResult.found()) {
            return;
        }
        UiBotLayout pcDataLayout = layoutSearchResult.getTargetUiBotLayout();
        // 将设计器中配置的字段开窗操作应用到原始PC UiBot中的字段内
        if (pcDataLayout.isTypeOfTable()) {
            this.processFieldSingleSelectWithPcTable(uiBotModel, pcDataLayout, singleSelectTmOperationList);
            uiBotModel.setAppliedDesignerFieldOperation(true);
        } else if (pcDataLayout.isTypeOfFormList()) {
            this.processFieldSingleSelectWithPcFormList(uiBotModel, pcDataLayout, singleSelectTmOperationList);
            uiBotModel.setAppliedDesignerFieldOperation(true);
        } else {
            return;
        }
    }

    /**
     * 针对PC UIBot的表格布局处理字段开窗操作
     *
     * @param uiBotModel
     * @param pcDataLayout
     * @param openWindowPageTmOperationList
     */
    private void processFieldOpenWindowWithPcTable(UiBotModel uiBotModel, UiBotLayout pcDataLayout, List<PcUiBotTmOperation> openWindowPageTmOperationList) {
        String schema = pcDataLayout.getSchema();
        // 因为是TABLE类型的layout，所以数据一定是数组
        List<Map<String, Object>> bizDataList = (List<Map<String, Object>>) uiBotModel.getPageData().get(schema);
        Assert.notNull(bizDataList, "pagedata." + schema + " cannot be null");

        List<UiBotTableColumnDefinition> tableColumnDefinitionList = pcDataLayout.getColumnDefs();
        if (CollectionUtils.isEmpty(tableColumnDefinitionList)) {
            return;
        }
        // 分解出明细数据一行的字段列表
        List<UiBotTableColumn> tableColumns = UiBotTableColumnDefinition.decomposeTableColumnDefinitions(tableColumnDefinitionList);
        Map<String, UiBotTableColumn> tableColumnMap = tableColumns.stream()
                .collect(Collectors.toMap(UiBotTableColumn::getSchema, c -> c, (c1, c2) -> c1));

        for (PcUiBotTmOperation pcUiBotTmOperation : openWindowPageTmOperationList) {
            String targetFieldId = pcUiBotTmOperation.getApplyToField();
            UiBotTableColumn targetColumn = tableColumnMap.get(targetFieldId);
            if (null == targetColumn) {
                continue;
            }
            // FIXME 需要用常量/枚举表示
            targetColumn.setType("OPERATION_EDITOR");
            // 从PC UIBot运行时接口实际返回的上下文对象，转换成可被PC逻辑分析处理的实体类
            PcUiBotOpenWindowDefineDTO pcUiBotOpenWindowDefineDTO = pcUiBotOpenWindowDefineAnalyzer
                    .analysis(JsonUtil.objectToJavaObject(uiBotModel.getExecuteContext(), PcUiBotExecuteContext.class), pcUiBotTmOperation, false);
            JSONArray operationArray = new JSONArray(1);
            operationArray.add(pcUiBotOpenWindowDefineDTO);
            targetColumn.setOperations(operationArray);
        }
    }

    /**
     * 针对PC UIBot的表单布局处理字段开窗操作
     *
     * @param uiBotModel
     * @param pcDataLayout
     * @param openWindowPageTmOperationList
     */
    private void processFieldOpenWindowWithPcFormList(UiBotModel uiBotModel, UiBotLayout pcDataLayout, List<PcUiBotTmOperation> openWindowPageTmOperationList) {
        String schema = pcDataLayout.getSchema();
        // 因为是FORM_LIST类型的layout，所以数据一定是对象
        Map<String, Object> bizData = (Map<String, Object>) uiBotModel.getPageData().get(schema);
        Assert.notNull(bizData, "pagedata." + schema + " cannot be null");

        List<UiBotLayout> pcFieldLayoutList = pcDataLayout.decomposeFieldsWithPcFormList(Collections.emptyList());
        Map<String, UiBotLayout> pcFieldLayoutMap = pcFieldLayoutList.stream()
                .collect(Collectors.toMap(UiBotLayout::getSchema, c -> c, (c1, c2) -> c1));

        for (PcUiBotTmOperation pcUiBotTmOperation : openWindowPageTmOperationList) {
            String targetFieldId = pcUiBotTmOperation.getApplyToField();
            UiBotLayout targetFieldLayout = pcFieldLayoutMap.get(targetFieldId);
            if (null == targetFieldLayout) {
                continue;
            }
            // FIXME 需要用常量/枚举表示
            targetFieldLayout.setType("FORM_OPERATION_EDITOR");
            // 从PC UIBot运行时接口实际返回的上下文对象，转换成可被PC逻辑分析处理的实体类
            PcUiBotOpenWindowDefineDTO pcUiBotOpenWindowDefineDTO = pcUiBotOpenWindowDefineAnalyzer
                    .analysis(JsonUtil.objectToJavaObject(uiBotModel.getExecuteContext(), PcUiBotExecuteContext.class), pcUiBotTmOperation, false);
            PcUiBotOperationDTO pcUiBotOperationDTO = JsonUtil.objectToJavaObject(pcUiBotTmOperation, new TypeReference<PcUiBotOperationDTO>() {
            });
            pcUiBotOperationDTO.setOpenWindowDefine(pcUiBotOpenWindowDefineDTO);
            JSONArray operationArray = new JSONArray(1);
            operationArray.add(pcUiBotTmOperation);
            targetFieldLayout.setOperations(operationArray);
        }
    }


    /**
     * 针对PC UIBot的表格布局处理字段单选操作
     *
     * @param uiBotModel
     * @param pcDataLayout
     * @param singleSelectTmOperationList
     */
    private void processFieldSingleSelectWithPcTable(UiBotModel uiBotModel, UiBotLayout pcDataLayout, List<PcUiBotTmOperation> singleSelectTmOperationList) {
        String schema = pcDataLayout.getSchema();
        // 因为是TABLE类型的layout，所以数据一定是数组
        List<Map<String, Object>> bizDataList = (List<Map<String, Object>>) uiBotModel.getPageData().get(schema);
        Assert.notNull(bizDataList, "pagedata." + schema + " cannot be null");

        List<UiBotTableColumnDefinition> tableColumnDefinitionList = pcDataLayout.getColumnDefs();
        if (CollectionUtils.isEmpty(tableColumnDefinitionList)) {
            return;
        }
        // 分解出明细数据一行的字段列表
        List<UiBotTableColumn> tableColumns = UiBotTableColumnDefinition.decomposeTableColumnDefinitions(tableColumnDefinitionList);
        Map<String, UiBotTableColumn> tableColumnMap = tableColumns.stream()
                .collect(Collectors.toMap(UiBotTableColumn::getSchema, c -> c, (c1, c2) -> c1));

        for (PcUiBotTmOperation pcUiBotTmOperation : singleSelectTmOperationList) {
            UiBotTableColumn targetColumn = tableColumnMap.get(pcUiBotTmOperation.getApplyToField());
            if (null == targetColumn) {
                continue;
            }
            // FIXME 需要用常量/枚举表示
            targetColumn.setType("SELECT");
            List<JSONObject> options = JsonUtil.objectToJavaObject(pcUiBotTmOperation.getEditorDefine().getOptions(),
                    new TypeReference<List<JSONObject>>() {
                    });

            List<UiBotTableColumnEditorOption> targetColumnEditorOptionList = new ArrayList<>(options.size());
            for (JSONObject option : options) {
                // 根据语言别更新单选的title
                JSONObject languageConfig = option.getJSONObject("language");
                String locale = AppRequestContext.getContextEntity().getLocale();

                targetColumnEditorOptionList.add(
                        new UiBotTableColumnEditorOption()
                                .setTitle(languageConfig.getJSONObject("title").getString(locale))
                                .setValue(String.valueOf(option.get("value")))
                );
            }
            UiBotTableColumnEditor targetColumnEditor = new UiBotTableColumnEditor()
                    // FIXME 需要用常量/枚举表示
                    .setType("SELECT")
                    .setOptions(targetColumnEditorOptionList);
            targetColumn.setEditor(targetColumnEditor);
        }
    }

    /**
     * 针对PC UIBot的表单布局处理字段单选操作
     *
     * @param uiBotModel
     * @param pcDataLayout
     * @param singleSelectTmOperationList
     */
    private void processFieldSingleSelectWithPcFormList(UiBotModel uiBotModel, UiBotLayout pcDataLayout, List<PcUiBotTmOperation> singleSelectTmOperationList) {
        String schema = pcDataLayout.getSchema();
        // 因为是FORM_LIST类型的layout，所以数据一定是对象
        Map<String, Object> bizData = (Map<String, Object>) uiBotModel.getPageData().get(schema);
        Assert.notNull(bizData, "pagedata." + schema + " cannot be null");

        List<UiBotLayout> pcFieldLayoutList = pcDataLayout.decomposeFieldsWithPcFormList(Collections.emptyList());
        Map<String, UiBotLayout> pcFieldLayoutMap = pcFieldLayoutList.stream()
                .collect(Collectors.toMap(UiBotLayout::getSchema, c -> c, (c1, c2) -> c1));

        for (PcUiBotTmOperation pcUiBotTmOperation : singleSelectTmOperationList) {
            UiBotLayout targetFieldLayout = pcFieldLayoutMap.get(pcUiBotTmOperation.getApplyToField());
            if (null == targetFieldLayout) {
                continue;
            }
            // FIXME 需要用常量/枚举表示
            targetFieldLayout.setType("SELECT");
            List<JSONObject> options = JsonUtil.objectToJavaObject(pcUiBotTmOperation.getEditorDefine().getOptions(),
                    new TypeReference<List<JSONObject>>() {
                    });

            List<UiBotFieldOption> targetFieldOptionList = new ArrayList<>(options.size());
            for (JSONObject option : options) {
                // 根据语言别更新单选的title
                JSONObject languageConfig = option.getJSONObject("language");
                String locale = AppRequestContext.getContextEntity().getLocale();

                targetFieldOptionList.add(
                        new UiBotFieldOption()
                                .setTitle(languageConfig.getJSONObject("title").getString(locale))
                                .setValue(String.valueOf(option.get("value")))
                );
            }
            targetFieldLayout.setOptions(targetFieldOptionList);
        }
    }
}
