package com.digiwin.mobile.mobileuibot.designer.uibot.service;

import com.digiwin.mobile.mobileuibot.designer.uibot.UiBotDesignerConfig;
import com.digiwin.mobile.mobileuibot.designer.uibot.UiBotDesignerFieldTagConfig;
import com.digiwin.mobile.mobileuibot.designer.uibot.taganalyzer.DesignerTagAnalyzer;
import com.digiwin.mobile.mobileuibot.designer.uibot.taganalyzer.DesignerTagAnalyzerFactory;
import com.digiwin.mobile.mobileuibot.proxy.knowledgemaps.model.KnowledgeMapsTag;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotBizFieldBase;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotModel;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.UiBotTargetLayoutSearchResult;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotLayout;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumn;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.table.UiBotTableColumnDefinition;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>功能描述：UIBot设计器『标签』配置之应用器实现类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: UiBotDesignerFieldTagApplierServiceImpl
 * @Author: zaregoto
 * @Date: 2023/3/1 15:15
 */
@Slf4j
@Service("uiBotDesignerFieldTagApplierService")
public class UiBotDesignerFieldTagApplierServiceImpl implements UiBotDesignerFieldTagApplierService {

    private final DesignerTagAnalyzerFactory designerTagAnalyzerFactory;

    public UiBotDesignerFieldTagApplierServiceImpl(DesignerTagAnalyzerFactory designerTagAnalyzerFactory) {
        this.designerTagAnalyzerFactory = designerTagAnalyzerFactory;
    }

    @Override
    public void applyFieldTag(UiBotModel uiBotModel, UiBotDesignerConfig uiBotDesignerConfig) {
        // 从PC端UIBot数据中提取出的，待处理的layout
        UiBotTargetLayoutSearchResult layoutSearchResult = uiBotModel.searchPcTargetLayout();
        if (!layoutSearchResult.found()) {
            return;
        }
        List<UiBotDesignerFieldTagConfig> tagConfigList = uiBotDesignerConfig.getTagConfigs();
        UiBotLayout pcDataLayout = layoutSearchResult.getTargetUiBotLayout();
        // 将设计器中配置的字段开窗操作应用到原始PC UiBot中的字段内
        if (pcDataLayout.isTypeOfTable()) {
            List<UiBotTableColumnDefinition> tableColumnDefinitionList = pcDataLayout.getColumnDefs();
            if (CollectionUtils.isEmpty(tableColumnDefinitionList)) {
                return;
            }
            List<UiBotTableColumn> tableColumns = UiBotTableColumnDefinition.decomposeTableColumnDefinitions(tableColumnDefinitionList);
            this.commonProcessTag(uiBotModel, tagConfigList, tableColumns);
            uiBotModel.setAppliedDesignerFieldTag(true);
        } else if (pcDataLayout.isTypeOfFormList()) {
            List<UiBotLayout> pcFieldLayoutList = pcDataLayout.decomposeFieldsWithPcFormList(Collections.emptyList());
            this.commonProcessTag(uiBotModel, tagConfigList, pcFieldLayoutList);
            uiBotModel.setAppliedDesignerFieldTag(true);
        } else {
            return;
        }
    }

    /**
     * 通用TAG解析与处理
     *
     * @param uiBotModel
     * @param tagConfigList    TAG配置清单
     * @param bizFieldBaseList 业务字段清单
     */
    private <T extends UiBotBizFieldBase> void commonProcessTag(UiBotModel uiBotModel,
                                                                List<UiBotDesignerFieldTagConfig> tagConfigList, List<T> bizFieldBaseList) {
        if (CollectionUtils.isEmpty(bizFieldBaseList)) {
            return;
        }
        Map<String, T> bizFieldBaseMap = bizFieldBaseList.stream()
                .collect(Collectors.toMap(T::getSchema, c -> c, (c1, c2) -> c1));
        for (UiBotDesignerFieldTagConfig uiBotDesignerFieldTagConfig : tagConfigList) {
            String targetFieldId = uiBotDesignerFieldTagConfig.getEspFieldId();
            UiBotBizFieldBase targetBizFieldBase = bizFieldBaseMap.get(targetFieldId);
            if (null == targetBizFieldBase) {
                continue;
            }
            String previousAnalyzedComponentTagCode = "";
            for (KnowledgeMapsTag knowledgeMapsTag : uiBotDesignerFieldTagConfig.getTags()) {
                // ORDER类型TAG不实现解析器，在FieldAndOrderApplierService内实现，所以可能为NULL
                DesignerTagAnalyzer tagAnalyzer = this.designerTagAnalyzerFactory.get(knowledgeMapsTag.getCode());
                if (null == tagAnalyzer) {
                    continue;
                }
                String currentTagCode = tagAnalyzer.getTagCode();
                /**
                 * 单个业务字段可处理的TAG分类
                 * 1. 改组件状态的TAG：无数量、种类限制；举例：不可编辑TAG
                 * 2. 改组件类型的TAG：只能允许处理1种，如果相同种类的有多个，则只处理第1个出现的。举例：文本域TAG、手写签名TAG
                 */
                if (Boolean.TRUE.equals(tagAnalyzer.tagCodeChangeComponentState())) {
                    tagAnalyzer.analyze(uiBotModel, targetBizFieldBase, knowledgeMapsTag);
                }
                if (!StringUtils.hasLength(previousAnalyzedComponentTagCode) && Boolean.TRUE.equals(tagAnalyzer.tagCodeDecideComponentType())) {
                    tagAnalyzer.analyze(uiBotModel, targetBizFieldBase, knowledgeMapsTag);
                    previousAnalyzedComponentTagCode = currentTagCode;
                }
            }
        }
    }
}
