package com.digiwin.mobile.mobileuibot.device;

import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.device.enums.AuthTypeEnum;
import com.digiwin.mobile.mobileuibot.device.model.ApiRequsetDeviceBind;
import com.digiwin.mobile.mobileuibot.device.model.DeviceBindAuthResp;
import com.digiwin.mobile.mobileuibot.device.service.DeviceBindService;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.proxy.semc.model.DeviceBindAuth;
import com.digiwin.mobile.mobileuibot.proxy.semc.model.DeviceBindConfig;
import com.digiwin.mobile.mobileuibot.proxy.semc.model.DeviceBindResp;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;
import java.util.Optional;

/**
 * <p>功能描述：设备绑定相关接口</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: DeviceBindController.java
 * @Author: wangjwc
 * @Date: created at 2025/7/31 15:48
 */
@RestController
@RequestMapping("/mobile/v1/device/bind")
public class DeviceBindController {

    // 支持的操作系统类型
    private static final List<String> SUPPORT_OS_TYPE = Lists.newArrayList("android", "ios");

    @Autowired
    private DeviceBindService deviceBindService;

    @Autowired
    private LocaleService localeService;

    /**
     * 租户级设备绑定开关配置
     * @param iamUserToken 用户token
     * @return 开关配置
     */
    @GetMapping("/config")
    public ApiResponse<DeviceBindConfig> deviceBindConfig(@RequestParam("iamUserToken") String iamUserToken) {
        return ApiResponse.buildOK(deviceBindService.deviceBindConfig());
    }

    /**
     * 账号已绑定设备列表
     * @param iamUserToken 用户token
     * @return 设备绑定列表
     */
    @GetMapping("/list")
    public ApiResponse<List<DeviceBindResp>> deviceBindList(@RequestParam("iamUserToken") String iamUserToken) {
        return ApiResponse.buildOK(deviceBindService.deviceBindList(iamUserToken));
    }

    /**
     * 账号登陆设备认证
     * @param requsetDeviceBind 请求入参
     * @return 返回
     */
    @PostMapping("/auth")
    public ApiResponse<DeviceBindAuthResp> deviceBindAuth(@Valid @RequestBody ApiRequsetDeviceBind requsetDeviceBind) {
        /*
          h5端：【为保障账号安全，当前租户已启用设备绑定功能，暂不支持通过H5 或 Web 端登录。请使用 APP 或桌面客户端访问】
          1、deviceAuthorize=true 验证通过
          2、deviceAuthorize=false
             2.1：bindEnabled=true&bindDeviceStatus==false，前端提示---管理员绑定
            【当前设备尚未和当前账号绑定！请联系租户管理员】
             2.2：bindEnabled=true&bindDeviceStatus==true，进一步判断
                2.2.1、accountExistBinding==true，前端提示，您的账号已绑定了同类型（同PC端或同移动端）设备，如您确实需要绑定多个设备，请联系管理员
                【当前账号已被其他移动设备绑定！若想绑定当前设备，请联系管理员添加】
                2.2.2、deviceExistBinding==true，当前设备已经被绑定过，前端提示 是和否
                【当前设备已绑定其他账号！您可将其与当前账号绑定】
                2.2.3、deviceExistBinding==false &accountExistBinding=false，前端提示 是和否
                【是否要绑定该设备】
         */
        DeviceBindAuthResp result = new DeviceBindAuthResp().setType(AuthTypeEnum.PASS.getValue());
        if (this.notSupportOsType(requsetDeviceBind.getOsType())) {
            // 校验租户设备绑定开关是否开启
            if (BooleanUtils.isTrue(Optional.ofNullable(deviceBindService.deviceBindConfig()).map(DeviceBindConfig::getBindEnabled).orElse(false))) {
                // 开启：不支持绑定
                result.setType(AuthTypeEnum.UNSUPPORT.getValue()).setMessage(localeService.getLanguageValue(requsetDeviceBind.getLocale(), "为保障账号安全，当前租户已启用设备绑定功能，暂不支持通过H5 或 Web 端登录。\n请使用 APP 或桌面客户端访问"));
            }
            return ApiResponse.buildOK(result);
        }
        // 认证结果
        DeviceBindAuth auth = deviceBindService.deviceBindAuth(requsetDeviceBind);
        if (auth == null || BooleanUtils.isTrue(auth.getDeviceAuthorize())) {
            return ApiResponse.buildOK(result);
        }
        Boolean bindEnabled = Optional.ofNullable(auth.getBindConfig()).map(DeviceBindConfig::getBindEnabled).orElse(false);
        Boolean bindDeviceStatus = Optional.ofNullable(auth.getBindConfig()).map(DeviceBindConfig::getBindDeviceStatus).orElse(false);
        if (BooleanUtils.isNotTrue(bindEnabled)) {
            return ApiResponse.buildOK(result);
        }
        // 开启了设备绑定功能(bindEnabled=true)
        if (BooleanUtils.isNotTrue(bindDeviceStatus)) {
            // 未开启登录首次绑定功能
            result.setType(AuthTypeEnum.OPENED.getValue()).setMessage(localeService.getLanguageValue(requsetDeviceBind.getLocale(), "当前设备尚未和当前账号绑定！\n请联系租户管理员"));
            return ApiResponse.buildOK(result);
        }
        // 开启了账号首次绑定功能(bindDeviceStatus=true)
        if (BooleanUtils.isTrue(auth.getAccountExistBinding())) {
            // 当前账号已被其他移动设备绑定
            result.setType(AuthTypeEnum.OPENED.getValue()).setMessage(localeService.getLanguageValue(requsetDeviceBind.getLocale(), "当前账号已被其他移动设备绑定！\n若想绑定当前设备，请联系管理员添加"));
            return ApiResponse.buildOK(result);
        }
        // 当前账号未被其他移动设备绑定(accountExistBinding=false)
        if (BooleanUtils.isTrue(auth.getDeviceExistBinding())) {
            // 当前设备已绑定其他账号
            result.setType(AuthTypeEnum.AUTO.getValue()).setMessage(localeService.getLanguageValue(requsetDeviceBind.getLocale(), "当前设备已绑定其他账号！\n您可将其与当前账号绑定"));
            return ApiResponse.buildOK(result);
        }
        // 设备、账号都未被绑定过(accountExistBinding=false &deviceExistBinding=false)
        result.setType(AuthTypeEnum.AUTO.getValue()).setMessage(localeService.getLanguageValue(requsetDeviceBind.getLocale(), "是否要绑定该设备"));
        return ApiResponse.buildOK(result);
    }

    private boolean notSupportOsType(String osType) {
        return SUPPORT_OS_TYPE.stream().noneMatch(type -> type.equalsIgnoreCase(osType));
    }

    /**
     * 账号设备首次绑定
     * @param requsetDeviceBind 请求入参
     * @return 返回
     */
    @PostMapping("/auto")
    public ApiResponse<Boolean> deviceBindAuto(@Valid @RequestBody ApiRequsetDeviceBind requsetDeviceBind) {
        if (this.notSupportOsType(requsetDeviceBind.getOsType())) {
            return ApiResponse.buildError(localeService.getLanguageValue(requsetDeviceBind.getLocale(), "为保障账号安全，当前租户已启用设备绑定功能，暂不支持通过H5 或 Web 端登录。\n请使用 APP 或桌面客户端访问"));
        }
        deviceBindService.deviceBindAuto(requsetDeviceBind);
        return ApiResponse.buildOK();
    }

}
