package com.digiwin.mobile.mobileuibot.experience;

import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.context.AppEnvDeployAreaEnum;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.request.RequestParameterUtil;
import com.digiwin.mobile.mobileuibot.experience.dao.ExperienceRoleRepository;
import com.digiwin.mobile.mobileuibot.experience.model.ExperienceRole;
import com.digiwin.mobile.mobileuibot.experience.model.ExperienceRoleIdEnum;
import com.digiwin.mobile.mobileuibot.mock.MockService;
import com.digiwin.mobile.mobileuibot.mock.enums.EnableEnum;
import com.digiwin.mobile.mobileuibot.model.db1.Mock;
import com.digiwin.mobile.mobileuibot.proxy.agiledataecho.model.EchoExperienceRoleConfigDataQueryDTO;
import com.digiwin.mobile.mobileuibot.proxy.agiledataecho.model.EchoExperienceRoleConfigDataResultDTO;
import com.digiwin.mobile.mobileuibot.proxy.agiledataecho.service.DigiwinAgileDataEchoProxyService;
import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 体验模式Controller
 *
 * @author zhangjj
 * @date 2022/9/14 10:34
 */
@RestController
@RequestMapping("/mobile/v1/experience")
public class ExperienceController {

    @Autowired
    private ExperienceRoleRepository experienceRoleRepository;

    @Autowired
    private MockService mockService;

    @Autowired
    private DigiwinAgileDataEchoProxyService echoProxyService;

    @PostMapping(value = "/role/list")
    public ApiResponse<Map> getExperienceRoleList(HttpServletRequest request) {
        Map<String, Object> data = new HashMap<>();
        try {
            Map<String, Object> params = RequestParameterUtil.getPostDataMap(request, false, "UTF-8");
            String locale = StringUtils.defaultIfBlank(String.valueOf(params.get("locale")), "zh_CN");
            String channel = (String)params.get("channel");

            //判断地区
            String district = Objects.equals(AppEnvDeployAreaEnum.CN.toString(), AppContext.getEnvDeployArea()) ? "CN" : "TW";
            // 查询移动端体验角色列表
            List<ExperienceRole> roleList = experienceRoleRepository.getRoleList();
            if(!CollectionUtils.isEmpty(roleList) && StringUtils.isNotBlank(channel) && channel.equals("line")){
                //判断移动端体验角色列表是否为空并且是否当前查询渠道为line，只需要返回和娜娜相关的应用即可
                roleList = roleList.stream().filter(role -> Objects.nonNull(role.getNanaAssistant()) && role.getNanaAssistant() == 1).collect(Collectors.toList());
            }
            // 查询敏捷数据移动端体验角色列表
            List<EchoExperienceRoleConfigDataResultDTO> experienceRoleConfigList = echoProxyService.queryExperienceRoleConfigList(
                    EchoExperienceRoleConfigDataQueryDTO.builder().locale(locale).district(district).build()
            );
            // 如果角色列表均为空，直接返回空列表
            if (CollectionUtils.isEmpty(roleList) && CollectionUtils.isEmpty(experienceRoleConfigList)) {
                data.put("roleList", new ArrayList<>());
                return ApiResponse.buildOK().setData(data);
            }
            // 如果移动端角色列表为空，根据敏捷数据体验角色列表进行填充
            if (CollectionUtils.isEmpty(roleList)) {
                roleList = experienceRoleConfigList.stream()
                        .map(dto -> new ExperienceRole(dto.getId(), dto.getRoleId(), dto.getRoleName(), dto.getSort()))
                        .collect(Collectors.toList());
            }
            // 设置角色名称，并移除角色名称映射
            roleList.forEach(role -> {
                role.setRoleName(Optional.ofNullable(role.getRoleNameMap()).map(map -> map.get(locale)).orElse(role.getRoleName()));
                role.setRoleNameMap(null);
            });
            // 合并敏捷数据角色
            roleList = handleAgileDataExperienceRoles(experienceRoleConfigList, roleList);
            // 对角色列表进行排序
            roleList.sort(Comparator.comparing(ExperienceRole::getSort));
            List<Map<String, Object>> list = JsonUtil.objectToJavaObject(roleList, new TypeReference<List<Map<String, Object>>>() {
            });
            // 添加MockList
            List<Mock> mockList = mockService.mockListOfPublish();
            if (!CollectionUtils.isEmpty(mockList)) {
                if(StringUtils.isNotBlank(channel) && channel.equals("line")){
                    //当前查询渠道为line，只需要返回和娜娜相关的MockList即可
                    mockList = mockList.stream().filter(mock -> Objects.nonNull(mock.getEnableNana()) && EnableEnum.YES.getValue().equals(mock.getEnableNana())).collect(Collectors.toList());
                }
                mockList.forEach(mock -> {
                    if (EnableEnum.YES.getValue().equals(mock.getEnableNana())) {
                        mock.setMockRoleId(ExperienceRoleIdEnum.ATHENA_MOCK.getValue());
                    }
                });
                list.addAll(JsonUtil.objectToJavaObject(mockList, new TypeReference<List<Map<String, Object>>>() {
                }));
            }
            data.put("roleList", list);
        } catch (IOException e) {
            return ApiResponse.buildError("获取角色列表时发生错误：" + e.getMessage());
        } catch (Exception e) {
            return ApiResponse.buildError("获取角色列表时发生异常：" + e.getMessage());
        }
        return ApiResponse.buildOK().setData(data);
    }

    /**
     * 合并敏捷数据角色
     *
     * @param experienceRoleConfigList
     * @param roleList
     * @return List<ExperienceRole>
     * @author yanfeng
     */
    private static List<ExperienceRole> handleAgileDataExperienceRoles(List<EchoExperienceRoleConfigDataResultDTO> experienceRoleConfigList, List<ExperienceRole> roleList) {
        if (!CollectionUtils.isEmpty(experienceRoleConfigList)) {
            Map<String, ExperienceRole> roleMap = roleList.stream()
                    .collect(Collectors.toMap(ExperienceRole::getRoleId, role -> role, (a, b) -> b));
            for (EchoExperienceRoleConfigDataResultDTO dto : experienceRoleConfigList) {
                roleMap.computeIfPresent(dto.getRoleId(), (id, role) -> {
                    role.setRoleName(dto.getRoleName());
                    role.setSort(dto.getSort());
                    return role;
                });
                roleMap.putIfAbsent(dto.getRoleId(), new ExperienceRole(dto.getId(), dto.getRoleId(), dto.getRoleName(), dto.getSort()));
            }
            roleList = new ArrayList<>(roleMap.values());
        }
        return roleList;
    }

}