package com.digiwin.mobile.mobileuibot.feedback;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.mobile.mobileuibot.api.ApiRequestAgileDataInAiChat;
import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.proxy.DigiwinAthenaApiResponse;
import com.digiwin.mobile.mobileuibot.proxy.adt.model.DigiwinAdtUserFeedback;
import com.digiwin.mobile.mobileuibot.proxy.adt.service.DigiwinAdtProxyService;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpStatus;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>功能描述：用户反馈接口</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: FeedbackController
 * @Author: zaregoto
 * @Date: 2024/1/25 21:16
 */
@RestController
@RequestMapping("/mobile/v1/feedback")
@Slf4j
@Validated
public class FeedbackController {

    @Autowired
    private DigiwinAdtProxyService digiwinAdtProxyService;

    @Autowired
    private LocaleService localeService;

    /**
     * 提交点赞or点踩的操作
     *
     * @param apiRequest
     * @return
     */
    @PostMapping(value = "/aiassistant/chat/submit")
    public ApiResponse<Boolean> aiAssistantChatSubmit(@Validated @RequestBody ApiRequestFeedbackActionInAiChat apiRequest) {
        // 敏捷数据引擎流水号
        Long generateSerialNo = apiRequest.getGenerateSerialNo();
        // 反馈操作类型
        Integer feedbackType = apiRequest.getFeedbackType();
        // 反馈中的会话信息
        ChatInfo chatInfo = apiRequest.getChatInfo();
        // 技能类型
        String skillType = chatInfo.getSkillType();
        // 提问及时间
        String qu = chatInfo.getQu();
        String quTime = chatInfo.getQuTime();
        // 回复及时间
        String a = chatInfo.getA();
        String aTime = chatInfo.getATime();
        // 快照id
        String snapshotId = StringUtils.hasLength(apiRequest.getSnapshotId()) ? apiRequest.getSnapshotId() : a;

        JSONObject params = new JSONObject();
        params.put("generateSerialNo", generateSerialNo);
        params.put("snapshotId", snapshotId);
        params.put("feedbackType", apiRequest.getFeedbackType());
        params.put("question", qu);
        params.put("tenantName", apiRequest.getTenantName());
        params.put("userId", apiRequest.getUserId());
        params.put("userName", apiRequest.getUserName());
        JSONObject extendInfo = new JSONObject();
        params.put("extendInfo", extendInfo);
        extendInfo.put("appInfo", apiRequest.getAppInfo());
        extendInfo.put("chatInfo", apiRequest.getChatInfo());
        extendInfo.put("deviceInfo", apiRequest.getDeviceInfo());
        DigiwinAthenaApiResponse<Boolean> sendResult = this.digiwinAdtProxyService.sendUserFeedback(
                apiRequest.getIamUserToken(), apiRequest.getTenantId(),
                apiRequest.getLocale(), params);

        if (null == sendResult) {
            return ApiResponse.buildError(this.localeService.getLanguageValue(apiRequest.getLocale(),
                    "ADT服务无响应，反馈失败，请稍后再试"));
        }
        if (sendResult.getStatus() == HttpStatus.SC_OK && sendResult.getResponse() == true) {
            return ApiResponse.buildOK().setData(true);
        } else {
            if (sendResult.getStatus() != HttpStatus.SC_OK) {
                return ApiResponse.buildError(sendResult.getErrorMessage());
            } else {
                return ApiResponse.buildError();
            }
        }
    }

    /**
     * 取消点赞or点踩的操作
     *
     * @param apiRequest
     * @return
     */
    @PostMapping(value = "/aiassistant/chat/cancel")
    public ApiResponse<Boolean> aiAssistantChatCancel(@Validated @RequestBody ApiRequestAgileDataInAiChat apiRequest) {
        // 敏捷数据引擎流水号
        Long generateSerialNo = apiRequest.getGenerateSerialNo();
        Map<String, Object> params = new HashMap<>();
        params.put("generateSerialNo", generateSerialNo);
        DigiwinAthenaApiResponse<Boolean> cancelResult = this.digiwinAdtProxyService.cancelUserFeedback(
                apiRequest.getIamUserToken(), apiRequest.getTenantId(),
                apiRequest.getLocale(), params);
        if (null == cancelResult) {
            return ApiResponse.buildError(this.localeService.getLanguageValue(apiRequest.getLocale(),
                    "ADT服务无响应，取消操作失败，请稍后再试"));
        }
        if (cancelResult.getStatus() == HttpStatus.SC_OK && cancelResult.getResponse() == true) {
            return ApiResponse.buildOK().setData(true);
        } else {
            if (cancelResult.getStatus() != HttpStatus.SC_OK) {
                return ApiResponse.buildError(cancelResult.getErrorMessage());
            } else {
                return ApiResponse.buildError();
            }
        }
    }

    /**
     * 提交用户反馈和意见内容
     *
     * @param apiRequest
     * @return
     */
    @PostMapping(value = "/aiassistant/chat/comment/submit")
    public ApiResponse<Boolean> aiAssistantChatCommentSubmit(@Validated @RequestBody ApiRequestFeedbackCommentInAiChat apiRequest) {
        // 敏捷数据引擎流水号
        Long generateSerialNo = apiRequest.getGenerateSerialNo();
        String snapshotId = apiRequest.getSnapshotId();
        List<FeedbackChoiceContent> contents = apiRequest.getBackContent();
        String opinion = apiRequest.getOpinion();

        if (CollectionUtils.isEmpty(contents) && !StringUtils.hasLength(opinion)) {
            return ApiResponse.buildError(this.localeService.getLanguageValue(apiRequest.getLocale(),
                    "请让我们知道需要改进的地方"));
        }

        Map<String, Object> params = new HashMap<>();
        Map<String, Object> answerBack = new HashMap<>(2);
        // 旧接口格式所需参数（为兼容两个ADT环境的版本）
        answerBack.put("opinion", opinion);
        answerBack.put("backContent", contents);
        params.put("answerBack", answerBack);

        // 新接口格式所需参数
        String checkOpinions = String.join("||", contents.stream().map(choice -> choice.getId() + "").collect(Collectors.toList()));
        String backContent = String.join("||", contents.stream().map(FeedbackChoiceContent::getText).collect(Collectors.toList()));
        // 用户选择的反馈项id字符串
        params.put("checkOpinions", checkOpinions);
        // 用户选择的反馈项文字字符串
        params.put("backContent", backContent);
        // 用户手工录入的反馈意见文字
        params.put("answerOpinion", opinion);

        params.put("generateSerialNo", generateSerialNo);
        params.put("snapshotId", snapshotId);

        DigiwinAthenaApiResponse<Boolean> submitResult = this.digiwinAdtProxyService.userAnswerFeedback(
                apiRequest.getIamUserToken(), apiRequest.getTenantId(),
                apiRequest.getLocale(), params);
        if (null == submitResult) {
            return ApiResponse.buildError(this.localeService.getLanguageValue(apiRequest.getLocale(),
                    "ADT服务无响应，意见提交失败，请稍后再试"));
        }
        if (submitResult.getStatus() == HttpStatus.SC_OK && submitResult.getResponse() == true) {
            return ApiResponse.buildOK().setData(true);
        } else {
            if (submitResult.getStatus() != HttpStatus.SC_OK) {
                return ApiResponse.buildError(submitResult.getErrorMessage());
            } else {
                return ApiResponse.buildError();
            }
        }
    }

    @PostMapping(value = "/aiassistant/chat/getBySerialNos")
    public ApiResponse<List<DigiwinAdtUserFeedback>> aiAssistantChatGetBySerialNos(@Validated @RequestBody ApiRequestFeedbackGetBatchInAiChat apiRequest) {
        // 敏捷数据引擎流水号
        List<Long> generateSerialNos = apiRequest.getGenerateSerialNos();
        DigiwinAthenaApiResponse<List<DigiwinAdtUserFeedback>> getUserFeedbackRsp = this.digiwinAdtProxyService
                .getUserFeedback(apiRequest.getIamUserToken(), apiRequest.getTenantId(),
                        apiRequest.getLocale(), generateSerialNos);
        if (null == getUserFeedbackRsp) {
            return ApiResponse.buildError(this.localeService.getLanguageValue(apiRequest.getLocale(),
                    "ADT服务无响应，历史反馈获取失败，请稍后再试"));
        }
        if (getUserFeedbackRsp.getStatus() == HttpStatus.SC_OK) {
            return ApiResponse.buildOK().setData(getUserFeedbackRsp.getResponse());
        } else {
            if (getUserFeedbackRsp.getStatus() != HttpStatus.SC_OK) {
                return ApiResponse.buildError(getUserFeedbackRsp.getErrorMessage());
            } else {
                return ApiResponse.buildError();
            }
        }
    }
}
