package com.digiwin.mobile.mobileuibot.file;

import com.digiwin.mobile.mobileuibot.api.*;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.http.HttpService;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.core.component.input.attachment.AttachmentRawDataDigiwinAthena;
import com.digiwin.mobile.mobileuibot.file.fileupload.UploadFile;
import com.digiwin.mobile.mobileuibot.proxy.aam.service.DigiwinAamProxyService;
import com.digiwin.mobile.mobileuibot.proxy.dmc.config.DigiwinDmcConfig;
import com.digiwin.mobile.mobileuibot.proxy.dmc.service.DigiwinDmcProxyService;
import io.micrometer.core.instrument.util.StringUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * @author zhangjj
 * @date 2021/6/16 0016 18:21
 */
@RestController
@RequestMapping(value = "/mobile/v1/file")
public class FileController {

    public static final String ERROR_CODE = "-1";
    public static final String SUCCESS_CODE = "200";

    private static final Logger logger = LoggerFactory.getLogger(FileController.class);

    @Autowired
    private DigiwinDmcProxyService digiwinDmcProxyService;
    @Autowired
    private DigiwinAamProxyService digiwinAamProxyService;
    @Autowired
    private FileService fileService;
    @Autowired
    private HttpService httpService;

    /**
     * 上传附件
     *
     * @param apiAttachment 附件信息
     * @return
     */
    @RequestMapping(value = "/upload", method = RequestMethod.POST)
    public ApiResponse<Map<String, Object>> upload(ApiUploadAttachment apiAttachment) {
        if (apiAttachment == null) {
            return ApiResponse.buildError("请求数据有误");
        }
        Map<String, Object> result = fileService.uploadFile(apiAttachment, 1);
        if (ERROR_CODE.equals(result.get("code"))) {
            return ApiResponse.buildError((String) result.get("message"));
        } else {
            return ApiResponse.buildOK().setData(result);
        }
    }

    /**
     * 下载附件
     *
     * @param fileId        附件id
     * @param iamUserToken  互联userToken
     * @param rawDataString 附件关联的原后台系统的关联数据
     * @return 实际文件的下载地址及需要的http头部参数（1.8.0开始实际头部参数没有用了）
     */
    @RequestMapping(value = "/download", method = RequestMethod.GET)
    public ApiResponse<Map<String, Object>> download(
            @RequestParam String fileId,
            @RequestParam String iamUserToken,
            @RequestParam(required = false, defaultValue = "{}") String rawDataString) {
        //todo 校验token
        AttachmentRawDataDigiwinAthena attachmentRawData =
                JsonUtil.jsonStringToObject(rawDataString, AttachmentRawDataDigiwinAthena.class);
        String bucket = Optional.ofNullable(attachmentRawData).map(AttachmentRawDataDigiwinAthena::getBuckets).orElse(null);
        if (StringUtils.isEmpty(bucket)) {
            bucket = DigiwinDmcConfig.ATHENA_BUCKET;
        }
        //获取dmc的token
        String dmcUserToken = digiwinDmcProxyService.getDmcUserToken();
        String downUrl = AppContext.getApiUrlSetting().getDmcUrl()
                + DigiwinDmcProxyService.API_PREFIX + "/file/"
                + bucket
                + "/preview/" + fileId;
        String dmcPreviewUrl = AppContext.getApiUrlSetting().getDmcUrl()
                + DigiwinDmcProxyService.API_PREFIX + "/file/"
                + bucket
                + "/online/preview/url/" + fileId;
        Map<String, Object> headerMap = new HashMap<>();
        headerMap.put("digi-middleware-auth-user", dmcUserToken);
        Map<String, Object> dataMap = new HashMap<>();
        //url---下载的URL
        //dmcPreviewUrl---dmc下载预览的文件地址
        dataMap.put("url", downUrl);
        try {
            ResponseEntity<Map> responseEntity = this.httpService.doGet(dmcPreviewUrl, Map.class);
            if (responseEntity.getStatusCode().is2xxSuccessful()) {
                if (!ObjectUtils.isEmpty(responseEntity.getBody()) && !ObjectUtils.isEmpty(responseEntity.getBody().get("data"))) {
                    HashMap data = (HashMap) responseEntity.getBody().get("data");
                    dmcPreviewUrl = String.valueOf(data.get("previewUrl"));
                } else {
                    dmcPreviewUrl = null;
                }
            } else {
                dmcPreviewUrl = null;
            }
        } catch (Exception e) {
            e.printStackTrace();
            dmcPreviewUrl = null;
        }
        dataMap.put("dmcPreviewUrl", dmcPreviewUrl);
        dataMap.put("header", headerMap);
        return ApiResponse.buildOK().setData(dataMap);
    }


    /**
     * 删除附件
     *
     * @param apiDeleteAttachment
     * @return
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    public ApiResponse<Map<String, Object>> delete(@RequestBody ApiDeleteAttachment apiDeleteAttachment) {
        String fileId = apiDeleteAttachment.getFileId();
        if (StringUtils.isBlank(fileId)) {
            return ApiResponse.buildError("fileId为空");
        }
        AttachmentRawDataDigiwinAthena rawDataDigiwinAthena
                = apiDeleteAttachment.getRawData();
        Map dataMap = new HashMap();
        if (rawDataDigiwinAthena != null) {
            //aam中删除附件
            Boolean disableAam = rawDataDigiwinAthena.getDisableAam();
            if (!Optional.ofNullable(disableAam).orElse(false)) {
                boolean result = digiwinAamProxyService.deleteFile(fileId,
                        rawDataDigiwinAthena.getUploadCategoryId(),
                        rawDataDigiwinAthena.getUploadCategory(),
                        apiDeleteAttachment.getIamUserToken());
                if (!result) {
                    dataMap.put("success", false);
                    return ApiResponse.buildOK().setData(dataMap);
                }
            }
        }
        //dmc中删除附件
        digiwinDmcProxyService.deleteFile(fileId);
        dataMap.put("success", true);
        return ApiResponse.buildOK().setData(dataMap);
    }

    /**
     * 上传文件
     *
     * @param uploadFile
     * @return
     */
    @RequestMapping(value = "/log/upload", method = RequestMethod.POST)
    public ApiResponse<Map<String, Object>> upload(UploadFile uploadFile) {
        Map<String, Object> result = fileService.uploadFile(uploadFile);
        if (ERROR_CODE.equals(result.get("code"))) {
            return ApiResponse.buildError((String) result.get("message"));
        } else {
            return ApiResponse.buildOK().setData(result);
        }
    }

    /**
     * 分段上传文件
     *
     * @param apiAttachment 附件信息
     * @return
     */
    @PostMapping(value = "/segmentedUploadFile")
    public ApiResponse<String> segmentedUploadFile(ApiSegmentedUploadAttachment apiAttachment) {
        String fileId = fileService.segmentedUploadFile(apiAttachment);
        if (fileId == null) {
            return ApiResponse.buildError("文件ID为空,上传失败");
        }
        return ApiResponse.buildOK().setData(fileId);
    }

    /**
     * 敏捷数据lite版本:上传附件
     *
     * @param apiAttachment 附件信息
     * @return
     */
    @PostMapping(value = "/uploadFileAgileData")
    public ApiResponse uploadFileAgileData(@Validated AgileDataUploadAttachment apiAttachment) {
        return fileService.uploadFileAgileData(apiAttachment);
    }

    /**
     * 敏捷数据lite版本:获取指定目录下的文件列表
     *
     * @param apiRequest
     * @return
     */
    @PostMapping("/getDirFileList")
    public ApiResponse getDirFileList(@RequestBody ApiRequest apiRequest) {
        return ApiResponse.buildOK().setData(fileService.getDirFileList(apiRequest));
    }

}