package com.digiwin.mobile.mobileuibot.file;

import cn.hutool.core.codec.Base64;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.dap.middleware.dmc.model.DirInfo;
import com.digiwin.dap.middleware.dmc.model.FileInfo;
import com.digiwin.dap.middleware.dmc.model.ShareInfo;
import com.digiwin.mobile.mobileuibot.aiassistant.constant.AiAssistantConstant;
import com.digiwin.mobile.mobileuibot.api.*;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.common.file.FileUtil;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.core.component.input.attachment.AttachmentRawDataDigiwinAthena;
import com.digiwin.mobile.mobileuibot.file.fileupload.ContentTypeEnum;
import com.digiwin.mobile.mobileuibot.file.fileupload.UploadFile;
import com.digiwin.mobile.mobileuibot.file.fileupload.UploadFileLocationEnum;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.mapper.db1.FileUploadRecordMapper;
import com.digiwin.mobile.mobileuibot.model.db1.FileUploadRecord;
import com.digiwin.mobile.mobileuibot.proxy.aam.model.AttachmentMessage;
import com.digiwin.mobile.mobileuibot.proxy.aam.model.AttachmentResponse;
import com.digiwin.mobile.mobileuibot.proxy.aam.service.DigiwinAamProxyService;
import com.digiwin.mobile.mobileuibot.proxy.adt.model.DigiwinAdtSubmitLiteProcessMessageResult;
import com.digiwin.mobile.mobileuibot.proxy.adt.service.DigiwinAdtProxyService;
import com.digiwin.mobile.mobileuibot.proxy.dmc.config.DigiwinDmcConfig;
import com.digiwin.mobile.mobileuibot.proxy.dmc.model.SegmentedUploadFileDTO;
import com.digiwin.mobile.mobileuibot.proxy.dmc.service.DigiwinDmcProxyService;
import io.micrometer.core.instrument.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author wuyang
 */
@Service("fileService")
public class FileServiceImpl extends ServiceImpl<FileUploadRecordMapper, FileUploadRecord> implements FileService {

    // 文件上传分块大小
    public static final int FILE_UPLOAD_PART_SIZE = 5 * 4 * 255 * 1024;
    @Autowired
    private FileService fileService;

    private static final Logger logger = LoggerFactory.getLogger(FileServiceImpl.class);

    /**
     * 行事历附件上传默认目录
     */
    private static final String CALENDAR_ATTACHMENT_DIR = "calendar_file_category";

    @Autowired
    private DigiwinDmcProxyService digiwinDmcProxyService;
    @Autowired
    private DigiwinAamProxyService digiwinAamProxyService;

    @Autowired
    private DigiwinAdtProxyService digiwinAdtProxyService;

    @Autowired
    private LocaleService localeService;

    @Override
    public Map<String, Object> uploadFile(ApiUploadAttachment apiAttachment, int uploadPC1) {
        boolean uploadPC = false;
        if (uploadPC1 != 0) {
            uploadPC = true;
        }
        Map<String, Object> result = new HashMap<>();
        if (apiAttachment == null) {
            result.put("code", "-1");
            result.put("message", "请求数据有误");
            return result;
        }
        StopWatch stopWatch = new StopWatch("uploadFile");
        stopWatch.start("get MultipartFile from request");
        MultipartFile file = apiAttachment.getFile();
        stopWatch.stop();

        if (file == null) {
            result.put("code", "-1");
            result.put("message", "文件为空");
            return result;
        }
        String uploadCategoryId = "";
        String uploadCategory = "";
        AttachmentRawDataDigiwinAthena rawDataDigiwinAthena = new AttachmentRawDataDigiwinAthena();
        if (StringUtils.isNotBlank(apiAttachment.getRawData()) && !"null".equals(apiAttachment.getRawData()) && !"NULL".equals(apiAttachment.getRawData())) {
            //获取文件目录
            rawDataDigiwinAthena = JsonUtil.jsonStringToObject(apiAttachment.getRawData(),
                    AttachmentRawDataDigiwinAthena.class);
            uploadCategoryId = rawDataDigiwinAthena.getUploadCategoryId();
            uploadCategory = rawDataDigiwinAthena.getUploadCategory();
            if (StringUtils.isBlank(uploadCategoryId) && StringUtils.isNotBlank(uploadCategory)) {
                //根据目录名称获取目录id
                uploadCategoryId = digiwinDmcProxyService.getDirIdByName(uploadCategory, rawDataDigiwinAthena.getBuckets());
            } else {
                uploadCategoryId = DigiwinDmcProxyService.DEFAULT_DIR;
            }
        } else {
            //这里是写死的行事历url
            uploadCategoryId = digiwinDmcProxyService.getDirIdByName(CALENDAR_ATTACHMENT_DIR);
        }

        //上传文件，放到指定目录id
        try {
            stopWatch.start("upload file to Digiwin DMC");
            //先创建目录，再上传文件
            if (StringUtils.isNotBlank(uploadCategoryId)) {
                digiwinDmcProxyService.createDir(uploadCategoryId, rawDataDigiwinAthena.getUploadCategory(), rawDataDigiwinAthena.getBuckets());
            }
            logger.error("begin -- upload file info, contentType:{}", file.getContentType());

            String fileExtendName = file.getOriginalFilename().substring(file.getOriginalFilename().lastIndexOf(".") + 1);
            String contentType = ContentTypeEnum.getContentTypeByName(fileExtendName);
            if (StringUtils.isEmpty(contentType)) {
                contentType = file.getContentType();
            }
            FileInfo fileInfo = digiwinDmcProxyService.uploadFile(file.getBytes(),
                    file.getOriginalFilename(), contentType, uploadCategoryId, rawDataDigiwinAthena.getBuckets());
            logger.error("end -- upload file info, fileId:{}, contentType:{}", fileInfo.getId(), contentType);
            stopWatch.stop();
            String fileId = fileInfo.getId();
            stopWatch.start("get file shareurl from Digiwin DMC");
            stopWatch.stop();
            Boolean disableAam = rawDataDigiwinAthena.getDisableAam();
            if (uploadPC && !Optional.ofNullable(disableAam).orElse(true)) {
                stopWatch.start("upload file to Athena ATDM");
                //上传文件到Athena
                AttachmentMessage attachmentMessage = new AttachmentMessage();
                attachmentMessage.setId(fileId);
                attachmentMessage.setName(apiAttachment.getFileName());
                attachmentMessage.setSize(apiAttachment.getFileSize());
                attachmentMessage.setCategory(uploadCategory);
                attachmentMessage.setCategoryId(uploadCategoryId);
                attachmentMessage.setTenantId(apiAttachment.getTenantId());
                attachmentMessage.setProjectId(rawDataDigiwinAthena.getTmTaskId());
                attachmentMessage.setRowDataKey(rawDataDigiwinAthena.getRowDataKey());
                attachmentMessage.setTaskId(rawDataDigiwinAthena.getTmActivityId());
                AttachmentResponse attachmentResponse =
                        digiwinAamProxyService.uploadFile(attachmentMessage, apiAttachment.getIamUserToken());
                stopWatch.stop();
                if (attachmentResponse.getStatus() != 200) {
                    result.put("code", "-1");
                    result.put("message", attachmentResponse.getStatusDescription());
                    return result;
                }
            }

            result.put("fileId", fileId);
            result.put("uploadCategory", uploadCategory);
            result.put("uploadCategoryId", uploadCategoryId);
            result.put("code", "200");
//            result.put("file", file.getBytes());
            return result;
        } catch (IOException e) {
            e.printStackTrace();
            result.put("code", "-1");
            result.put("message", "File Read IO Exception!");
            return result;
        } finally {
            logger.debug("FileName:{}, FileSize:{} bytes\n{} ms\n{}",
                    file.getOriginalFilename(), file.getSize(), stopWatch.getTotalTimeMillis(), stopWatch.prettyPrint());
        }
    }

    @Override
    public Map<String, Object> uploadFile(String strBase64, String buckets, String uploadCategory,
                                          boolean disableAam, String iamUserToken, String tenantId) {
        Map<String, Object> result = new HashMap<>();
        if (StrUtil.isAllBlank(strBase64, buckets, uploadCategory)) {
            result.put("code", "-1");
            result.put("message", "请求数据有误");
            return result;
        }
        //解析base64
        byte[] decode = Base64.decode(strBase64);
        int fileSize = decode.length;
        if (decode == null || decode.length == 0) {
            result.put("code", "-1");
            result.put("message", "base64为空");
            return result;
        }
        //获取文件目录
        String uploadCategoryId = digiwinDmcProxyService.getDirIdByName(uploadCategory, buckets);
        StopWatch stopWatch = new StopWatch("uploadFile");
        //上传文件，放到指定目录id
        String originalFilename = UUID.randomUUID() + ".png";
        try {
            stopWatch.start("upload file to Digiwin DMC");
            //先创建目录，再上传文件
            if (StringUtils.isNotBlank(uploadCategoryId)) {
                digiwinDmcProxyService.createDir(uploadCategoryId, uploadCategory, buckets);
            }
            FileInfo fileInfo = digiwinDmcProxyService.uploadFile(decode,
                    originalFilename, null, uploadCategoryId, buckets);
            stopWatch.stop();
            String fileId = fileInfo.getId();
            stopWatch.start("get file shareurl from Digiwin DMC");
            stopWatch.stop();
            if (!Optional.ofNullable(disableAam).orElse(true) && StrUtil.isAllNotBlank(iamUserToken, tenantId)) {
                stopWatch.start("upload file to Athena ATDM");
                //上传文件到Athena
                AttachmentMessage attachmentMessage = new AttachmentMessage();
                attachmentMessage.setId(fileId);
                attachmentMessage.setName(originalFilename);
                attachmentMessage.setSize(fileSize);
                attachmentMessage.setCategory(uploadCategory);
                attachmentMessage.setCategoryId(uploadCategoryId);
                attachmentMessage.setTenantId(tenantId);
                AttachmentResponse attachmentResponse =
                        digiwinAamProxyService.uploadFile(attachmentMessage, iamUserToken);
                stopWatch.stop();
                if (attachmentResponse.getStatus() != 200) {
                    result.put("code", "-1");
                    result.put("message", attachmentResponse.getStatusDescription());
                    return result;
                }
            }
            logger.error("来来来，我们看看。fileId:{}", fileId);
            result.put("fileId", fileId);
            result.put("uploadCategory", uploadCategory);
            result.put("uploadCategoryId", uploadCategoryId);
            result.put("code", "200");
//            result.put("file", file.getBytes());
            return result;
        } catch (Exception e) {
            e.printStackTrace();
            result.put("code", "-1");
            result.put("message", "上传失败");
            return result;
        } finally {
            logger.debug("FileName:{}, FileSize:{} bytes\n{} ms\n{}",
                    originalFilename, fileSize, stopWatch.getTotalTimeMillis(), stopWatch.prettyPrint());
        }
    }

    @Override
    public Map<String, Object> uploadFile(UploadFile uploadFile) {
        Map<String, Object> result = new HashMap<>();
        if (null == uploadFile) {
            result.put("code", "-1");
            result.put("message", "请求数据有误");
            return result;
        }
        List<MultipartFile> files = uploadFile.getFiles();
        if (CollectionUtils.isEmpty(files)) {
            result.put("code", "-1");
            result.put("message", "请求数据有误");
            return result;
        }

        String dirName = "athenamobile_log_nfclogin";
        String dirId = digiwinDmcProxyService.getDirIdByName(dirName, DigiwinDmcConfig.ATHENA_BUCKET);

        List<FileUploadRecord> fileUploadRecordList = new ArrayList<>();
        for (MultipartFile file : files) {
            if (file == null) {
                result.put("code", "-1");
                result.put("message", "文件为空");
                return result;
            }
            try {
                if (StringUtils.isNotBlank(dirId)) {
                    // 先创建目录，再上传文件
                    digiwinDmcProxyService.createDir(dirId, dirName, DigiwinDmcConfig.ATHENA_BUCKET);
                }
                // 上传文件，放到指定目录id
                String fileId = digiwinDmcProxyService.uploadFile(file.getBytes(),
                        file.getOriginalFilename(), file.getContentType(), dirId, DigiwinDmcConfig.ATHENA_BUCKET).getId();

                //获取分享链接
                ShareInfo shareInfo = digiwinDmcProxyService.shareFile(fileId, DigiwinDmcConfig.ATHENA_BUCKET);

                fileUploadRecordList.add(new FileUploadRecord()
                        .setUserId(uploadFile.getUserId())
                        .setUserName(uploadFile.getUserName())
                        .setTenantId(uploadFile.getTenantId())
                        .setAppPackageName(uploadFile.getPackageName())
                        .setUploadLocation(UploadFileLocationEnum.DMC.getLocation())
                        .setDirName(dirName)
                        .setFileId(fileId)
                        .setFileName(file.getOriginalFilename())
                        .setOriginalFileName(file.getOriginalFilename())
                        .setFileUrl(shareInfo.getUrl())
                        .setFileSize(file.getSize())
                        .setUploadTime(new Date())
                        .setAppVersion(uploadFile.getVersion())
                        .setAppDeviceType(uploadFile.getDeviceType()));
            } catch (Exception e) {
                e.printStackTrace();
                logger.error("file upload error :", e);
                result.put("code", "-1");
                result.put("message", "File Read IO Exception!");
                return result;
            } finally {
                logger.debug("file upload success, FileName:{}, FileSize:{}", file.getOriginalFilename(), file.getSize());
            }
        }
        fileService.saveBatch(fileUploadRecordList);
        return result;
    }

    @Override
    public String segmentedUploadFile(ApiSegmentedUploadAttachment uploadFile) {
        String iamUserToken = uploadFile.getIamUserToken();
        String bucket = uploadFile.getBucket();
        Assert.isTrue(!uploadFile.getFile().isEmpty(), "MultipartFile cannot be null.");
        Assert.isTrue(StringUtils.isNotBlank(iamUserToken), "iamUserToken cannot be null.");
        Assert.isTrue(StringUtils.isNotBlank(bucket), "bucket cannot be null.");
        //获取空文件id
        SegmentedUploadFileDTO dto = new SegmentedUploadFileDTO();
        dto.setFileName(uploadFile.getFile().getOriginalFilename());
        dto.setDisplayName(uploadFile.getDisplayName());
        dto.setDescription(uploadFile.getDescription());
        dto.setTag(uploadFile.getTag());
        dto.setExpireDate(uploadFile.getExpireDate());
        dto.setDirectoryId(uploadFile.getDirectoryId());
        dto.setTenantId(uploadFile.getTenantId());
        String uploadId = digiwinDmcProxyService.segmentedUploadFile(dto, bucket, iamUserToken);
        if (null == uploadId) {
            return null;
        }
        long fileLength = uploadFile.getFile().getSize();
        int partCount = (int) Math.ceil((double) fileLength / FILE_UPLOAD_PART_SIZE);
        if (partCount > 10000) {
            throw new ServiceException("SegmentedUploadFile: Total parts count should not exceed 10000.");
        }
        return partUpload(uploadFile.getFile(), uploadId, partCount, fileLength, iamUserToken, bucket);
    }

    /**
     * 分段上传文件
     */
    private String partUpload(MultipartFile file, String fileId, int partCount, long fileLength, String token, String bucket) {
        try (FileChannel channel = new RandomAccessFile(FileUtil.multipartFileToFile(file), "r").getChannel()) {
            MappedByteBuffer map = channel.map(FileChannel.MapMode.READ_ONLY, 0, fileLength);
            for (int i = 0; i < partCount; i++) {
                int startPos = i * FILE_UPLOAD_PART_SIZE;
                int curPartSize = (i + 1 == partCount) ? (int) (fileLength - startPos) : FILE_UPLOAD_PART_SIZE;
                byte[] buffer = new byte[curPartSize];
                map.get(buffer, 0, curPartSize);
                Resource resource = new ByteArrayResource(buffer);
                digiwinDmcProxyService.segmentUploadCreateFile(resource, token, bucket, fileId, startPos, startPos + curPartSize - 1, (int) fileLength);
                logger.debug("总共：{}段, 第{}段上传成功：fileId={},startPos={},curPartSize={}.", partCount, (i + 1), fileId, startPos, curPartSize);
            }
            return fileId;
        } catch (Exception e) {
            logger.error("分段上传失败：fileId={},partSize={},partCount={},curPartSize={}.", fileId, FILE_UPLOAD_PART_SIZE, partCount, fileLength, e);
            return null;
        }
    }

    @Override
    public ApiResponse uploadFileAgileData(AgileDataUploadAttachment apiAttachment) {
        String iamUserToken = apiAttachment.getIamUserToken();
        String tenantId = apiAttachment.getTenantId();
        String directoryName = apiAttachment.getDirectoryName();
        String bucket = apiAttachment.getBucket();
        String locale = apiAttachment.getLocale();
        // 使用demo示例文件处理
        if (apiAttachment.getIfCommonFlag()) {
            String fileId = apiAttachment.getFileId();
            if (StringUtils.isBlank(fileId)) {
                return ApiResponse.buildError(localeService.getLanguageValue(locale, "文件id不能为空"));
            }
            return getAdtProcessMessageResponse(iamUserToken, tenantId, true, fileId, locale);
        }
        // 用户自己上传文件处理
        if (apiAttachment.getFile() == null) {
            return ApiResponse.buildError(localeService.getLanguageValue(locale, "文件不能为空"));
        }
        FileInfo info;
        try {
            // 获取文件目录
            String directoryId = digiwinDmcProxyService.getDirIdByName(directoryName, bucket);
            if (StringUtils.isBlank(directoryId)) {
                DirInfo dirInfo = digiwinDmcProxyService.createDir(directoryId, directoryName, bucket);
                directoryId = dirInfo.getId();
            }
            info = uploadDirFile(apiAttachment, directoryId, bucket);
            if (info == null) {
                return ApiResponse.buildError(localeService.getLanguageValue(locale, "文件上传失败"));
            }
        } catch (Exception e) {
            logger.error("敏捷数据Lite版上传文件失败, error:{}", e.getMessage(), e);
            return ApiResponse.buildError(localeService.getLanguageValue(locale, "文件上传失败"));
        }
        // 调用adt接口，透传文件url
        return getAdtProcessMessageResponse(iamUserToken, tenantId, false, info.getId(), locale);
    }

    @Override
    public List<Map<String, String>> getDirFileList(ApiRequest apiRequest) {
        String tenantId = apiRequest.getTenantId();
        String dirName = AiAssistantConstant.AGILE_DATA_DEMONSTRATION_TENANT_ID.equalsIgnoreCase(tenantId) ? "lite_executiveAssistantUploadFile" : "lite_attachment_demo";
        // 获取文件目录
        String directoryId = digiwinDmcProxyService.getDirIdByName(dirName, DigiwinDmcConfig.AGILE_DATA_BUCKET);
        if (StringUtils.isBlank(directoryId)) {
            return Collections.emptyList();
        }
        List<FileInfo> fileInfos;
        try {
            fileInfos = digiwinDmcProxyService.getFileInfos(directoryId, DigiwinDmcConfig.AGILE_DATA_BUCKET);
        } catch (Exception e) {
            logger.error("获取文件信息失败", e);
            return Collections.emptyList();
        }
        if (CollectionUtils.isEmpty(fileInfos)) {
            return Collections.emptyList();
        }
        return fileInfos.stream()
                .map(fileInfo -> {
                    Map<String, String> map = new HashMap<>();
                    map.put("previewFileUrl", getFileUrl(fileInfo.getId()));
                    map.put("fileStreamUrl", geFileStreamUrl(fileInfo.getId()));
                    map.put("fileId", fileInfo.getId());
                    map.put("name", fileInfo.getFileName());
                    return map;
                })
                .collect(Collectors.toList());
    }

    /**
     * 上传文件到指定目录
     *
     * @param apiAttachment
     * @param directoryId
     * @return info
     * @throws IOException
     */
    private FileInfo uploadDirFile(AgileDataUploadAttachment apiAttachment, String directoryId, String bucket) throws IOException {
        MultipartFile file = apiAttachment.getFile();
        FileInfo fileInfo = new FileInfo();
        fileInfo.setFileName(file.getOriginalFilename());
        fileInfo.setDisplayName(file.getOriginalFilename());
        fileInfo.setDescription(file.getOriginalFilename());
        fileInfo.setDirectoryId(directoryId);
        fileInfo.setTenantId(apiAttachment.getTenantId());
        fileInfo.setBucket(bucket);
        return digiwinDmcProxyService.uploadFile(file.getInputStream(), fileInfo);
    }

    /**
     * 敏捷数据:lite版文件推送处理接口
     *
     * @param iamUserToken
     * @param tenantId
     * @param fileId       文件id
     * @param ifCommonFlag 是否使用demo示例文件
     * @param locale       语言类别
     * @return ApiResponse
     * @author yanfeng
     */
    private ApiResponse getAdtProcessMessageResponse(String iamUserToken, String tenantId, Boolean ifCommonFlag, String fileId, String locale) {
        DigiwinAdtSubmitLiteProcessMessageResult adtSubmitLiteProcessMessageResult = new DigiwinAdtSubmitLiteProcessMessageResult();
        //如果是大陆正式区演示租户：高管AI数智助理(演示) 则直接返回死数据
        if (AiAssistantConstant.AGILE_DATA_DEMONSTRATION_TENANT_ID.equalsIgnoreCase(tenantId)) {
            handleDemonstrationTenant(locale, adtSubmitLiteProcessMessageResult);
        } else {
            // 调用adt接口，透传文件url
            adtSubmitLiteProcessMessageResult = digiwinAdtProxyService.submitLiteProcessMessage(iamUserToken, tenantId,
                    geFileStreamUrl(fileId), ifCommonFlag);
            if (adtSubmitLiteProcessMessageResult == null) {
                //adt接口调用失败，删除文件
                /*if (!ifCommonFlag) {
                    digiwinDmcProxyService.deleteFile(fileId, bucket);
                }*/
                return ApiResponse.buildError(localeService.getLanguageValue(locale, "服务异常,请稍后重试"));
            }
            adtSubmitLiteProcessMessageResult.setFileUrl(getFileUrl(fileId));
        }
        return ApiResponse.buildOK().setData(adtSubmitLiteProcessMessageResult);
    }

    /**
     * 处理敏捷数据大陆正式区演示租户：高管AI数智助理(演示) 使用上传Excel文件功能-死数据返回
     */
    private void handleDemonstrationTenant(String locale, DigiwinAdtSubmitLiteProcessMessageResult adtSubmitLiteProcessMessageResult) {
        adtSubmitLiteProcessMessageResult.setSendNana(true);
        adtSubmitLiteProcessMessageResult.setPrompt(localeService.getLanguageValue(locale, "你已成功上传Excel文件，请参考下方示例问句提问"));
        adtSubmitLiteProcessMessageResult.setSuccess(true);
        adtSubmitLiteProcessMessageResult.setMultiDialogue(false);
        adtSubmitLiteProcessMessageResult.setSentences(Arrays.asList(
                localeService.getLanguageValue(locale, "2024年预测金额的趋势"),
                localeService.getLanguageValue(locale, "2024年各个业务员的预测金额")));
    }

    /**
     * dmc文件流url
     *
     * @param fileId
     */
    private String geFileStreamUrl(String fileId) {
        return AppContext.getApiUrlSetting().getDmcUrl() + "/api/dmc/v2/file/" + DigiwinDmcConfig.AGILE_DATA_BUCKET + "/preview/" + fileId;
    }

    /**
     * dmc文件预览服务url
     *
     * @param fileId
     */
    private String getFileUrl(String fileId) {
        return AppContext.getApiUrlSetting().getDmcUrl() + "/api/dmc/v2/file/" + DigiwinDmcConfig.AGILE_DATA_BUCKET + "/online/preview/" + fileId;
    }

}