package com.digiwin.mobile.mobileuibot.locale.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.context.SpringContextHolder;
import com.digiwin.mobile.mobileuibot.locale.cache.LocaleCache;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.locale.service.TranslateService;
import com.digiwin.mobile.mobileuibot.mapper.db1.LanguageSourceMapper;
import com.digiwin.mobile.mobileuibot.model.db1.LanguageSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.util.List;

/**
 * @author zhangjj
 * @date 2022/11/9 10:12
 */
@Service
public class LocaleServiceImpl implements LocaleService {
    private static final Logger logger =
            LoggerFactory.getLogger(LocaleServiceImpl.class);

    @Autowired
    private LanguageSourceMapper languageSourceMapper;

    @Autowired
    private LocaleCache localeCache;

    @Autowired
    private TranslateService translateService;

    @Override
    public void cacheAllLanguageSourceList() {
        List<LanguageSource> languageSourceList = languageSourceMapper.selectList(new LambdaQueryWrapper<>());
        localeCache.putLanguageSourceList(languageSourceList);
    }

    @Override
    public String getLanguageValue(String phrase) {
        return SpringContextHolder.getBean(LocaleService.class)
                .getLanguageValue(AppRequestContext.getContextEntity().getLocale(), phrase);
    }

    @Override
    public String getLanguageValue(String locale, String phrase) {
        if (!StringUtils.hasLength(phrase)) {
            return phrase;
        }
        String sourceValue = localeCache.getLanguageCache(locale, phrase);
        if (StringUtils.hasLength(sourceValue)) {
            //缓存中有
            return sourceValue;
        }
        LambdaQueryWrapper<LanguageSource> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(LanguageSource::getLocale, locale);
        wrapper.eq(LanguageSource::getPhrase, phrase);
        LanguageSource dbLanguageSource = languageSourceMapper.selectOne(wrapper, false);
        if (dbLanguageSource != null) {
            //db中有
            localeCache.putLanguageSource(dbLanguageSource);
            return dbLanguageSource.getSourceValue();
        }
        //缓存和db中都没有
        LanguageSource zhCnlanguageSource = new LanguageSource(phrase, "zh_CN", phrase);
        addOrUpdateLanguageSource(zhCnlanguageSource);
        LanguageSource zhTwlanguageSource = new LanguageSource(phrase, "zh_TW", translateService.zh2Hant(phrase));
        addOrUpdateLanguageSource(zhTwlanguageSource);

        // 英文翻译通过luckyCola翻译 和百度翻译来查询，查不到返回繁体方式处理
        LanguageSource enUslanguageSource = null;
        if ("en_US".equalsIgnoreCase(locale)) {
            String transalateValueByEn = translateService.zh2EN(phrase);
            if (StringUtils.hasLength(transalateValueByEn)) {
                enUslanguageSource = new LanguageSource(phrase, "en_US", translateService.zh2EN(phrase));
            } else {
                enUslanguageSource = new LanguageSource(phrase, "en_US", zhTwlanguageSource.getSourceValue());
            }
            addOrUpdateLanguageSource(enUslanguageSource);
        }
        if ("zh_TW".equalsIgnoreCase(locale) && zhTwlanguageSource != null) {
            return zhTwlanguageSource.getSourceValue();
        }
        if ("en_US".equalsIgnoreCase(locale) && enUslanguageSource != null) {
            return enUslanguageSource.getSourceValue();
        }
        return phrase;
    }

    @Override
    public Boolean addOrUpdateLanguageSource(LanguageSource languageSource) {
        if (languageSource == null || !StringUtils.hasLength(languageSource.getPhrase())) {
            return false;
        }

        LambdaQueryWrapper<LanguageSource> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(LanguageSource::getLocale, languageSource.getLocale());
        wrapper.eq(LanguageSource::getPhrase, languageSource.getPhrase());

        LanguageSource dbLanguageSource = languageSourceMapper.selectOne(wrapper, false);
        boolean result;
        if (dbLanguageSource == null) {
            result = languageSourceMapper.insert(languageSource) > 0;
        } else {
            languageSource.setSourceId(dbLanguageSource.getSourceId());
            result = languageSourceMapper.updateById(languageSource) > 0;
        }
        if (result) {
            localeCache.putLanguageSource(languageSource);
            return true;
        }
        return false;
    }

}
