package com.digiwin.mobile.mobileuibot.locale.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.mobile.mobileuibot.cache.LuckyColaCache;
import com.digiwin.mobile.mobileuibot.common.calculate.UUIDUtil;
import com.digiwin.mobile.mobileuibot.common.http.HttpService;
import com.digiwin.mobile.mobileuibot.common.http.model.AthenaBasicHttpRequestHead;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.map.MapUtil;
import com.digiwin.mobile.mobileuibot.config.SysEnvConfig;
import com.digiwin.mobile.mobileuibot.locale.model.TranslateRequest;
import com.digiwin.mobile.mobileuibot.locale.model.TranslateResponse;
import com.digiwin.mobile.mobileuibot.locale.service.TranslateService;
import com.digiwin.service.ITranslateClientService;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.DigestUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author zhangjj
 * @date 2022/11/3 16:49
 */
@Service
public class TranslateServiceImpl implements TranslateService {

    private static final Logger logger = LoggerFactory.getLogger(TranslateServiceImpl.class);


    //简体中文转繁体中文
    private static final String ZH_CN_TO_ZH_TW = "zh2Hant";

    @Autowired
    private ITranslateClientService translateClient;

    @Autowired
    private SysEnvConfig sysEnvConfig;

    @Autowired
    private HttpService httpService;

    @Autowired
    private LuckyColaCache luckyColaCache;

    @Override
    public String zh2Hant(String value) {
        try {
            String resultJsonStr = translateClient.translate(value, new String[]{ZH_CN_TO_ZH_TW});
            JSONObject resultJson = JSONObject.parseObject(resultJsonStr);
            return (String) resultJson.get(ZH_CN_TO_ZH_TW);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return value;
    }

    @Override
    public String zh2EN(String value) {
        if (!StringUtils.hasLength(value)) {
            return value;
        }

        // 先通过luckyCola翻译获取英文
        // luckyCola无字符限制，只是频繁调用到达一定次数可能会失败，失败一定次数又正常
        String transalateValueByEn = luckyColaTransalateEn(value);

        // luckyCola获取失败，再通过百度翻译获取英文
        // 使用百度翻译标准版，每月5万字符限制， 可通过管理台查看使用情况 https://fanyi-api.baidu.com/api/trans/product/desktop?req=detail
        if (!StringUtils.hasLength(transalateValueByEn)) {
            transalateValueByEn = baiduTransalateEn(value);
        }
        return transalateValueByEn;
    }

    @Override
    public String getColaKey() {
        String url = "https://luckycola.com.cn/ai/getColaKey";
        Map<String, Object> param = new HashMap<>();
        param.put("uid", sysEnvConfig.getLuckycolaUserId());
        param.put("appKey", sysEnvConfig.getLuckycolaAppKey());
        ResponseEntity<Map> responseEntity =
                httpService.doPost(url, new AthenaBasicHttpRequestHead(), JsonUtil.javaObjectToJsonString(param), Map.class);
        if (0 == (Integer) responseEntity.getBody().get("code")) {
            Map<String, Object> data = (Map<String, Object>) responseEntity.getBody().get("data");
            return (String) MapUtil.getOrDefault(data, "cola_key", "");
        }
        return "";
    }

    public String luckyColaTransalateEn(String value) {
        String transalateValue = "";
        try {
            String url = "https://luckycola.com.cn/tools/fanyi";
            Map<String, Object> param = new HashMap<>();
            param.put("text", value);
            param.put("ColaKey", luckyColaCache.getColaKey());
            param.put("fromlang", "ZH");
            param.put("tolang", "EN");
            ResponseEntity<Map> responseEntity =
                    httpService.doPost(url, new AthenaBasicHttpRequestHead(), JsonUtil.javaObjectToJsonString(param), Map.class);
            if (0 == (Integer) responseEntity.getBody().get("code")) {
                Map<String, Object> data = (Map<String, Object>) responseEntity.getBody().get("data");
                transalateValue = (String) MapUtil.getOrDefault(data, "dst", "");
            }
            // colaKey无效,重新访问获取并刷新
            if (-99 == (Integer) responseEntity.getBody().get("code")) {
                logger.error("luckyCola colaKey is invalid, prepare refresh");
                String colaKey = getColaKey();
                luckyColaCache.putValue(colaKey);
            }
        } catch (Exception e) {
            logger.error("luckyCola translate cn to us failed");
        }
        return transalateValue;
    }

    public String baiduTransalateEn(String value) {
        String transalateValue = "";
        String appId = sysEnvConfig.getBaiduDeveloperAppId();
        String secretKey = sysEnvConfig.getBaiduDeveloperSecretKey();
        try {
            String uuid = UUIDUtil.getUuid();
            String secretData = appId + value + uuid + secretKey;
            String sign = DigestUtils.md5DigestAsHex(secretData.getBytes("utf-8"));

            StringBuffer baiduUrlBuffer = new StringBuffer();
            baiduUrlBuffer.append("http://api.fanyi.baidu.com/api/trans/vip/translate")
                    .append("?")
                    .append("q=").append(value)
                    .append("&from=zh")
                    .append("&to=en")
                    .append("&appid=").append(appId)
                    .append("&salt=").append(uuid)
                    .append("&sign=").append(sign);

            ResponseEntity<Map> responseEntity = httpService.doGet(baiduUrlBuffer.toString(), Map.class);
            Object transResultObj = responseEntity.getBody().get("trans_result");
            if (ObjectUtils.isNotEmpty(transResultObj)) {
                List<Map<String, Object>> transResult = (List<Map<String, Object>>) transResultObj;
                transalateValue = (String) MapUtil.getOrDefault(transResult.get(0), "dst", "");
            }
        } catch (Exception e) {
            logger.error("baidu translate cn to us failed");
        }
        return transalateValue;
    }


    @Override
    public TranslateResponse translate(TranslateRequest translateRequest) {
        TranslateResponse translateResponse = new TranslateResponse();
        String value = translateRequest.getContent();
        List<String> convertTypes = translateRequest.getConvertTypes();
        if (CollectionUtils.isEmpty(convertTypes)) {
            return translateResponse;
        }
        List<String> types = new ArrayList<>();
        if (convertTypes.contains("zh2Hant")) {
            types.add("zh2Hant");
        }
        if (convertTypes.contains("zh2Hans")) {
            types.add("zh2Hans");
        }
        if (CollectionUtils.isNotEmpty(types)) {
            try {
                String resultJsonStr = translateClient.translate(value, types.toArray(new String[0]));
                JSONObject resultJson = JSONObject.parseObject(resultJsonStr);
                translateResponse.setZh2Hant(resultJson.getString("zh2Hant"));
                translateResponse.setZh2Hans(resultJson.getString("zh2Hans"));
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        if (convertTypes.contains("zh2En")) {
            translateResponse.setZh2En(zh2EN(value));
        }
        return translateResponse;
    }
}
