package com.digiwin.mobile.mobileuibot.login;

import com.digiwin.mobile.mobileuibot.aiassistant.constant.AiAssistantConstant;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.crypto.aes.AESUtils;
import com.digiwin.mobile.mobileuibot.common.crypto.rsa.RSAUtils;
import com.digiwin.mobile.mobileuibot.experience.model.ExperienceRoleIdEnum;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.model.db1.Mock;
import com.digiwin.mobile.mobileuibot.proxy.cac.service.DigiwinCacProxyService;
import com.digiwin.mobile.mobileuibot.proxy.digiwhale.model.DigiwhaleLoginResponse;
import com.digiwin.mobile.mobileuibot.proxy.digiwhale.model.IamAuth;
import com.digiwin.mobile.mobileuibot.proxy.eoc.model.DigiwinEocAddUpdateEmp;
import com.digiwin.mobile.mobileuibot.proxy.eoc.model.DigiwinEocAddUpdateEmpDept;
import com.digiwin.mobile.mobileuibot.proxy.eoc.model.EocPerson;
import com.digiwin.mobile.mobileuibot.proxy.eoc.service.DigiwinEocProxyService;
import com.digiwin.mobile.mobileuibot.proxy.iam.model.DigiwinIamTokenRefreshTenant;
import com.digiwin.mobile.mobileuibot.proxy.iam.model.DigiwinIamUserAddJoinTenant;
import com.digiwin.mobile.mobileuibot.proxy.iam.service.DigiwinIamProxyService;
import io.micrometer.core.instrument.util.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.jetbrains.annotations.NotNull;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * <p>功能描述：登录服务实现类</p>
 *
 * @FileName: LoginServiceImpl
 * @Author: zaregoto
 * @Date: 2024/4/9 17:33
 */
@Slf4j
@Service("loginService")
public class LoginServiceImpl implements LoginService {

    @Resource
    private DigiwinIamProxyService digiwinIamProxyService;

    @Resource
    private DigiwinCacProxyService digiwinCacProxyService;

    @Resource
    private DigiwinEocProxyService digiwinEocProxyService;

    @Resource
    private LocaleService localeService;

    @Override
    public void handleUserByVerificationCodeInExperience(
            String mobilePhone, String verificationCode, String locale, String clientAgent, String expRoleId,
            DigiwhaleLoginResponse loginResponse, Mock mock, String iamDeviceId) throws Exception {
        IamAuth iamAuth = loginResponse.getIamAuth();
        String iamUserToken;
        String userId;
        if (Objects.isNull(iamAuth)) {
            iamAuth = new IamAuth();
            iamAuth.setSid(loginResponse.getSid());
            iamAuth.setUserId(loginResponse.getUserId());
            iamAuth.setUserName(loginResponse.getUserName());
            loginResponse.setIamAuth(iamAuth);
            iamUserToken = loginResponse.getToken();
            userId = loginResponse.getUserId();
        } else {
            iamUserToken = iamAuth.getToken();
            userId = iamAuth.getUserId();
        }

        /**
         * 1. 设置用户默认偏好语言
         * 2. 加入租户
         * 3. 切换到目标租户并刷新token
         * 4. 校验是否加入成功
         * 5. 授权租户用户商品
         * 6. 设置当前租户为预设租户
         * 7. 新增员工并绑定
         * 8. 登录超管账号，并赋予当前用户指定角色
         */
        // TODO 失败错误处理及提示
        // 设置当前用户的默认偏好语言
        boolean setDefaultLanguageResult = this.digiwinIamProxyService.setUserDefaultAcceptLanguage(
                iamUserToken, 0L, iamAuth.getSid(), locale);

        DigiwinIamUserAddJoinTenant joinTenantResult = this.addUserToTargetTenantInExperience(iamUserToken, expRoleId, mock);
        DigiwinIamTokenRefreshTenant refreshTenantResult = this.digiwinIamProxyService.tokenRefreshTenant(joinTenantResult.getTenantId(), "", iamUserToken);
        if (null != refreshTenantResult.getSuccess() && !refreshTenantResult.getSuccess()) {
            // 未成功切换租户的错误处理：如用户被停用
            this.buildError(loginResponse,
                    localeService.getLanguageValue(locale, "账号已被停用，登录失败。" + refreshTenantResult.getErrorMessage()));
            return;
        }
        // 刷新后的结果更新到登录结果
        iamAuth.setTenantSid(refreshTenantResult.getTenantSid());
        iamAuth.setTenantId(refreshTenantResult.getTenantId());
        iamAuth.setTenantName(refreshTenantResult.getTenantName());
        iamAuth.setToken(refreshTenantResult.getUserToken());

        // 重新使用新的token
        iamUserToken = iamAuth.getToken();
        //重新刷新租户Id
        loginResponse.setAccessToken(iamAuth.getToken());
        loginResponse.setTenantId(refreshTenantResult.getTenantId());
        loginResponse.setTenantSid(refreshTenantResult.getTenantSid());
        loginResponse.setTenantName(refreshTenantResult.getTenantName());


        if (!this.digiwinIamProxyService.checkUserInTenant(userId, iamUserToken)) {
            // 未成功加入租户的错误处理
            this.buildError(loginResponse, localeService.getLanguageValue(locale, "未能加入租户，请检查租户或用户是否有效"));
            return;
        }

        boolean countIdAddResult = this.addUserToCountingInExperience(joinTenantResult.getTenantId(), userId, expRoleId, mock);
        if (!countIdAddResult) {
            // 未成功完成授权的错误处理
            this.buildError(loginResponse, localeService.getLanguageValue(locale, "未成功完成默认授权，请检查应用商品、租户或用户是否有效"));
            return;
        }

        // 设置当前的目标租户为当前用户的预设租户，防止有多租户时租户随机切换。但预设失败不影响主流程，故不中止
        boolean setDefaultTenantResult = this.digiwinIamProxyService.setUserDefaultTenant(iamUserToken, joinTenantResult.getTenantId(), "");

        EocPerson eocPersonInfo = digiwinEocProxyService.getEocInfo(mobilePhone, iamUserToken);
        if (Objects.isNull(eocPersonInfo)) {
            // 新增员工并绑定
            DigiwinEocAddUpdateEmp empInfo = this.createDigiwinEocAddUpdateEmpInExperience(locale, expRoleId, mobilePhone, iamAuth, mock);
            boolean addResult = this.digiwinEocProxyService.addOrUpdateEocEmp(locale, userId, iamUserToken, empInfo);
            if (!addResult) {
                // 未成功绑定租户员工的错误处理
                this.buildError(loginResponse, localeService.getLanguageValue(locale, "未成功绑定租户员工，请检查租户或用户是否有效"));
                return;
            }
        }

        // 先用超管账号登录，再赋予当前普通用户以超管角色
        DigiwhaleLoginResponse superadminLoginResponse = this.loginWithSuperAdminInExperience(locale, clientAgent,
                expRoleId, joinTenantResult.getTenantId(), mock, iamDeviceId);
        if (null == superadminLoginResponse) {
            // 未成功绑定角色的错误
            this.buildError(loginResponse, localeService.getLanguageValue(locale, "未成功获取系统管理角色，请检查角色、租户或用户是否有效"));
            return;
        }

        boolean roleAddResult = false;
        if (superadminLoginResponse != null) {
            roleAddResult = this.digiwinIamProxyService.associationRoleAddUsers(
                    superadminLoginResponse.getIamAuth().getToken(), getRoleId(joinTenantResult.getTenantId()), Collections.singletonList(userId));
        }
        if (!roleAddResult) {
            // 未成功绑定角色的错误
            this.buildError(loginResponse, localeService.getLanguageValue(locale, "未成功设置用户角色，请检查角色、租户或用户是否有效"));
            return;
        }
    }

    /**
     * 获取角色Id
     * @param tenantId
     * @return
     */
    private String getRoleId(String tenantId) {
        String userRoleId = "superadmin";

        String athenaMuiUrl = AppContext.getApiUrlSetting().getMuiUrl().trim();
        if (Objects.equals("https://athena-test.digiwincloud.com.cn", athenaMuiUrl)) {
            if ("AthenaNjTest".equals(tenantId)) {
                userRoleId = "TestRole";
            }
        }
        if (Objects.equals("https://athena.digiwincloud.com.cn", athenaMuiUrl)) {
            //华为生产敏捷数据-数据集2.0演示,租户：DigiwinGptDemo
            if (AiAssistantConstant.AGILE_DATA_HW_LITE_TENANT_ID.equals(tenantId)) {
                userRoleId = "DigiwinGptDemoRole";
            }
        }
        return userRoleId;
    }

    /**
     * 华为生产敏捷数据-数据集2.0演示,租户：DigiwinGptDemo
     * @return
     */
    @NotNull
    private static List<DigiwinEocAddUpdateEmpDept> getDigiwinEocAddUpdateEmpDeptsPro() {
        List<DigiwinEocAddUpdateEmpDept> deptList = new ArrayList<>();
        DigiwinEocAddUpdateEmpDept dept = new DigiwinEocAddUpdateEmpDept();
        dept.setSid(926947488722944L);
        dept.setId("A00000");
        dept.setName("總部(虛擬部門)");
        dept.setMainDept(true);
        dept.setCorpId("DigiwinGptDemo");
        dept.setCorpName("GPT生成式问数");
        dept.setLevelSid(1926947487748096L);
        dept.setDirectorId("");
        dept.setDirectorName("");
        deptList.add(dept);
        return deptList;
    }

    /**
     * 华为Test区敏捷数据-数据集2.0演示,租户：DigiwinGptDemo
     * @return
     */
    @NotNull
    private static List<DigiwinEocAddUpdateEmpDept> getDigiwinEocAddUpdateEmpDeptsTest() {
        List<DigiwinEocAddUpdateEmpDept> deptList = new ArrayList<>();
        DigiwinEocAddUpdateEmpDept dept = new DigiwinEocAddUpdateEmpDept();
        dept.setSid(1118086959779840L);
        dept.setId("test01");
        dept.setName("test01");
        dept.setMainDept(true);
        dept.setCorpId("nj01");
        dept.setCorpName("nj01");
        dept.setLevelSid(1113748446408704L);
        dept.setDirectorId("");
        dept.setDirectorName("");
        deptList.add(dept);
        return deptList;
    }

    private DigiwhaleLoginResponse buildError(DigiwhaleLoginResponse loginResponse, String errorMessage) {
        loginResponse.setError(HttpStatus.UNAUTHORIZED.value() + "");
        loginResponse.setErrorDescription(errorMessage);
        return loginResponse;
    }

    /**
     * 在体验模式下，按体验角色id，将用户添加到目标租户下
     *
     * @param iamUserToken
     * @param expRoleId
     * @return
     */
    private DigiwinIamUserAddJoinTenant addUserToTargetTenantInExperience(String iamUserToken, String expRoleId, Mock mock) {
        String tenantId = "";
        if (mock != null) {
            // mock集成娜娜功能场景
            tenantId = mock.getNanaTenantId();
        } else {
            // 依据雅典娜平台的地址，决定目标租户id
            String athenaMuiUrl = AppContext.getApiUrlSetting().getMuiUrl().trim();
            if (Objects.equals("https://athena-paas.digiwincloud.com.cn", athenaMuiUrl)) {
                // paas区
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_CXO.getValue())) {
                    tenantId = "DigiwinGptDemo";
                }
            } else if (Objects.equals("https://athena-test.digiwincloud.com.cn", athenaMuiUrl)) {
                // 华为测试区
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_CXO.getValue())) {
                    tenantId = "AthenaNjTest";
                }
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_TRAVEL.getValue())) {
                    tenantId = "erp2nanatrmdemo";
                } else if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_ASAAA.getValue())) {
                    tenantId = "0000011111";
                }
            } else if (Objects.equals("https://athena.digiwincloud.com.cn", athenaMuiUrl)) {
                // 华为正式区
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_CXO.getValue())) {
                    tenantId = "DigiwinGptDemo";
                }
            } else if (Objects.equals("https://athena-test.digiwincloud.com", athenaMuiUrl)) {
                // 微软测试区
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_CXO.getValue())) {
                    tenantId = "DigiwinGptDemo";
                } else if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_TRAVEL.getValue())) {
                    tenantId = "erp2nanatrmdemo";
                }
            } else if (Objects.equals("https://athena.digiwincloud.com", athenaMuiUrl)) {
                // 微软正式区
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_CXO.getValue())) {
                    tenantId = "agiledemo";
                } else if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_TRAVEL.getValue())) {
                    tenantId = "erp2nanatrmdemo";
                } else if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_ASAAA.getValue())) {
                    tenantId = "athena";
                }
            }
        }

        DigiwinIamUserAddJoinTenant joinTenantResult = this.digiwinIamProxyService.userAddJoinTenant(
                iamUserToken, tenantId, "", "", "");
        joinTenantResult.setTenantId(tenantId);

        return joinTenantResult;
    }

    /**
     * 在体验模式下，创建一个企业员工的并绑定
     *
     * @param locale      语言别
     * @param expRoleId   体验角色id
     * @param mobilePhone 手机号。作为员工id
     * @param iamAuth
     * @return
     */
    private DigiwinEocAddUpdateEmp createDigiwinEocAddUpdateEmpInExperience(String locale, String expRoleId, String mobilePhone, IamAuth iamAuth, Mock mock) {
        DigiwinEocAddUpdateEmp digiwinEocAddUpdateEmp = new DigiwinEocAddUpdateEmp();
        String replacerId = "";
        if (mock != null) {
            // mock集成娜娜功能场景
            replacerId = "";
        } else {
            // 依据雅典娜平台的地址，决定员工账号的交接人（若不填交接后续无法在界面上方便停用账号）
            String athenaMuiUrl = AppContext.getApiUrlSetting().getMuiUrl().trim();
            if (Objects.equals("https://athena-paas.digiwincloud.com.cn", athenaMuiUrl)) {
                // paas区 无需求不处理
            } else if (Objects.equals("https://athena-test.digiwincloud.com.cn", athenaMuiUrl)) {
                //华为Test区敏捷数据-数据集2.0演示,租户：DigiwinGptDemo,要加入部门，否则app登录进去会报重要设定提醒
                if ("AthenaNjTest".equals(iamAuth.getTenantId())) {
                    digiwinEocAddUpdateEmp.setDepts(getDigiwinEocAddUpdateEmpDeptsTest());
                }
            } else if (Objects.equals("https://athena.digiwincloud.com.cn", athenaMuiUrl)) {
                //华为生产区敏捷数据-数据集2.0演示,租户：DigiwinGptDemo,要加入部门，否则app登录进去会报重要设定提醒
                if (AiAssistantConstant.AGILE_DATA_HW_LITE_TENANT_ID.equals(iamAuth.getTenantId())) {
                    digiwinEocAddUpdateEmp.setDepts(getDigiwinEocAddUpdateEmpDeptsPro());
                }
            } else if (Objects.equals("https://athena-test.digiwincloud.com", athenaMuiUrl)) {
                // 微软测试区 无需求不处理
            } else if (Objects.equals("https://athena.digiwincloud.com", athenaMuiUrl)) {
                // 微软正式区 TODO
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_CXO.getValue())) {
                    replacerId = "";
                }
            } else if (Objects.equals("https://athena-hz-test.digiwincloud.com.cn", athenaMuiUrl)) {
                // 湖州环境
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_CXO.getValue())) {
                    replacerId = "pur001";
                }
            } else {
                // TODO 未来定制或其他特殊情况
            }
        }

        //line登录手机号为邮箱，所以取@前面的,防止eoc报错
        if (mobilePhone.contains("@")) {
            int atIndex = mobilePhone.indexOf("@");
            mobilePhone = mobilePhone.substring(0, atIndex);
        }
        return digiwinEocAddUpdateEmp
                /**
                 * 员工id只能是字母、数字、下划线(_)和减号(-)
                 * 故直接用手机号
                 */
                .setId(mobilePhone)
                .setName(iamAuth.getUserName())
                .setUserId(iamAuth.getUserId())
                .setUserName(iamAuth.getUserName())
                .setStatus(true)
                .setReplacerId(replacerId);
    }

    /**
     * 在体验模式下，对租户+用户进行商品授权。
     * 若expRoleId=CXO,则授权的商品为四个：asada（数据助理）、dcdp（数采平台）、dcc（数据能力中心）、A20240412173606-bmdDev（高管数智助理）
     * --注：高管数智助理这个应用，目前仅需在台湾的微软正式区环境中需要处理。
     * 若expRoleId=TRAVEL,则授权的商品为两个：trmnana、ASA
     *
     * @param tenantId  租户id
     * @param userId    账号id
     * @param expRoleId 体验角色id
     * @return
     */
    private Boolean addUserToCountingInExperience(String tenantId, String userId, String expRoleId, Mock mock) {
        if (mock != null) {
            // mock集成娜娜功能场景
            boolean result = true;
            String code = mock.getNanaAuthorizationCode();
            if (org.springframework.util.StringUtils.hasLength(code)) {
                String[] split = code.split(",");
                for (String s : split) {
                    result = result && this.digiwinCacProxyService.countingUserAdd(s, tenantId, userId);
                }
            }
            return result;
        }
        String athenaMuiUrl = AppContext.getApiUrlSetting().getMuiUrl().trim();
        if (expRoleId.trim().equals(ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_CXO.getValue())) {
            if (Objects.equals("https://athena-test.digiwincloud.com", athenaMuiUrl)) {
                // 微软测试区因环境原因，没有应用授权等，故不处理授权，直接返回true
                return true;
            }

            //华为Test 数据集2.0演示,租户：DigiwinGptDemo,授权应用以下应用:
            if (Objects.equals("https://athena-test.digiwincloud.com.cn", athenaMuiUrl)) {
                //数据中台(dmp)
                boolean dmpCountingAddResult = this.digiwinCacProxyService.countingUserAdd("dmp", tenantId, userId);
                // 大数据采集平台
                boolean dcdpResult = this.digiwinCacProxyService.countingUserAdd("dcdp", tenantId, userId);
                //陈言测试2.0
                boolean agileDataCountingAddResult = this.digiwinCacProxyService.countingUserAdd("Mfe0920564AT", tenantId, userId);
                return dmpCountingAddResult && dcdpResult && agileDataCountingAddResult;
            }

            /**
             * 台湾的体验环境在微软正式区的敏捷数据体验租户，有高管数智助理的应用授权，需要一同判断。
             * 大陆的体验环境在阿里测试区的湖州高管数智AI助理租户，没有高管数智助理的应用授权，不需要判断。FIXME 若后面加了授权，这边代码需要改。
             * 添加流量包权限。
             */
            if (Objects.equals("https://athena.digiwincloud.com", athenaMuiUrl)) {
                //数据助理
                boolean asadaCountingAddResult = this.digiwinCacProxyService.countingUserAdd("asada", tenantId, userId);
                // 大数据采集平台
                boolean dcdpCountingAddResult = this.digiwinCacProxyService.countingUserAdd("dcdp", tenantId, userId);
                // 数据能力中心
                boolean dccCountingAddResult = this.digiwinCacProxyService.countingUserAdd("dcc", tenantId, userId);
                //高管数智助理
                boolean cxoAssistantCountingAddResult = this.digiwinCacProxyService.countingUserAdd("A20240412173606-bmdDev", tenantId, userId);
                //數智特助-對話流量包
                boolean trafficPackageCountingAddResult = this.digiwinCacProxyService.countingUserAdd("1216700ALL000001", tenantId, userId);
                //2024.9.11因仁杰移动chatFile需求特加此三应用授权
                //2024.10.16 发生问题:登录不进去了，原因是kmo应用U数满了，需要去鼎捷云调整应用的U数
                boolean kcfCountingAddResult = this.digiwinCacProxyService.countingUserAdd("KCF", tenantId, userId);
                boolean kmoCountingAddResult = this.digiwinCacProxyService.countingUserAdd("KMO", tenantId, userId);
                boolean kbsCountingAddResult = this.digiwinCacProxyService.countingUserAdd("KBS", tenantId, userId);
                return asadaCountingAddResult && dcdpCountingAddResult
                        && dccCountingAddResult && cxoAssistantCountingAddResult && trafficPackageCountingAddResult
                        && kcfCountingAddResult && kmoCountingAddResult && kbsCountingAddResult;
            }

            //华为生产 数据集2.0演示,租户：DigiwinGptDemo,授权应用以下应用:
            if (Objects.equals("https://athena.digiwincloud.com.cn", athenaMuiUrl)) {
                //数据中台(dmp)
                boolean dmpCountingAddResult = this.digiwinCacProxyService.countingUserAdd("dmp", tenantId, userId);
                // 大数据采集平台
                boolean dcdpResult = this.digiwinCacProxyService.countingUserAdd("dcdp", tenantId, userId);
                //旗舰版体验
                boolean agileDataCountingAddResult = this.digiwinCacProxyService.countingUserAdd("Ma7fab895fAT", tenantId, userId);
                return dmpCountingAddResult && dcdpResult && agileDataCountingAddResult;
            }

        } else if (expRoleId.trim().equals(ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_TRAVEL.getValue())) {
            if (Objects.equals("https://athena-test.digiwincloud.com", athenaMuiUrl)) {
                // 微软测试区因环境原因，没有应用授权等，故不处理授权，直接返回true
                return true;
            }
            boolean trmnanaCountingAddResult = this.digiwinCacProxyService.countingUserAdd("trmnana", tenantId, userId);
            boolean asaCountingAddResult = this.digiwinCacProxyService.countingUserAdd("ASA", tenantId, userId);
            return trmnanaCountingAddResult && asaCountingAddResult;
        } else if (expRoleId.trim().equals(ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_ASAAA.getValue())) {
            if (Objects.equals("https://athena-test.digiwincloud.com", athenaMuiUrl)) {
                // 微软测试区因环境原因，没有应用授权等，故不处理授权，直接返回true
                return true;
            }
            boolean trmnanaCountingAddResult = this.digiwinCacProxyService.countingUserAdd("asaaa", tenantId, userId);
            boolean asaCountingAddResult = this.digiwinCacProxyService.countingUserAdd("ASA", tenantId, userId);
            return trmnanaCountingAddResult && asaCountingAddResult;
        }
        return false;
    }

    private String getEncryptPublicKey() throws Exception {
        // 获取服务端公钥
        String serverPublicKey = digiwinIamProxyService.doGetPublicKey();
        if (StringUtils.isEmpty(serverPublicKey)) {
            log.error("未拿到互联的公钥。。。。");
        } else {
            log.error("拿到互联公钥为：" + serverPublicKey);
        }
        // 根据服务端公钥加密客户端公钥
        return RSAUtils.encryptByPublicKey(RSAUtils.CLIENT_PUBLIC_KEY, serverPublicKey);
    }

    /**
     * RSA+AES加密密码
     *
     * @param userName
     * @param encryptPublicKey 加密后的公钥
     * @return
     * @throws Exception
     */
    private String getEncryptPwd(String userName, String encryptPublicKey) throws Exception {
        // 获取加密后的AES的key值
        String aesPrivateKey = digiwinIamProxyService.doGetAesKey(encryptPublicKey);
        // 根据客户端私钥解密 加密的aes的key值
        String aesKey = new String(RSAUtils.decryptByPrivateKey(Base64.decodeBase64(aesPrivateKey), RSAUtils.CLIENT_PRIVATE_KEY));
        // 拿解密后的AES的key值对密码进行aes加密
        return AESUtils.aesEncryptByBase64(userName, aesKey);
    }

    /**
     * 在体验模式下，使用一个环境有关的固定超管账号登录，并返回响应内容
     *
     * @param locale         语言别
     * @param clientAgent    登录客户端信息
     * @param expRoleId      体验角色id
     * @param targetTenantId 目标租户id
     * @return
     */
    private DigiwhaleLoginResponse loginWithSuperAdminInExperience(String locale, String clientAgent,
                                                                   String expRoleId, String targetTenantId, Mock mock, String iamDeviceID) throws Exception {
        String userId = "";
        String password = "";

        if (mock != null) {
            // mock集成娜娜功能场景
            userId = mock.getNanaAdminAccount();
            password = mock.getNanaAdminPassword();
        } else {
            String athenaMuiUrl = AppContext.getApiUrlSetting().getMuiUrl().trim();
            if (Objects.equals("https://athena-paas.digiwincloud.com.cn", athenaMuiUrl)) {
                // paas区
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_CXO.getValue())) {
                    userId = "DigiwinGpt123";
                    password = "DigiwinGpt123";
                }
            } else if (Objects.equals("https://athena-test.digiwincloud.com.cn", athenaMuiUrl)) {
                // 华为测试区
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_CXO.getValue())) {
                    if ("AthenaNjTest".equals(targetTenantId)) {
                        userId = "AthenaNjTest";
                        password = "72282866Digiwin";
                    } else {
                        userId = "DigiwinGpt123";
                        password = "DigiwinGpt123";
                    }

                }
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_TRAVEL.getValue())) {
                    userId = "demohhb123";
                    password = "Demo1234";
                } else if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_ASAAA.getValue())) {
                    userId = "metis001";
                    password = "Metis0001";
                }
            } else if (Objects.equals("https://athena.digiwincloud.com.cn", athenaMuiUrl)) {
                // 华为正式区
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_CXO.getValue())) {
                    userId = "DigiwinGptDemo";
                    password = "DigiwinGptDemo123";
                }
            } else if (Objects.equals("https://athena-test.digiwincloud.com", athenaMuiUrl)) {
                // 微软测试区
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_TRAVEL.getValue())) {
                    userId = "demohhb";
                    password = "Demohhb123";
                } else {
                    // 账号所有者：DAP-林丽
                    userId = "hawkeye";
                    password = "123abcABC";
                }
            } else if (Objects.equals("https://athena.digiwincloud.com", athenaMuiUrl)) {
                // 微软正式区 TODO
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_CXO.getValue())) {
                    // 账号所有者：凌翔
                    userId = "lingxiang202401@gmail.com";
                    password = "Digiwin2024";
                } else if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_TRAVEL.getValue())) {
                    userId = "demohhb";
                    password = "Demohhb123";
                } else if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_ASAAA.getValue())) {
                    userId = "metis001";
                    password = "Metis0001";
                }
            } else if (Objects.equals("https://athena-hz-test.digiwincloud.com.cn", athenaMuiUrl)) {
                // 湖州环境
                if (Objects.equals(expRoleId.trim(), ExperienceRoleIdEnum.NANA_DATA_ASSISTANT_CXO.getValue())) {
                    userId = "qcuser001";
                    password = "qcuser001";
                }
            } else {
                // TODO 未来定制或其他特殊情况
            }
        }
        String encryptPublicKey = getEncryptPublicKey();
        ResponseEntity<DigiwhaleLoginResponse> superadminLoginResponseEntity = this.digiwinIamProxyService.userLoginByEncodePassword(
                encryptPublicKey, userId, getEncryptPwd(password, encryptPublicKey),
                locale, targetTenantId, clientAgent, null, iamDeviceID);

        return superadminLoginResponseEntity.getStatusCode().is2xxSuccessful() ? superadminLoginResponseEntity.getBody() : null;
    }
}