package com.digiwin.mobile.mobileuibot.mock;

import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.config.SysEnvConfig;
import com.digiwin.mobile.mobileuibot.core.pagesetting.PageSettingIdPresetEnum;
import com.digiwin.mobile.mobileuibot.environment.EnvironmentConfig;
import com.digiwin.mobile.mobileuibot.environment.EnvironmentConfigService;
import com.digiwin.mobile.mobileuibot.environment.config.EnvConfigModel;
import com.digiwin.mobile.mobileuibot.environment.config.EnvServerConfig;
import com.digiwin.mobile.mobileuibot.experience.model.ExperienceRoleIdEnum;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.mock.enums.EnableEnum;
import com.digiwin.mobile.mobileuibot.mock.enums.MockDeployEnvironmentEnum;
import com.digiwin.mobile.mobileuibot.mock.enums.PublishEnum;
import com.digiwin.mobile.mobileuibot.mock.model.MockData;
import com.digiwin.mobile.mobileuibot.mock.request.MockVO;
import com.digiwin.mobile.mobileuibot.mock.v2.MockV2Model;
import com.digiwin.mobile.mobileuibot.mock.v2.MockV2Service;
import com.digiwin.mobile.mobileuibot.model.db1.Mock;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * mock数据处理controller
 *
 * @author zhangjj
 * @date 2022/8/30 17:35
 */
@RestController
@RequestMapping("/mobile/v1/mock")
public class MockController {

    @Autowired
    private MockService mockService;

    @Autowired
    private LocaleService localeService;

    @Autowired
    private EnvironmentConfigService environmentConfigService;

    @Autowired
    private MockV2Service mockV2Service;

    @Autowired
    private SysEnvConfig sysEnvConfig;

    /**
     * mock数据处理
     *
     * @param request
     * @return
     */
    @PostMapping(value = "/data/post")
    public ApiResponse mockPostData(HttpServletRequest request) {
        return handleMockData(request);
    }

    @GetMapping(value = "/data/get")
    public ApiResponse mockGetData(HttpServletRequest request) {
        return handleMockData(request);
    }

    ApiResponse handleMockData(HttpServletRequest request) {
        MockData mockData = (MockData) request.getAttribute("mockData");
        String roleCode = AppRequestContext.getContextEntity().getExperienceRoleId();
        // 为大陆安卓应用商店可成功上架做特殊处理。如果是工作台，则使用项目经理角色的Demo体验数据，保证工作台内容更真实些
        if (Objects.equals(PageSettingIdPresetEnum.DASHBOARD.toString(), mockData.getPageId())) {
            roleCode = AppRequestContext.REQUEST_CONTEXT_VALUE_EXPERIENCE_MODE_ROLECODE_PROJECT_MANAGER;
        }
        String responseBodyStr = JsonUtil.javaObjectToJsonString(mockData.getResponseBody(roleCode));
        if (StringUtils.hasLength(responseBodyStr)) {
            responseBodyStr = responseBodyStr.replaceAll("\\$\\{BASE_URL\\}", AppContext.getBaseUrl());
            responseBodyStr = responseBodyStr.replace("\"userId\":\"dcp001\"", "\"userId\":\"" + AppRequestContext.getContextEntity().getUserId() + "\"");
            responseBodyStr = responseBodyStr.replace("userId=dcp001", "userId=" + AppRequestContext.getContextEntity().getUserId());
            responseBodyStr = responseBodyStr.replace("/mobile/v1/uibot/component?clientId=454886b20b5b7f91&tenantId=DCP", "/mobile/v1/uibot/component?clientId=454886b20b5b7f91&tenantId=DCP&userId=" + AppRequestContext.getContextEntity().getUserId());
        }
        return ApiResponse.buildOK().setData(JsonUtil.jsonStringToObject(responseBodyStr, Object.class));
    }

    @PostMapping(value = "/list")
    public ApiResponse<List<Mock>> mockList(@RequestBody MockVO vo) {
        String recordUserId = vo.getRecordUserId();
        boolean isMockSuperAdmin = this.checkIsMockSuperAdmin(recordUserId);
        List<Mock> mockList = mockService.mockList(recordUserId, isMockSuperAdmin);
        String envDeployEnvironment = sysEnvConfig.getEnvDeployEnvironment();
        String description = MockDeployEnvironmentEnum.getDescriptionByKey(envDeployEnvironment);
        List<Mock> list = Optional.ofNullable(mockList).orElse(Collections.emptyList());
        list.forEach(mock -> {
            mock.setDeployArea(description);
            if (EnableEnum.YES.getValue().equals(mock.getEnableNana())) {
                mock.setMockRoleId(ExperienceRoleIdEnum.ATHENA_MOCK.getValue());
            }
        });
        return ApiResponse.buildOK().setData(list);
    }

    @PostMapping(value = "/select")
    public ApiResponse<Mock> select(@RequestBody MockVO vo) {
        String mockId = vo.getMockId();
        String recordUserId = vo.getRecordUserId();
        String locale = vo.getLocale();
        Mock mock = mockService.selectByMockId(mockId, recordUserId);
        if (null == mock) {
            return ApiResponse.buildError(localeService.getLanguageValue(locale, "mock记录不存在"));
        }
        return ApiResponse.buildOK().setData(mock);
    }

    @PostMapping(value = "/delete")
    public ApiResponse<String> delete(@RequestBody MockVO vo) {
        String mockId = vo.getMockId();
        String recordUserId = vo.getRecordUserId();
        String locale = vo.getLocale();
        Mock mock = mockService.selectByMockId(mockId, recordUserId);
        if (null == mock) {
            return ApiResponse.buildError(localeService.getLanguageValue(locale, "mock记录不存在"));
        }
        if (PublishEnum.PUBLISH.getValue().equals(mock.getPublish())) {
            return ApiResponse.buildError(localeService.getLanguageValue(locale, "已发布状态不能删除"));
        }
        mockService.delete(mockId, recordUserId);
        return ApiResponse.buildOK();
    }

    @PostMapping(value = "/publish")
    public ApiResponse<String> publish(@RequestBody MockVO vo) {
        String mockId = vo.getMockId();
        String recordUserId = vo.getRecordUserId();
        String locale = vo.getLocale();
        Integer publish = vo.getPublish();
        Mock mock = mockService.selectByMockId(mockId, recordUserId);
        if (null == mock) {
            return ApiResponse.buildError(localeService.getLanguageValue(locale, "mock记录不存在"));
        }
        if (null == PublishEnum.exist(publish)) {
            return ApiResponse.buildError(localeService.getLanguageValue(locale, "请传入正确的参数值"));
        }
        mockService.publish(mockId, recordUserId, publish);
        return ApiResponse.buildOK();
    }

    @PostMapping(value = "/category/list")
    public ApiResponse<List<String>> categoryList(@RequestBody MockVO vo) {
        String locale = vo.getLocale();
        List<String> list = mockService.categoryList().stream()
                .map(s -> localeService.getLanguageValue(locale, s))
                .collect(Collectors.toList());
        return ApiResponse.buildOK().setData(list);
    }

    @PostMapping(value = "/start/record")
    public ApiResponse<MockVO> startRecord(@RequestBody MockVO vo) {
        if (StringUtils.hasLength(vo.getMockId())) {
            // 修改
            Mock mock = mockService.selectByMockId(vo.getMockId(), vo.getRecordUserId());
            if (null == mock) {
                return ApiResponse.buildError(localeService.getLanguageValue(vo.getLocale(), "mock记录不存在"));
            }
            if (PublishEnum.PUBLISH.getValue().equals(mock.getPublish())) {
                return ApiResponse.buildError(localeService.getLanguageValue(vo.getLocale(), "已发布状态不能修改"));
            }
        }
        return ApiResponse.buildOK().setData(mockService.startRecord(vo));
    }

    /**
     * 有管理者的權限，可管理所有的mock資料，例：修改、刪除等等
     *
     * @param recordUserId 录制人id
     * @return
     */
    private boolean checkIsMockSuperAdmin(String recordUserId) {
        List<EnvironmentConfig> environmentConfigs = environmentConfigService.getEnvironmentConfig();
        if (CollectionUtils.isEmpty(environmentConfigs)) {
            return false;
        }
        EnvironmentConfig environmentConfig = environmentConfigs.get(0);
        EnvConfigModel envConfigModel = environmentConfig.getConfig();
        if (null == envConfigModel) {
            return false;
        }
        EnvServerConfig serverConfig = envConfigModel.getServerConfig();
        if (null == serverConfig) {
            return false;
        }
        List<String> mockSuperAdmins = serverConfig.getMockSuperAdmins();
        return mockSuperAdmins.contains(recordUserId);
    }

    @PostMapping("/insertOrUpdateMockV2Data")
    public ApiResponse<String> insertOrUpdateMockV2Data(@RequestBody MockV2Model mockV2Model) {
        try {
            mockV2Service.insertOrUpdateMockV2Data(mockV2Model);
        } catch (ServiceException e) {
            return ApiResponse.buildError(e.getMessage());
        }
        return ApiResponse.buildOK();
    }
}
