package com.digiwin.mobile.mobileuibot.mongodb;

import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.mongodb.dto.MongoDbAddDTO;
import com.digiwin.mobile.mobileuibot.mongodb.dto.MongoDbDeleteDTO;
import com.digiwin.mobile.mobileuibot.mongodb.dto.MongoDbQueryDTO;
import com.digiwin.mobile.mobileuibot.mongodb.dto.MongoDbUpdateDTO;
import com.mongodb.BasicDBObject;
import com.mongodb.DBObject;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.bson.types.ObjectId;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * mongoDb
 *
 * @author yanfeng
 * @date 2024/01/31
 */
@RestController
@RequestMapping(value = "/mobile/v1/mongodb")
public class MongoDbController {

    @Autowired
    private MongoTemplate mongoTemplate;

    @Autowired
    @Qualifier("secondMongoTemplate")
    private MongoTemplate secondMongoTemplate;

    @GetMapping("/query")
    public ApiResponse query(@Valid MongoDbQueryDTO dto) {
        Query query = new Query();
        if (StringUtils.isNotBlank(dto.getId())) {
            query.addCriteria(Criteria.where("_id").is(dto.getId()));
        }
        if (StringUtils.isNotBlank(dto.getIncludeField())) {
            query.fields().include("_id", dto.getIncludeField());

        }
        //比对口令
        if (StringUtils.isNotBlank(dto.getWordOfCommand())) {
            if (!"真嘟假嘟".equals(dto.getWordOfCommand())) {
                return ApiResponse.buildError("口令错误");
            }
        }
        List<Map<String, Object>> list = new ArrayList<>();
        List<Map> mongoDbList = null;
        if ("主库".equalsIgnoreCase(dto.getDatabaseName())) {
            mongoDbList = mongoTemplate.find(query, Map.class, dto.getCollectionName());
        }
        if ("副库".equalsIgnoreCase(dto.getDatabaseName())) {
            mongoDbList = secondMongoTemplate.find(query, Map.class, dto.getCollectionName());
        }

        if (CollectionUtils.isEmpty(mongoDbList)) {
            return ApiResponse.buildOK("暂无数据");
        }
        for (Map map : mongoDbList) {
            HashMap<String, Object> mapResult = new HashMap<>(map.size());
            map.put("_id", String.valueOf(map.get("_id")));
            mapResult.putAll(map);
            list.add(mapResult);
        }
        return ApiResponse.buildOK().setData(list);
    }

    @PostMapping("/query")
    public ApiResponse queryPost(@RequestBody @Valid MongoDbQueryDTO dto) {
        Query query = new Query();
        if (StringUtils.isNotBlank(dto.getId())) {
            query.addCriteria(Criteria.where("_id").is(dto.getId()));
        }
        if (!CollectionUtils.isEmpty(dto.getQueryField())) {
            dto.getQueryField().forEach((k, v) -> query.addCriteria(Criteria.where(k).is(v)));
        }
        if (StringUtils.isNotBlank(dto.getIncludeField())) {
            query.fields().include("_id", dto.getIncludeField());
        }
        if (!CollectionUtils.isEmpty(dto.getIncludeFields())) {
            dto.getIncludeFields().forEach(query.fields()::include);
        }
        if (!CollectionUtils.isEmpty(dto.getExcludeFields())) {
            dto.getExcludeFields().forEach(query.fields()::exclude);
        }
        //比对口令
        if (StringUtils.isNotBlank(dto.getWordOfCommand())) {
            if (!"真嘟假嘟".equals(dto.getWordOfCommand())) {
                return ApiResponse.buildError("口令错误");
            }
        }
        List<Map<String, Object>> list = new ArrayList<>();
        List<Map> mongoDbList = null;
        if ("主库".equalsIgnoreCase(dto.getDatabaseName())) {
            mongoDbList = mongoTemplate.find(query, Map.class, dto.getCollectionName());
        }
        if ("副库".equalsIgnoreCase(dto.getDatabaseName())) {
            mongoDbList = secondMongoTemplate.find(query, Map.class, dto.getCollectionName());
        }

        if (CollectionUtils.isEmpty(mongoDbList)) {
            return ApiResponse.buildOK("暂无数据");
        }
        for (Map map : mongoDbList) {
            HashMap<String, Object> mapResult = new HashMap<>(map.size());
            map.put("_id", String.valueOf(map.get("_id")));
            mapResult.putAll(map);
            list.add(mapResult);
        }
        return ApiResponse.buildOK().setData(list);
    }

    @PostMapping("/update")
    public ApiResponse updateLoggingLevel(@RequestBody @Valid MongoDbUpdateDTO dto) {
        //比对口令
        if (StringUtils.isNotBlank(dto.getWordOfCommand())) {
            if (!"真嘟假嘟".equals(dto.getWordOfCommand())) {
                return ApiResponse.buildError("口令错误");
            }
        }
        Query query = new Query();
        query.addCriteria(Criteria.where("_id").is(dto.getId()));
        Update update = new Update();
        if (ObjectUtils.isNotEmpty(dto.getMapData())) {
            for (Map.Entry<String, Object> s : dto.getMapData().entrySet()) {
                update.set(s.getKey(), dto.getMapData().get(s.getKey()));
            }
            if ("主库".equalsIgnoreCase(dto.getDatabaseName())) {
                mongoTemplate.updateFirst(query, update, dto.getCollectionName());
            }
            if ("副库".equalsIgnoreCase(dto.getDatabaseName())) {
                secondMongoTemplate.updateFirst(query, update, dto.getCollectionName());
            }
        }
        Map map = null;
        if ("主库".equalsIgnoreCase(dto.getDatabaseName())) {
            map = mongoTemplate.findById(dto.getId(), Map.class, dto.getCollectionName());
        }
        if ("副库".equalsIgnoreCase(dto.getDatabaseName())) {
            map = secondMongoTemplate.findById(dto.getId(), Map.class, dto.getCollectionName());
        }
        return ApiResponse.buildOK().setData(map);
    }

    @PostMapping("/add")
    public ApiResponse add(@RequestBody @Valid MongoDbAddDTO dto) {
        //比对口令
        if (StringUtils.isNotBlank(dto.getWordOfCommand())) {
            if (!"真嘟假嘟".equals(dto.getWordOfCommand())) {
                return ApiResponse.buildError("口令错误");
            }
        }
        if (ObjectUtils.isEmpty(dto.getMapData())) {
            return ApiResponse.buildError("mapData为空");
        }
        if (StringUtils.isBlank(MapUtils.getString(dto.getMapData(), "_id"))) {
            return ApiResponse.buildError("mapData请传入_id");
        }

        String id = MapUtils.getString(dto.getMapData(), "_id");
        Query query = new Query();
        query.addCriteria(Criteria.where("_id").is(id));
        boolean isExist = false;
        if ("主库".equalsIgnoreCase(dto.getDatabaseName())) {
            isExist = mongoTemplate.exists(query, dto.getCollectionName());
        }
        if ("副库".equalsIgnoreCase(dto.getDatabaseName())) {
            isExist = secondMongoTemplate.exists(query, dto.getCollectionName());
        }
        if (isExist) {
            return ApiResponse.buildError("mapData中对应id已存在");
        }

        dto.getMapData().put("_id", new ObjectId(MapUtils.getString(dto.getMapData(), "_id")));
        DBObject dbObject = new BasicDBObject();
        dbObject.putAll(dto.getMapData());

        DBObject result = new BasicDBObject();
        if ("主库".equalsIgnoreCase(dto.getDatabaseName())) {
            result = mongoTemplate.save(dbObject, dto.getCollectionName());
        }
        if ("副库".equalsIgnoreCase(dto.getDatabaseName())) {
            result = secondMongoTemplate.save(dbObject, dto.getCollectionName());
        }
        if (Objects.nonNull(result)) {
            result.put("_id", String.valueOf(result.get("_id")));
        }
        return ApiResponse.buildOK().setData(result);
    }

    @PostMapping("/delete")
    public ApiResponse delete(@RequestBody @Valid MongoDbDeleteDTO dto) {
        //比对口令
        if (StringUtils.isNotBlank(dto.getWordOfCommand())) {
            if (!"真嘟假嘟".equals(dto.getWordOfCommand())) {
                return ApiResponse.buildError("口令错误");
            }
        }

        Map dataMap = new HashMap<>();
        if ("主库".equalsIgnoreCase(dto.getDatabaseName())) {
            dataMap = mongoTemplate.findById(new ObjectId(dto.getId()), Map.class, dto.getCollectionName());
        }
        if ("副库".equalsIgnoreCase(dto.getDatabaseName())) {
            dataMap = secondMongoTemplate.findById(new ObjectId(dto.getId()), Map.class, dto.getCollectionName());
        }
        if (CollectionUtils.isEmpty(dataMap)) {
            return ApiResponse.buildError("此id对应数据不存在");
        }

        Query query = new Query();
        query.addCriteria(Criteria.where("_id").is(dto.getId()));
        if ("主库".equalsIgnoreCase(dto.getDatabaseName())) {
            mongoTemplate.remove(query, dto.getCollectionName());
        }
        if ("副库".equalsIgnoreCase(dto.getDatabaseName())) {
            secondMongoTemplate.remove(query, dto.getCollectionName());
        }
        dataMap.put("_id", String.valueOf(dataMap.get("_id")));
        return ApiResponse.buildOK().setData(dataMap);
    }
}
