package com.digiwin.mobile.mobileuibot.mongodb.designerconfig;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.annotation.JSONField;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.core.common.EnvScope;
import com.digiwin.mobile.mobileuibot.mongodb.designerconfig.common.MobileDesignerConfigTypeEnum;
import com.digiwin.mobile.mobileuibot.openapi.designer.enums.DataVersionEnum;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotTmDataState;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.PcUiBotTmPage;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.LayoutComponentTypeEnum;
import com.digiwin.mobile.mobileuibot.proxy.uibot.model.layout.UiBotPageLayout;
import com.digiwin.mobile.mobileuibot.task.filter.TaskFilterMatchEnum;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import lombok.Data;
import lombok.experimental.Accessors;
import org.springframework.data.annotation.Id;
import org.springframework.data.mongodb.core.mapping.Document;
import org.springframework.util.CollectionUtils;

import java.io.Serializable;
import java.util.List;
import java.util.Optional;

/**
 * <p>功能描述：存储移动设计态 设计的移动配置数据</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: MobileDesignerConfig.java
 * @Author: wangjwc
 * @Date: created at 2023/8/10 12:42
 */
@Data
@Accessors(chain = true)
@Document(collection = "mobileDesignerConfig")
public class MobileDesignerConfig implements Serializable {
    private static final long serialVersionUID = -2755427549549760951L;

    @Id
    private String id;
    /**
     * 应用code,注意：为空表示通用配置
     */
    private String application;
    /**
     * 项目id
     */
    private String tmTaskId;
    /**
     * 任务id
     */
    private String tmActivityId;
    /**
     * @see TaskFilterMatchEnum
     * 如果没值，则默认精确匹配
     */
    private String tmActivityIdMatch;
    /**
     * 所属pattern
     */
    private String tmPattern;
    /**
     * 所属分类
     */
    private String tmCategory;
    /**
     * 所属类别
     * @see MobileDesignerConfigTypeEnum
     */
    private String type;
    /**
     * 使用的环境范围设置
     */
    private List<EnvScope> envScopes;
    /**
     * 设计态设计的全量数据
     */
    private JSONObject data;

    /**
     * 规则信息
     */
    private List<JSONObject> rules;
    private String createTime;
    private String updateTime;

    /**
     * 数据版本
     * @see DataVersionEnum
     */
    private String version;

    public List<JSONObject> getRules() {
        return rules = Optional.ofNullable(this.rules).orElse(Lists.newArrayList());
    }

    @JsonProperty(access = JsonProperty.Access.WRITE_ONLY)
    @JSONField(serialize = false)
    public PcUiBotTmPage getPages() {
        return Optional.ofNullable(data)
                .map(r -> r.getJSONObject("pageView"))
                .map(r -> r.getJSONObject("pages"))
                .map(r -> r.getJSONObject("task-detail-mobile"))
                .map(r -> JsonUtil.objectToJavaObject(r,
                        new TypeReference<PcUiBotTmPage>() {
                        })
                )
                .orElse(null);
    }

    @JsonProperty(access = JsonProperty.Access.WRITE_ONLY)
    @JSONField(serialize = false)
    public PcUiBotTmPage getPagesProject() {
        return Optional.ofNullable(data)
                .map(r -> r.getJSONObject("pageView"))
                .map(r -> r.getJSONObject("pages"))
                .map(r -> r.getJSONObject("project-detail-mobile"))
                .map(r -> JsonUtil.objectToJavaObject(r,
                        new TypeReference<PcUiBotTmPage>() {
                        })
                )
                .orElse(null);
    }

    public Boolean existMobileDsl(PcUiBotTmPage page) {
        if (page == null) {
            return false;
        }
        List<PcUiBotTmDataState> dataStates = page.getDataStates();
        if (CollectionUtils.isEmpty(dataStates)) {
            return false;
        }
        for (PcUiBotTmDataState dataState : dataStates) {
            List<UiBotPageLayout> pageLayout = dataState.getPageLayout();
            if (CollectionUtils.isEmpty(pageLayout)) {
                continue;
            }
            UiBotPageLayout mainPageLayout = new UiBotPageLayout();
            for (UiBotPageLayout uiBotPageLayout : pageLayout) {
                if (LayoutComponentTypeEnum.MAIN_PAGE.getValue().equalsIgnoreCase(uiBotPageLayout.getComponentType())) {
                    mainPageLayout = uiBotPageLayout;
                    // 因set-up-page可能会存在多个页面， 取第一个，为主页面
                    break;
                }
            }
            if (!CollectionUtils.isEmpty(mainPageLayout.getBody())) {
                // 存在页面，则认为存在移动DSL
                return true;
            }
        }
        return false;
    }

    /**
     * 根据当前启动的环境类型，判断当前页面过滤设置是否生效。
     *
     * @param appEnvType 系统当前启动环境类型。参考AppContext.envType
     * @return true-生效，false-不生效
     */
    public Boolean validWithAppEnvType(final String appEnvType) {
        // 如果没有配置页面可使用的环境，则表示不知道在哪个环境可使用，故直接返回false，表示不匹配
        if (this.getEnvScopes() == null || this.getEnvScopes().isEmpty()) {
            return false;
        }
        boolean result = false;
        for (EnvScope envScope : this.getEnvScopes()) {
            String type = envScope.getType();
            boolean envScopeValid = envScope.getValid();
            if (appEnvType.equals(type)) {
                result = envScopeValid;
                break;
            }
        }
        return result;
    }
}
