package com.digiwin.mobile.mobileuibot.mongodb.designerconfig;

import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.core.pagetaskrelation.PageTaskRelationService;
import com.digiwin.mobile.mobileuibot.openapi.designer.enums.DataVersionEnum;
import com.digiwin.mobile.mobileuibot.task.filter.TaskFilter;
import com.digiwin.mobile.mobileuibot.task.filter.TaskFilterService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * <p>功能描述：操作 MongoDB，获取 mobileDesignerConfig 数据</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: MobileDesignerConfigRepository.java
 * @Author: wangjwc
 * @Date: created at 2023/8/10 12:58
 */
@Component
public class MobileDesignerConfigRepository {

    @Autowired
    private MongoTemplate mongoTemplate;
    @Autowired
    @Qualifier("secondMongoTemplate")
    private MongoTemplate secondMongoTemplate;

    @Autowired
    private PageTaskRelationService pageTaskRelationService;

    @Autowired
    private TaskFilterService taskFilterService;

    /**
     * 通过任务id，判断当前环境下数据是否有效
     */
    public Boolean isExsitTmActivityId(String tmActivityId, String type, String version) {
        return this.findOne(tmActivityId, type, version) != null;
    }

    public Boolean isExsitTmActivityIdOfNotAppCode(String tmActivityId, String type, String version, Set<String> notAppCodeList) {
        return this.findOne(tmActivityId, type, version, notAppCodeList) != null;
    }

    private MobileDesignerConfig findOne(String tmActivityId, String type, String version) {
        return this.findOne(tmActivityId, type, version, null);
    }

    private MobileDesignerConfig findOne(String tmActivityId, String type, String version, Set<String> notAppCodeList) {
        // 查询指定字段
        Query queryAll = new Query();
        queryAll.addCriteria(Criteria.where("type").is(type))
                .addCriteria(Criteria.where("version").is(version))
                .fields().exclude("data").exclude("rules");
        List<MobileDesignerConfig> list = mongoTemplate.find(queryAll, MobileDesignerConfig.class)
                .stream()
                .filter(config -> {
                    // 过滤不在notAppCodeList中的应用
                    if (CollectionUtils.isEmpty(notAppCodeList)) {
                        return true;
                    }
                    return !notAppCodeList.contains(config.getApplication());
                })
                // 过滤当前环境下的数据
                .filter(config -> config.validWithAppEnvType(AppContext.getEnvType()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(list)) {
            return null;
        }
        for (MobileDesignerConfig designerConfig : list) {
            boolean match = TaskFilter.match(tmActivityId, designerConfig.getTmActivityId(), designerConfig.getTmActivityIdMatch());
            if (match) {
                tmActivityId = designerConfig.getTmActivityId();
                break;
            }
        }
        Query query = new Query();
        query.addCriteria(Criteria.where("type").is(type))
                .addCriteria(Criteria.where("tmActivityId").is(tmActivityId))
                .addCriteria(Criteria.where("version").is(version));
        MobileDesignerConfig findOne = this.mongoTemplate.findOne(query, MobileDesignerConfig.class);
        return Optional.ofNullable(findOne).orElse(new MobileDesignerConfig()).validWithAppEnvType(AppContext.getEnvType()) ? findOne : null;
    }

    public MobileDesignerConfig selectOneDsl(String tmActivityId, String type) {
        // wjw FIXME: 2025/2/13 因本地DSL配置是手动插入的，故数据都一样，取任意version版本数据即可
        return this.findOne(tmActivityId, type, DataVersionEnum.V1_0.getVersion());
    }

//    public void saveDsl(MobileDesignerConfig config) {
//        // 先新增规则时会出现type=null的情况,故type传递null
//        MobileDesignerConfig queryConfig = selectOneDsl(config.getTmActivityId(), null);
//
//        // 当 type=dsl-task 时，解析data节点，判断是否存在执行者与当责者配置
//        if (MobileDesignerConfigTypeEnum.TASK.getValue().equalsIgnoreCase(config.getType())) {
//            Boolean taskExist = config.existMobileDsl(config.getPages());
//            Boolean projectExist = config.existMobileDsl(config.getPagesProject());
//
//            if (BooleanUtils.isNotTrue(taskExist) && BooleanUtils.isNotTrue(projectExist)) {
//                // 无执行者与当责者配置
//                return;
//            }
//
//            if (BooleanUtils.isTrue(taskExist)) {
//                // 处理taskFilter和pageTaskRelation表
//                handleTaskFilterAndRelation(config);
//            }
//
//            if (queryConfig == null) {
//                if (BooleanUtils.isTrue(taskExist)) {
//                    MobileDesignerConfig task = JsonUtil.objectToJavaObject(config, MobileDesignerConfig.class);
//                    task.setType(MobileDesignerConfigTypeEnum.TASK.getValue());
//                    this.insertDsl(task);
//                }
//                if (BooleanUtils.isTrue(projectExist)) {
//                    MobileDesignerConfig project = JsonUtil.objectToJavaObject(config, MobileDesignerConfig.class);
//                    project.setType(MobileDesignerConfigTypeEnum.PROJECT.getValue());
//                    this.insertDsl(project);
//                }
//                return;
//            }
//            if (!StringUtils.hasLength(queryConfig.getType())) {
//                // type 字段为空，表示先新增规则，后续更新DSL
//                if (BooleanUtils.isTrue(taskExist) && BooleanUtils.isNotTrue(projectExist)) {
//                    // 只存在执行者
//                    config.setType(MobileDesignerConfigTypeEnum.TASK.getValue());
//                    this.updateDsl(config, queryConfig);
//                } else if (BooleanUtils.isNotTrue(taskExist) && BooleanUtils.isTrue(projectExist)) {
//                    // 只存在当责者
//                    config.setType(MobileDesignerConfigTypeEnum.PROJECT.getValue());
//                    this.updateDsl(config, queryConfig);
//                } else if (BooleanUtils.isTrue(taskExist) && BooleanUtils.isTrue(projectExist)) {
//                    MobileDesignerConfig project = JsonUtil.objectToJavaObject(config, MobileDesignerConfig.class);
//                    // 存在执行者与当责者
//                    config.setType(MobileDesignerConfigTypeEnum.TASK.getValue());
//                    this.updateDsl(config, queryConfig);
//
//                    project.setType(MobileDesignerConfigTypeEnum.PROJECT.getValue());
//                    project.setRules(queryConfig.getRules());
//                    this.insertDsl(project);
//                }
//            } else if (MobileDesignerConfigTypeEnum.TASK.getValue().equalsIgnoreCase(queryConfig.getType())) {
//                MobileDesignerConfig project = JsonUtil.objectToJavaObject(config, MobileDesignerConfig.class);
//                // 查出来是执行者配置
//                config.setType(MobileDesignerConfigTypeEnum.TASK.getValue());
//                this.updateDsl(config, queryConfig);
//                if (BooleanUtils.isTrue(projectExist)) {
//                    // 存在当责者
//                    project.setType(MobileDesignerConfigTypeEnum.PROJECT.getValue());
//                    MobileDesignerConfig projectConfig = this.findOne(config.getTmActivityId(), MobileDesignerConfigTypeEnum.PROJECT.getValue());
//                    if (projectConfig == null) {
//                        // 新增当责者配置
//                        project.setRules(queryConfig.getRules());
//                        this.insertDsl(project);
//                    } else {
//                        // 更新当责者配置
//                        this.updateDsl(project, projectConfig);
//                    }
//                }
//            } else if (MobileDesignerConfigTypeEnum.PROJECT.getValue().equalsIgnoreCase(queryConfig.getType())) {
//                MobileDesignerConfig task = JsonUtil.objectToJavaObject(config, MobileDesignerConfig.class);
//                // 查出来是当责者配置
//                config.setType(MobileDesignerConfigTypeEnum.PROJECT.getValue());
//                this.updateDsl(config, queryConfig);
//                if (BooleanUtils.isTrue(taskExist)) {
//                    // 存在执行者
//                    task.setType(MobileDesignerConfigTypeEnum.TASK.getValue());
//                    MobileDesignerConfig taskConfig = this.findOne(config.getTmActivityId(), MobileDesignerConfigTypeEnum.TASK.getValue());
//                    if (taskConfig == null) {
//                        // 新增执行者配置
//                        task.setRules(queryConfig.getRules());
//                        this.insertDsl(task);
//                    } else {
//                        // 更新执行者配置
//                        this.updateDsl(task, taskConfig);
//                    }
//                }
//            }
//        } else {
//            if (queryConfig == null) {
//                this.insertDsl(config);
//                return;
//            }
//            this.updateDsl(config, queryConfig);
//        }
//    }

//    private void insertDsl(MobileDesignerConfig config) {
//        List<EnvScope> envScopeList = this.getDefaultEnvScopes();
//        config.setEnvScopes(envScopeList);
//        String time = DateTimeUtil.getTodayTimeUseDefaultPattern();
//        config.setCreateTime(time);
//        config.setUpdateTime(time);
//        this.mongoTemplate.insert(config);
//    }

//    private void updateDsl(MobileDesignerConfig config, MobileDesignerConfig queryConfig) {
//        ObjectId id = new ObjectId(queryConfig.getId());
//        Query query = new Query();
//        query.addCriteria(Criteria.where("_id").is(id));
//        Update update = new Update();
//        if (!StringUtils.hasLength(queryConfig.getApplication())) {
//            update.set("application", config.getApplication());
//        }
//        if (!StringUtils.hasLength(queryConfig.getTmPattern())) {
//            update.set("tmPattern", config.getTmPattern());
//        }
//        if (!StringUtils.hasLength(queryConfig.getTmCategory())) {
//            update.set("tmCategory", config.getTmCategory());
//        }
//
//        List<EnvScope> envScopesList = queryConfig.getEnvScopes();
//        if (CollectionUtils.isEmpty(envScopesList)) {
//            envScopesList = this.getDefaultEnvScopes();
//        }
//        update.set("envScopes", envScopesList);
//        update.set("type", config.getType());
//        update.set("data", config.getData());
//        if (UiBotModelBuildApcStandProductionProgressPageStrategy.getTmActivityId().equals(queryConfig.getTmActivityId())
//                && CollectionUtils.isEmpty(config.getData())) {
//            // wjw TODO: 2024/8/13 临时处理，APC-机位生产进度,因走定制处理，前端传递的data为null时不更新data节点
//            update.set("data", queryConfig.getData());
//        }
//        String time = DateTimeUtil.getTodayTimeUseDefaultPattern();
//        if (!StringUtils.hasLength(queryConfig.getCreateTime())) {
//            update.set("createTime", time);
//        }
//        update.set("updateTime", time);
//        this.mongoTemplate.updateMulti(
//                query,
//                update,
//                MobileDesignerConfig.class);
//    }

//    private void handleTaskFilterAndRelation(MobileDesignerConfig config) {
//        TaskFilter taskFilter = taskFilterService.selectByDesigner(config.getTmActivityId());
//        if (null == taskFilter) {
//            // 新增 taskFilter 表
//            taskFilterService.insertByDesigner(config.getTmActivityId());
//        }
//
//        PageTaskRelation pageTaskRelation = pageTaskRelationService.selectByDesigner(config.getTmActivityId());
//        if (null == pageTaskRelation) {
//            // 新增 pageTaskRelation 表
//            pageTaskRelationService.insertByDesigner(config.getTmActivityId());
//        }
//    }

//    private List<EnvScope> getDefaultEnvScopes() {
//        List<EnvScope> list = new ArrayList<>();
//        EnvScope prod = new EnvScope();
//        prod.setType(AppEnvTypeEnum.PRODUCTION.getEnvType());
//        prod.setValid(false);
//        list.add(prod);
//        EnvScope test = new EnvScope();
//        test.setType(AppEnvTypeEnum.TEST.getEnvType());
//        test.setValid(true);
//        list.add(test);
//        EnvScope dev = new EnvScope();
//        dev.setType(AppEnvTypeEnum.DEVELOP.getEnvType());
//        dev.setValid(true);
//        list.add(dev);
//        return list;
//    }

//    public List<MobileDesignerConfig> deleteDsl(String tmActivityId) {
//        Query query = new Query();
//        query.addCriteria(Criteria.where("tmActivityId").is(tmActivityId));
//        // 会删除执行者与当责者记录
//        List<MobileDesignerConfig> allAndRemove = mongoTemplate.findAllAndRemove(query, MobileDesignerConfig.class);
//        // 删除 taskFilter 表
//        taskFilterService.deleteByDesigner(tmActivityId);
//        // 删除 pageTaskRelation 表
//        pageTaskRelationService.deleteByDesigner(tmActivityId);
//        return allAndRemove;
//    }

//    public MobileDesignerConfig selectOneRule(String tmActivityId, String ruleId) {
//        // 因执行者与当责者规则只根据path区分，本地存储一致，故返回任意一个,传递null即可
//        MobileDesignerConfig designerConfig = this.findOne(tmActivityId, null);
//        if (designerConfig == null) {
//            return designerConfig;
//        }
//        List<JSONObject> rules = designerConfig.getRules().stream()
//                .filter(rule -> ruleId.equals(rule.getString("key"))).collect(Collectors.toList());
//        return designerConfig.setRules(rules);
//    }

    public MobileDesignerConfig selectAllRule(String tmActivityId, String type) {
        // wjw FIXME: 2025/2/13 因本地rule配置是手动插入的，故数据都一样，取任意version版本数据即可
        MobileDesignerConfig designerConfig = this.findOne(tmActivityId, type, DataVersionEnum.V1_0.getVersion());
        if (designerConfig == null) {
            return designerConfig;
        }
        return designerConfig;
    }

//    public void saveRule(MobileDesignerConfig config) {
//        config.getRules().forEach(rule -> {
//            if (!rule.containsKey("key")) {
//                // 新增的规则，增加 主键
//                String key = config.getTmActivityId() + "_" + System.currentTimeMillis();
//                rule.put("id", key);
//                rule.put("key", key);
//                rule.put("code", key);
//            }
//        });
//        MobileDesignerConfig designerConfig = this.findOne(config.getTmActivityId(), null);
//        if (designerConfig == null) {
//            // 新增 --- 无具体的type，需要在保存DSL时再设置
//            this.mongoTemplate.insert(config);
//            return;
//        }
//
//        String type = designerConfig.getType();
//        if (MobileDesignerConfigTypeEnum.TASK.getValue().equalsIgnoreCase(type)) {
//            // 因执行者与当责者规则只根据path区分，故本地存储要一致
//            this.saveRule(designerConfig, config.getRules());
//            MobileDesignerConfig project = this.findOne(config.getTmActivityId(), MobileDesignerConfigTypeEnum.PROJECT.getValue());
//            if (project != null) {
//                this.saveRule(project, config.getRules());
//            }
//        } else if (MobileDesignerConfigTypeEnum.PROJECT.getValue().equalsIgnoreCase(type)) {
//            // 因执行者与当责者规则只根据path区分，故本地存储要一致
//            this.saveRule(designerConfig, config.getRules());
//            MobileDesignerConfig task = this.findOne(config.getTmActivityId(), MobileDesignerConfigTypeEnum.TASK.getValue());
//            if (task != null) {
//                this.saveRule(task, config.getRules());
//            }
//        } else {
//            this.saveRule(designerConfig, config.getRules());
//        }
//    }

//    private void saveRule(MobileDesignerConfig designerConfig, List<JSONObject> saveRules) {
//        // 规则存在就替换，不存在就新增
//        Map<String, JSONObject> designerConfigRuleMapByKey = designerConfig.getRules().stream()
//                .collect(Collectors.toMap(e -> e.getString("key"), c -> c, (c1, c2) -> c1, LinkedHashMap::new));
//        Map<String, JSONObject> paramConfigMapBykey = saveRules.stream().collect(Collectors
//                .toMap(e -> e.getString("key"), c -> c, (c1, c2) -> c1));
//        designerConfigRuleMapByKey.putAll(paramConfigMapBykey);
//        List<JSONObject> designerConfigRule = designerConfigRuleMapByKey.values().stream().collect(Collectors.toList());
//
//        // 重新保存
//        this.mongoTemplate.updateMulti(
//                new Query(Criteria.where("_id").is(new ObjectId(designerConfig.getId()))),
//                Update.update("rules", designerConfigRule),
//                MobileDesignerConfig.class);
//    }

//    public void deleteRule(String tmActivityId, String ruleId) {
//        MobileDesignerConfig designerConfig = this.findOne(tmActivityId, null);
//        if (designerConfig == null) {
//            return;
//        }
//        String type = designerConfig.getType();
//        if (MobileDesignerConfigTypeEnum.TASK.getValue().equalsIgnoreCase(type)) {
//            // 因执行者与当责者规则只根据path区分，故本地存储要一致
//            this.deleteRule(designerConfig, ruleId);
//            MobileDesignerConfig project = this.findOne(tmActivityId, MobileDesignerConfigTypeEnum.PROJECT.getValue());
//            if (project != null) {
//                this.deleteRule(project, ruleId);
//            }
//        } else if (MobileDesignerConfigTypeEnum.PROJECT.getValue().equalsIgnoreCase(type)) {
//            // 因执行者与当责者规则只根据path区分，故本地存储要一致
//            this.deleteRule(designerConfig, ruleId);
//            MobileDesignerConfig task = this.findOne(tmActivityId, MobileDesignerConfigTypeEnum.TASK.getValue());
//            if (task != null) {
//                this.deleteRule(task, ruleId);
//            }
//        } else {
//            this.deleteRule(designerConfig, ruleId);
//        }
//
//    }

//    private void deleteRule(MobileDesignerConfig designerConfig, String ruleId) {
//        // 其它规则
//        List<JSONObject> otherRules = designerConfig.getRules().stream()
//                .filter(rule -> !ruleId.equals(rule.getString("key"))).collect(Collectors.toList());
//        // 重新保存
//        this.mongoTemplate.updateMulti(
//                new Query(Criteria.where("_id").is(new ObjectId(designerConfig.getId()))),
//                Update.update("rules", otherRules),
//                MobileDesignerConfig.class);
//    }

    public List<MobileDesignerConfig> findByType(String type, String version, Set<String> notAppCodeList) {
        Query query = new Query();
        query.addCriteria(Criteria.where("type").is(type))
                .addCriteria(Criteria.where("version").is(version));
        if (!CollectionUtils.isEmpty(notAppCodeList)) {
            query.addCriteria(Criteria.where("application").nin(notAppCodeList));
        }
        return this.mongoTemplate.find(query, MobileDesignerConfig.class)
                .stream().filter(config -> config.validWithAppEnvType(AppContext.getEnvType()))
                .collect(Collectors.toList());
    }

    public void deleteByCode(String appCode, String version) {
        Query query = new Query();
        query.addCriteria(Criteria.where("application").is(appCode))
                .addCriteria(Criteria.where("version").is(version));
        // AB库同时操作
        mongoTemplate.findAllAndRemove(query, MobileDesignerConfig.class);
        secondMongoTemplate.findAllAndRemove(query, MobileDesignerConfig.class);
    }

    public List<MobileDesignerConfig> selectByCode(String appCode, String version) {
        Query query = new Query();
        query.addCriteria(Criteria.where("application").is(appCode))
                .addCriteria(Criteria.where("version").is(version));
        List<MobileDesignerConfig> list = this.mongoTemplate.find(query, MobileDesignerConfig.class);
        return CollectionUtils.isEmpty(list) ? new ArrayList<>() : list;
    }

    public void batchInsert(List<MobileDesignerConfig> mobileDesignerConfigs) {
        if (CollectionUtils.isEmpty(mobileDesignerConfigs)) {
            return;
        }
        // AB库同时操作
        mongoTemplate.insertAll(mobileDesignerConfigs);
        secondMongoTemplate.insertAll(mobileDesignerConfigs);
    }

    public void updataData(MobileDesignerConfig config) {
        if (config == null || !StringUtils.hasLength(config.getTmActivityId())
                || config.getData() == null || !StringUtils.hasLength(config.getApplication())) {
            return;
        }
        // 查询条件type不能加，因为任务当责者和执行者此接口保存传递同一个类型
        Query query = new Query()
                .addCriteria(Criteria.where("tmActivityId").is(config.getTmActivityId()))
                // 标准应用
                .addCriteria(Criteria.where("application").is(config.getApplication()));
        // 查询标准应用下作业是否存在数据
        if (this.mongoTemplate.findOne(query, MobileDesignerConfig.class) == null) {
            // 不存在，返回
            return;
        }
        Update update = new Update().set("data", config.getData());
        // AB库同时操作
        this.mongoTemplate.updateMulti(query, update, MobileDesignerConfig.class);
        this.secondMongoTemplate.updateMulti(query, update, MobileDesignerConfig.class);

    }
}